<?php
// my_telegram_daemon.php

require __DIR__ . '/vendor/autoload.php';

use App\Services\SendTelegram; // Ajusta el namespace si es diferente
use App\Templates\TelegramTemplate; // Ajusta el namespace si es diferente

$telegramSender = new SendTelegram();
$telegramTemplate = new TelegramTemplate();

$botInstances = ["bot1", "bot2", "bot3", "bot4", "bot5"];
$systemdServiceName = "trading-bot";

// --- NUEVO: Configuración para la advertencia de caducidad del SecretID ---
$secretIdTTLSeconds = 31536000; // 1 año en segundos (aproximadamente 365 días * 24 horas * 3600 segundos)
$warningThresholdSeconds = 10 * 24 * 3600; // 10 días en segundos
$secretIdExpiryWarningSent = false; // Flag para enviar la advertencia solo una vez por ciclo

// Obtener la fecha de inyección/generación del SecretID de la variable de entorno
$secretIdInjectionDate = getenv('SECRET_ID_INJECTION_DATE');
if ($secretIdInjectionDate === false) {
    // Si la variable no está seteada, asumimos que no se puede verificar la caducidad
    echo "[" . date('Y-m-d H:i:s') . "] Demonio: SECRET_ID_INJECTION_DATE no encontrada. No se verificará la caducidad del SecretID.\n";
    $secretIdInjectionDate = 0; // Deshabilita la comprobación si no hay fecha
} else {
    $secretIdInjectionDate = (int)$secretIdInjectionDate;
    echo "[" . date('Y-m-d H:i:s') . "] Demonio: Fecha de inyección del SecretID: " . date('Y-m-d H:i:s', $secretIdInjectionDate) . "\n";
}

// --- FASE INICIAL: Arrancar todos los bots y suprimir notificaciones ---
echo "[" . date('Y-m-d H:i:s') . "] Demonio: Iniciando fase de arranque inicial de todos los bots. No se enviarán notificaciones de alerta en esta fase.\n";

foreach ($botInstances as $instance) {
    $serviceFullName = "{$systemdServiceName}@{$instance}.service";
    echo "-> Intentando iniciar el bot: {$instance}...\n";
    shell_exec("/usr/bin/sudo /usr/bin/systemctl start " . escapeshellarg($serviceFullName) . " > /dev/null 2>&1");
    sleep(2); 
    $initialStatus = trim(shell_exec("/usr/bin/systemctl is-active " . escapeshellarg($serviceFullName) . " 2>/dev/null"));
    if ($initialStatus === "active") {
        echo "   Bot {$instance} iniciado exitosamente.\n";
    } else {
        echo "   ADVERTENCIA: Bot {$instance} no se pudo iniciar inmediatamente. Estado: {$initialStatus}. Se monitoreará en el bucle principal.\n";
    }
}
echo "[" . date('Y-m-d H:i:s') . "] Demonio: Fase de arranque inicial de bots completada. Iniciando monitoreo regular.\n";

// --- Bucle Principal de Monitoreo (Operación Normal) ---
while (true) {
    echo "[" . date('Y-m-d H:i:s') . "] Demonio: Verificando el estado de los bots...\n";

    // --- NUEVO: Comprobación y Notificación de Caducidad del SecretID ---
    if ($secretIdInjectionDate > 0 && !$secretIdExpiryWarningSent) { // Solo si tenemos fecha y no hemos enviado la advertencia
        $secondsSinceInjection = time() - $secretIdInjectionDate;
        $secondsUntilExpiry = $secretIdTTLSeconds - $secondsSinceInjection;

        if ($secondsUntilExpiry <= $warningThresholdSeconds && $secondsUntilExpiry > 0) {
            $daysUntilExpiry = round($secondsUntilExpiry / (24 * 3600));
            $warningMessage = "¡URGENTE! El SecretID de Vault para AppRole 'trading-bot' caducará en aproximadamente *{$daysUntilExpiry} días*.\nPor favor, contacta a los administradores para generar y reemplazar el SecretID y reiniciar los servicios pertinentes.";
            $telegramSender->sendAlertTelegram($warningMessage);
            $secretIdExpiryWarningSent = true; // Para enviar la advertencia solo una vez por cada vez que el demonio arranca
            echo "[" . date('Y-m-d H:i:s') . "] Demonio: Advertencia de caducidad de SecretID enviada.\n";
        } elseif ($secondsUntilExpiry <= 0) {
             $expiredMessage = "¡CRÍTICO! El SecretID de Vault para AppRole 'trading-bot' *HA CADUCADO*.\nLos bots podrían fallar la autenticación con Vault. Se requiere intervención INMEDIATA para generar y reemplazar el SecretID y reiniciar los servicios.";
             $telegramSender->sendAlertTelegram($expiredMessage);
             $secretIdExpiryWarningSent = true; // Para evitar spam
             echo "[" . date('Y-m-d H:i:s') . "] Demonio: Alerta de SecretID caducado enviada.\n";
        }
    }
    // --- FIN NUEVA COMPROBACIÓN ---


    foreach ($botInstances as $instance) {
        $serviceFullName = "{$systemdServiceName}@{$instance}.service";
        
        $statusCommand = "/usr/bin/systemctl is-active " . escapeshellarg($serviceFullName);
        $status = trim(shell_exec($statusCommand . " 2>/dev/null"));

        if ($status !== "active") {
            echo "ALERTA: Instancia {$instance} no está ACTIVA. Estado: {$status}.\n";

            $logCommand = "/usr/bin/journalctl -u " . escapeshellarg($serviceFullName) . " -n 20 --no-pager 2>/dev/null";
            $logs = shell_exec($logCommand);
            
            $alertMessage = "ALERTA: Instancia del bot '{$instance}' no está ACTIVA.\nEstado: *{$status}*.\n\nÚltimos logs:\n```\n{$logs}\n```";
            $telegramSender->sendAlertTelegram($alertMessage);

            echo "Intentando reiniciar {$instance}...\n";
            $restartCommand = "/usr/bin/sudo /usr/bin/systemctl restart " . escapeshellarg($serviceFullName);
            shell_exec($restartCommand . " > /dev/null 2>&1");
            
            sleep(5); 
            
            $newStatus = trim(shell_exec($statusCommand . " 2>/dev/null"));

            if ($newStatus === "active") {
                $infoMessage = "INFO: Instancia del bot '{$instance}' se ha *REINICIADO* exitosamente.";
                $telegramSender->sendAlertTelegram($infoMessage);
                echo "INFO: {$instance} reiniciado y activo.\n";
            } else {
                $errorMessage = "ALERTA CRÍTICA: Instancia del bot '{$instance}' *FALLÓ AL REINICIAR*.\nEstado actual: *{$newStatus}*.\nSe requiere intervención manual URGENTE.";
                $telegramSender->sendAlertTelegram($errorMessage);
                echo "ERROR: {$instance} no se pudo reiniciar.\n";
            }
        }
    }

    sleep(60); 
}