#!/bin/bash

# Este script configura HashiCorp Vault en un VPS Ubuntu.
# Está diseñado para ser ejecutado como root o con un usuario con privilegios sudo.

# --- ¡NOTA IMPORTANTE PARA EL EQUIPO DE INSTALACIÓN! ---
# El binario de Vault, por defecto en las instalaciones de 'apt', se ubica en '/usr/bin/vault'.
# Sin embargo, el archivo de servicio 'systemd' que viene con el paquete a veces puede
# referenciar incorrectamente a '/usr/local/bin/vault'.
# ESTE SCRIPT INCLUYE UN PASO ESPECÍFICO (Punto 5.1) PARA CORREGIR AUTOMÁTICAMENTE ESTA RUTA.
# Es CRÍTICO que esta corrección se aplique correctamente para que el servicio de Vault pueda iniciar.
# Por favor, presta especial atención a ese paso.
# --------------------------------------------------------

# --- 0. VARIABLES DE CONFIGURACIÓN ---
# ¡IMPORTANTE! Reemplaza estos valores con los de tu VPS de producción.
# Si estos valores no se configuran correctamente, la generación de certificados y
# la configuración de Vault FAllARÁN.
VAULT_PUBLIC_IP="194.164.161.14" # Ejemplo: 203.0.113.45
VPS_HOSTNAME="alinestra"     # Ejemplo: vaultserver-prod

# Rutas estándar de Vault
VAULT_CONFIG_DIR="/etc/vault.d/config"
VAULT_CERTS_DIR="/etc/vault.d/certs"
VAULT_POLICIES_DIR="/etc/vault.d/policies"
VAULT_DATA_DIR="/opt/vault/data"
VAULT_LOG_FILE="/var/log/vault.log" # Opcional: para redirigir logs de Systemd si lo deseas

echo "--- INICIANDO CONFIGURACIÓN DE VAULT ---"
echo "Variables configuradas:"
echo "  IP Pública del VPS: ${VAULT_PUBLIC_IP}"
echo "  Hostname del VPS: ${VPS_HOSTNAME}"
echo "-----------------------------------"

# Asegúrate de que curl esté instalado para algunas operaciones (ej. ifconfig.co)
sudo apt update && sudo apt install -y curl wget gnupg nano

---

### 1. Instalación de HashiCorp Vault

echo "--- 1. INSTALACIÓN DE HASHICORP VAULT ---"


### 2. Configuración de Directorios y Permisos

echo "--- 2. CONFIGURACIÓN DE DIRECTORIOS Y PERMISOS ---"

# 2.1 Crear directorios de Vault
echo "Creando directorios para Vault..."
sudo mkdir -p "$VAULT_CONFIG_DIR" || { echo "Error al crear ${VAULT_CONFIG_DIR}. Abortando."; exit 1; }
sudo mkdir -p "$VAULT_CERTS_DIR" || { echo "Error al crear ${VAULT_CERTS_DIR}. Abortando."; exit 1; }
sudo mkdir -p "$VAULT_POLICIES_DIR" || { echo "Error al crear ${VAULT_POLICIES_DIR}. Abortando."; exit 1; }
sudo mkdir -p "$VAULT_DATA_DIR" || { echo "Error al crear ${VAULT_DATA_DIR}. Abortando."; exit 1; }

# 2.2 Crear usuario 'vault' (si no existe, el paquete de instalación suele crearlo)
echo "Asegurando que el usuario 'vault' existe..."
sudo useradd --system --home /etc/vault.d --shell /bin/false vault || true # || true para evitar que el script se detenga si el usuario ya existe

# 2.3 Asignar permisos a los directorios
echo "Asignando permisos a los directorios de Vault..."
sudo chown -R vault:vault "$VAULT_CONFIG_DIR" || { echo "Error chown ${VAULT_CONFIG_DIR}. Abortando."; exit 1; }
sudo chown -R vault:vault "$VAULT_CERTS_DIR" || { echo "Error chown ${VAULT_CERTS_DIR}. Abortando."; exit 1; }
sudo chown -R vault:vault "$VAULT_POLICIES_DIR" || { echo "Error chown ${VAULT_POLICIES_DIR}. Abortando."; exit 1; }
sudo chown -R vault:vault "$VAULT_DATA_DIR" || { echo "Error chown ${VAULT_DATA_DIR}. Abortando."; exit 1; }

sudo chmod -R 750 "$VAULT_DATA_DIR" || { echo "Error chmod ${VAULT_DATA_DIR}. Abortando."; exit 1; }
sudo chmod -R 750 "$VAULT_CONFIG_DIR" || { echo "Error chmod ${VAULT_CONFIG_DIR}. Abortando."; exit 1; }
sudo chmod -R 750 "$VAULT_POLICIES_DIR" || { echo "Error chmod ${VAULT_POLICIES_DIR}. Abortando."; exit 1; }
sudo chmod -R 700 "$VAULT_CERTS_DIR" || { echo "Error chmod ${VAULT_CERTS_DIR}. Abortando."; exit 1; } # Los certificados son sensibles
echo "-----------------------------------"

---

### 3. Generación de Certificados TLS (Autofirmados con SANs)

echo "--- 3. GENERACIÓN DE CERTIFICADOS TLS ---"

# 3.1 Detener el servicio de Vault temporalmente (si estuviera corriendo por alguna razón)
echo "Deteniendo el servicio de Vault para generar certificados..."
sudo systemctl stop vault

# 3.2 Crear el archivo de configuración de OpenSSL para SANs
echo "Creando vault-csr.conf para la generación de certificados en ${VAULT_CERTS_DIR}..."
sudo bash -c "cat <<EOF > ${VAULT_CERTS_DIR}/vault-csr.conf
[ req ]
default_bits        = 2048
default_md          = sha256
default_keyfile     = vault.key
distinguished_name  = req_distinguished_name
req_extensions      = v3_req
x509_extensions     = v3_req # The extentions to add to the self signed cert

[ req_distinguished_name ]
countryName                 = Country Name (2 letter code)
countryName_default         = ES
stateOrProvinceName         = State or Province Name (full name)
stateOrProvinceName_default = Madrid
localityName                = Locality Name (eg, city)
localityName_default        = Madrid
organizationName            = Organization Name (eg, company)
organizationName_default    = YourCompany
organizationalUnitName      = Organizational Unit Name (eg, section)
organizationalUnitName_default = IT
commonName                  = Common Name (eg, YOUR name)
commonName_default          = ${VPS_HOSTNAME} # Usamos el hostname del VPS
emailAddress                = Email Address
emailAddress_default        = admin@yourcompany.com

[ v3_req ]
keyUsage = critical, digitalSignature, keyEncipherment
extendedKeyUsage = serverAuth, clientAuth
subjectAltName = @alt_names

[ alt_names ]
DNS.1 = localhost
DNS.2 = ${VPS_HOSTNAME} # Tu nombre de host
IP.1 = 127.0.0.1
IP.2 = ${VAULT_PUBLIC_IP} # Tu IP pública
EOF" || { echo "Error al crear vault-csr.conf. Abortando."; exit 1; }

# 3.3 Generar la clave privada de Vault
echo "Generando clave privada de Vault (vault.key)..."
sudo openssl genrsa -out "${VAULT_CERTS_DIR}/vault.key" 2048 || { echo "Error al generar vault.key. Abortando."; exit 1; }

# 3.4 Generar la Solicitud de Firma de Certificado (CSR)
echo "Generando CSR (vault.csr)..."
# Presiona Enter para todos los campos, los valores se toman de vault-csr.conf
sudo openssl req -new -key "${VAULT_CERTS_DIR}/vault.key" -out "${VAULT_CERTS_DIR}/vault.csr" -config "${VAULT_CERTS_DIR}/vault-csr.conf" -nodes || { echo "Error al generar vault.csr. Abortando."; exit 1; }

# 3.5 Auto-firmar el certificado
echo "Auto-firmando el certificado (vault.crt)..."
sudo openssl x509 -req -days 365 -in "${VAULT_CERTS_DIR}/vault.csr" -signkey "${VAULT_CERTS_DIR}/vault.key" -out "${VAULT_CERTS_DIR}/vault.crt" -extfile "${VAULT_CERTS_DIR}/vault-csr.conf" -extensions v3_req || { echo "Error al auto-firmar vault.crt. Abortando."; exit 1; }

# 3.6 Ajustar permisos de los certificados
echo "Ajustando permisos de los certificados..."
sudo chown vault:vault "${VAULT_CERTS_DIR}/vault.key" "${VAULT_CERTS_DIR}/vault.crt" || { echo "Error chown en certificados. Abortando."; exit 1; }
sudo chmod 640 "${VAULT_CERTS_DIR}/vault.key" || { echo "Error chmod en vault.key. Abortando."; exit 1; }
sudo chmod 644 "${VAULT_CERTS_DIR}/vault.crt" || { echo "Error chmod en vault.crt. Abortando."; exit 1; }

echo "Certificados TLS generados y con permisos ajustados."
echo "-----------------------------------"

---

### 4. Configuración del Archivo `vault.hcl`

echo "--- 4. CONFIGURACIÓN DEL ARCHIVO vault.hcl ---"

# 4.1 Crear el archivo de configuración principal de Vault
echo "Creando el archivo de configuración de Vault: ${VAULT_CONFIG_DIR}/vault.hcl..."
sudo bash -c "cat <<EOF > ${VAULT_CONFIG_DIR}/vault.hcl
storage \"raft\" {
  path = \"${VAULT_DATA_DIR}\"
  node_id = \"node1\" # Cambiar si tienes más nodos Raft en el clúster (requiere setup avanzado)
  performance_multiplier = 1
}

listener \"tcp\" {
  address       = \"127.0.0.1:8200\" # Vault solo escucha conexiones desde esta misma máquina
  tls_disable   = \"false\"
  tls_cert_file = \"${VAULT_CERTS_DIR}/vault.crt\"
  tls_key_file  = \"${VAULT_CERTS_DIR}/vault.key\"
}

api_addr = \"http://127.0.0.1:8200\" # El bot se comunicará con Vault en localhost
cluster_addr = \"http://127.0.0.1:8201\" # Raft también usará localhost para comunicación interna

disable_mlock = false # Importante: Requiere configuración en systemd service (Punto 5)
cluster_name = \"vault-cluster-prod-vps\" # Nombre para tu clúster de producción
log_level = \"info\" # Puedes cambiar a \"debug\" si necesitas más detalle en los logs
EOF" || { echo "Error al crear vault.hcl. Abortando."; exit 1; }

echo "Archivo vault.hcl creado."
echo "-----------------------------------"

### 5. Ajuste del Servicio Systemd de Vault


# 5 Recargar Systemd y reiniciar el servicio de Vault
echo "Recargando Systemd y reiniciando el servicio de Vault..."
sudo systemctl daemon-reload || { echo "Error al recargar Systemd. Abortando."; exit 1; }
sudo systemctl start vault || { echo "Error al iniciar el servicio de Vault. Abortando."; exit 1; }

echo "Verificando el estado inicial del servicio Vault (debería estar 'active (running)', 'Sealed: true' y 'Initialized: false' en 'vault status'):"
sudo systemctl status vault --no-pager
echo "-----------------------------------"

---

### 6. Configuración de Variables de Entorno para la CLI de Vault

echo "--- 6. CONFIGURACIÓN DE VARIABLES DE ENTORNO PARA LA CLI ---"

echo "Configurando VAULT_ADDR y VAULT_CACERT en /etc/environment..."
# ¡Importante! No añadir comentarios ni espacios extra dentro de las líneas de variables en /etc/environment
sudo bash -c "cat <<EOF > /etc/environment
PATH=\"/usr/local/sbin:/usr/local/bin:/usr/sbin:/usr/bin:/sbin:/bin:/usr/games:/usr/local/games:/snap/bin\"
VAULT_ADDR=\"https://127.0.0.1:8200\"
VAULT_CACERT=\"${VAULT_CERTS_DIR}/vault.crt\"
EOF" || { echo "Error al configurar /etc/environment. Abortando."; exit 1; }

echo "Variables de entorno configuradas. Para que surtan efecto en esta sesión SSH,"
echo "debes CERRAR y VOLVER A ABRIR la sesión SSH, o REINICIAR el VPS."
echo "Después, podrás ejecutar 'vault status' sin errores TLS."
echo "-----------------------------------"

echo ""
echo "############################################################"
echo "### ¡¡¡PAUSA MUY IMPORTANTE AQUÍ!!!                       ###"
echo "###                                                       ###"
echo "### POR FAVOR, REINICIA TU VPS AHORA:                     ###"
echo "###   sudo reboot                                         ###"
echo "###                                                       ###"
echo "### O CIERRA Y VUELVE A ABRIR TU SESIÓN SSH:              ###"
echo "###   exit (y luego vuelve a conectarte)                  ###"
echo "###                                                       ###"
echo "### ESTO ES CRÍTICO para que las variables de entorno se  ###"
echo "### carguen y la CLI de Vault funcione correctamente.      ###"
echo "###                                                       ###"
echo "### Después de reiniciar/re-loguear, ejecuta este script  ###"
echo "### de nuevo, pero desde el punto 7 en adelante.           ###"
echo "############################################################"
exit 0 # Salir aquí para forzar el reinicio/re-login

# --- LOS SIGUIENTES PASOS DEBEN EJECUTARSE DESPUÉS DE REINICIAR/RE-LOGUEAR ---

---

### 7. Inicialización y Desellado de Vault (¡PASOS MANUALES CRÍTICOS!)

echo "--- 7. INICIALIZACIÓN Y DESELLADO DE VAULT ---"

echo "Verificando el estado de Vault (debería ser 'Initialized: false' y 'Sealed: true' después de un reinicio):"
vault status || { echo "Error al verificar el estado de Vault. Asegúrate de haber reiniciado o re-logueado. Abortando."; exit 1; }

echo ""
echo "--- ¡PASO CRÍTICO: INICIALIZACIÓN DE VAULT! ---"
echo "Ejecuta 'vault operator init' y GUARDA TODA la salida en un lugar MUY SEGURO fuera de este VPS."
echo "Necesitarás las 'Unseal Keys' y el 'Initial Root Token'."
echo "Pulsa Enter para continuar con la inicialización o Ctrl+C para salir."
read -p "Presiona Enter para ejecutar vault operator init..."

# 7.1 Inicializar Vault
vault operator init || { echo "Error durante la inicialización de Vault. Abortando."; exit 1; }

echo ""
echo "--- ¡PASO CRÍTICO: DESELLADO DE VAULT! ---"
echo "Vault está sellado. Necesitarás introducir 3 de las 5 'Unseal Keys' que acabas de guardar."
echo "Ejecuta 'vault operator unseal' 3 veces, pegando una clave diferente CADA VEZ."

# Guía para el usuario sobre cómo desellar (no ejecutable directamente en el script, es manual)
echo "Ejemplo:"
echo "vault operator unseal # Pega la CLAVE 1"
echo "vault operator unseal # Pega la CLAVE 2"
echo "vault operator unseal # Pega la CLAVE 3"
echo ""
echo "Después de cada clave, revisa el 'Unseal Progress'. Cuando llegue a '3/3', Vault se desellará."
echo "Una vez desellado, ejecuta 'vault status' para confirmar que 'Sealed: false'."
read -p "Presiona Enter para continuar con el desellado manual (y luego ejecuta vault status)..."

# 7.2 Verificación después del desellado
echo "Verificando el estado de Vault (debería ser 'Sealed: false'):"
vault status || { echo "Vault no se deselló correctamente. Abortando."; exit 1; }
echo "-----------------------------------"

---

### 8. Configuración de Motores de Secretos y Autenticación

echo "--- 8. CONFIGURACIÓN DE MOTORES DE SECRETOS Y AUTENTICACIÓN ---"

echo "--- ¡PASO CRÍTICO: AUTENTICACIÓN COMO ROOT! ---"
echo "Necesitas autenticarte con el 'Initial Root Token' que guardaste durante la inicialización (Punto 7.1)."
echo "Ejecuta: vault login <TU_ROOT_TOKEN_INICIAL>"
echo "Y verifica que estás autenticado con: vault token lookup"
read -p "Presiona Enter para continuar una vez que te hayas autenticado como root..."

# 8.1 Habilitar el motor de secretos KV v2 en la ruta 'secret/'
echo "Habilitando el motor de secretos KV v2 en la ruta 'secret/'..."
vault secrets enable -path=secret kv-v2 || { echo "Error al habilitar el motor KV v2. Abortando."; exit 1; }

# 8.2 Habilitar el método de autenticación AppRole
echo "Habilitando el método de autenticación AppRole..."
vault auth enable approle || { echo "Error al habilitar AppRole. Abortando."; exit 1; }

# 8.3 Crear la política de acceso para el bot
echo "Creando el archivo de la política 'trading-bot-policy.hcl' en ${VAULT_POLICIES_DIR}/..."
sudo bash -c "cat <<EOF > ${VAULT_POLICIES_DIR}/trading-bot-policy.hcl
path \"secret/data/trading-bot/*\" {
  capabilities = [\"read\"]
}

# Permitir al token de AppRole renovar su lease (necesario para checkAndRenewToken)
path \"auth/approle/login\" {
  capabilities = [\"create\", \"update\"]
}

# Permitir al token de AppRole consultar el estado de su lease (necesario para checkAndRenewToken)
path \"sys/leases/lookup\" {
  capabilities = [\"read\"]
}
EOF" || { echo "Error al crear el archivo de política. Abortando."; exit 1; }

echo "Escribiendo la política 'trading-bot-policy' en Vault..."
vault policy write trading-bot-policy "${VAULT_POLICIES_DIR}/trading-bot-policy.hcl" || { echo "Error al escribir la política. Abortando."; exit 1; }

# 8.4 Crear el rol de AppRole para el bot
echo "Creando el rol de AppRole 'trading-bot'..."
vault write auth/approle/role/trading-bot \
    token_ttl="1h" \
    token_max_ttl="24h" \
    secret_id_ttl="23h" \
    token_policies="trading-bot-policy" \
    bind_secret_id="true" \
    secret_id_num_uses="0" \
    secret_id_bound_cidrs="127.0.0.1/32" || { echo "Error al crear el rol AppRole. Abortando."; exit 1; }

echo "Configuración de AppRole completada."
echo "-----------------------------------"

---

### 9. Obtención de Credenciales de AppRole y Almacenamiento de Secretos (¡PASOS MANUALES CRÍTICOS!)

echo ""
echo "--- 9. OBTENCIÓN Y ALMACENAMIENTO DE CREDENCIALES DE VAULT (AppRole) ---"
echo ""

cat <<EOF
Este paso es fundamental para que tus bots puedan autenticarse de forma segura con Vault. Vamos a obtener el RoleID y un SecretID para el método de autenticación AppRole.

¡IMPORTANTE! Hemos configurado el rol 'trading-bot' en Vault con un secret_id_ttl (tiempo de vida del SecretID) prolongado (ej. 1 año o "nunca expira"). Esto asegura que el SecretID que obtengas sea válido por un periodo extendido, permitiendo que tus servicios de larga duración (bots y demonio de monitoreo) lo usen para autenticarse.

Estos valores son las credenciales que tu bot y demonio usarán para solicitar tokens de Vault y acceder a los secretos. El SecretID es MUY SENSIBLE y debe ser tratado como una contraseña de alta importancia.

Guarda AMBOS valores (RoleID y SecretID) de forma EXTREMADAMENTE SEGURA, FUERA DE ESTE VPS (ej. en un gestor de contraseñas seguro, un almacén de credenciales para tu CI/CD, etc.). Estos valores se inyectarán como variables de entorno (VAULT_APPROLE_ROLE_ID y VAULT_APPROLE_SECRET_ID) en la configuración de tus servicios.
EOF

# 9.1 Obtener el RoleID
echo ""
echo "9.1. Obteniendo el RoleID del rol 'trading-bot':"
vault read auth/approle/role/trading-bot/role-id || { echo "Error al leer RoleID. Abortando."; exit 1; }

# 9.2 Generar un SecretID (este SecretID tendrá la duración configurada en el secret_id_ttl de tu rol)
echo ""
echo "9.2. Generando un SecretID para el rol 'trading-bot':"
vault write -f auth/approle/role/trading-bot/secret-id || { echo "Error al generar SecretID. Abortando."; exit 1; }

echo ""
echo "--- ¡PASO CRÍTICO: ESCRITURA DE SECRETOS DE TRADING PARA CADA INSTANCIA! ---"
echo "Asegúrate de ejecutar este comando para CADA UNA de tus instancias de bot (ej. bot1, bot2, bot3)."
echo "Reemplaza 'INSTANCE_NAME' por el nombre de la instancia y los valores por tus credenciales REALES."
echo ""
echo "Ejemplo para la instancia 'bot1':"
echo "vault kv put secret/data/bots/bot1 \\"
echo "    api_key=\"<TU_API_KEY_REAL_BOT1>\" \\"
echo "    secret_key=\"<TU_SECRET_KEY_REAL_BOT1>\" \\"
echo "    passphrase=\"<TU_PASSPHRASE_REAL_BOT1>\" # Si aplica"
echo ""
echo "Ejemplo para la instancia 'bot2':"
echo "vault kv put secret/data/bots/bot2 \\"
echo "    api_key=\"<TU_API_KEY_REAL_BOT2>\" \\"
echo "    secret_key=\"<TU_SECRET_KEY_REAL_BOT2>\" \\"
echo "    passphrase=\"<TU_PASSPHRASE_REAL_BOT2>\" # Si aplica"
echo ""

read -p "Presiona Enter para continuar una vez que hayas ejecutado el comando de escritura de secretos para TODAS tus instancias..."

echo "-----------------------------------"
echo "--- CONFIGURACIÓN DE VAULT COMPLETADA EXITOSAMENTE ---"
echo "Recuerda integrar RoleID y SecretID en tu aplicación y ajustar tu código PHP."
echo "¡Mucha suerte con el entorno de trabajo!"
echo "-----------------------------------"