<?php

/**
 * Bot de Trading Automatizado con Binance Futures API
 *
 * Este bot implementa estrategias de trading algorítmico utilizando:
 * - ReactPHP para programación asíncrona
 * - HashiCorp Vault para gestión segura de credenciales
 * - WebSocket de Binance para datos en tiempo real
 * - Múltiples algoritmos de trading (RSI, MACD, EMA)
 * - Sistema de logging profesional con rotación diaria
 *
 * @author Alan
 * @version 2.4 - Corrección de instanciación de VCM
 * @since 2025
 */

declare(strict_types=1);

namespace App;

use App\{
    AlgoritApi,
    SetOrdersBuySel,
    Settings,
    ResponseApi,
    Auth, // Clase Auth
    WebSocketApi,
    ValidResponse,
    DailyRotatingLogger,
    VaultCredentialsManager // Clase VaultCredentialsManager
};
use React\EventLoop\LoopInterface;
use React\EventLoop\Loop;
use GuzzleHttp\Client as GuzzleClient;
use React\Promise\PromiseInterface;
use Monolog\Logger; // Importar Logger para las constantes de nivel de log

/**
 * Clase principal del Bot de Trading con logging profesional
 */
class TradingBot
{
    // Constantes del sistema: centralizadas y de fácil acceso
    private const DEFAULT_ALGORITHM = 'ImpulseMacdEma100';
    private const CANDLE_INTERVAL_MINUTES = 15; // Definimos el intervalo de la vela en minutos
    private const LISTEN_KEY_RENEWAL_INTERVAL = 1800; // 30 minutos
    private const POSITION_CHECK_INTERVAL = 60; // 1 minuto
    private const LOG_RETENTION_DAYS = 7; // Días de retención de logs
    private const GUZZLE_TIMEOUT = 30; // segundos
    private const BINANCE_BASE_URI = 'https://fapi.binance.com/';
    private const USER_AGENT = 'TradingBot/2.4';
    private const CONFIG_DIR = __DIR__ . '/../../config/';
    private const LOG_DIR = __DIR__ . '/../../logs';
    private const ENV_FILE_DIR = __DIR__ . '/../../bots';
    private const RUNTIME_DIR = __DIR__ . '/../../runtime'; // Directorio para VaultCredentialsManager

    // Propiedades principales: tipado estricto cuando sea posible
    private LoopInterface $loop;
    private GuzzleClient $httpClient;
    private Auth $authService;
    private ResponseApi $responseApi;
    private WebSocketApi $webSocketApi;
    private SetOrdersBuySel $apiSetOrdersBuySel;
    private AlgoritApi $apiAlgoritmo;
    private ValidResponse $validResponse;
    private DailyRotatingLogger $logger;
    private VaultCredentialsManager $vaultCredentialsManager; // Propiedad para VaultCredentialsManager

    // Configuración
    private array $config;
    private array $order;
    private array $porini;
    private string $instanceName;
    private string $algorithm;
    private string $symbol;
    private string $currency;

    /**
     * Constructor del Bot de Trading
     * @param array $commandArgs Argumentos de línea de comandos
     * @throws \RuntimeException Si la inicialización falla críticamente
     */
    public function __construct(array $commandArgs)
    {
        $this->loop = Loop::get();
        $this->initializeLogger();
        $this->logger->info('Iniciando constructor de TradingBot');
        try {
            $this->loadConfiguration($commandArgs);
            $this->initializeHttpClient();
            // Inicializar VaultCredentialsManager primero, ya que es el que obtiene las credenciales
            $this->initializeVaultCredentialsManager();
            // Auth ahora recibe VaultCredentialsManager
            $this->initializeAuthService();
            $this->initializeApiServices();
        } catch (\Throwable $e) {
            $this->handleCriticalError($e);
            throw new \RuntimeException("Fallo en la inicialización del bot: " . $e->getMessage(), 0, $e);
        }
        $this->logger->info('TradingBot inicializado correctamente');
    }

    /**
     * Inicializa el sistema de logging profesional
     * @throws \RuntimeException Si no se puede inicializar el logger
     */
    private function initializeLogger(): void
    {
        try {
            if (!is_dir(self::LOG_DIR) && !mkdir(self::LOG_DIR, 0755, true) && !is_dir(self::LOG_DIR)) {
                throw new \RuntimeException(sprintf('Directory "%s" was not created', self::LOG_DIR));
            }

            $logPath = self::LOG_DIR . '/trading_bot.log';
            $this->logger = new DailyRotatingLogger($logPath, self::LOG_RETENTION_DAYS);

            $this->logger->info('Sistema de logging inicializado', [
                'log_path' => $logPath,
                'retention_days' => self::LOG_RETENTION_DAYS
            ]);
        } catch (\Throwable $e) {
            error_log("[LOGGER_INIT_ERROR] No se pudo inicializar el logger: " . $e->getMessage());
            throw new \RuntimeException("Fallo crítico al inicializar sistema de logging: " . $e->getMessage(), 0, $e);
        }
    }

    /**
     * Carga y valida la configuración del bot
     *
     * @param array $commandArgs Argumentos de línea de comandos
     * @throws \RuntimeException Si la configuración es inválida o el archivo no existe
     */
    private function loadConfiguration(array $commandArgs): void
    {
        $this->algorithm = $commandArgs['type'] ?? self::DEFAULT_ALGORITHM;
        $configFileName = $commandArgs['config'] ?? null;

        $this->logger->debug('Argumentos de línea de comandos procesados', [
            'algorithm' => $this->algorithm,
            'config_file' => $configFileName
        ]);

        if (empty($configFileName)) {
            throw new \RuntimeException("CRÍTICO: No se proporcionó el parámetro --config. Use: --config=archivo.php");
        }

        $this->instanceName = basename($configFileName, '.php');
        $configFilePath = self::CONFIG_DIR . $configFileName;

        if (!file_exists($configFilePath)) {
            throw new \RuntimeException("CRÍTICO: Archivo de configuración '{$configFilePath}' no encontrado.");
        }

        $instanceSpecificSettings = require $configFilePath;

        // Fusionar configuraciones de forma más segura
        $defaultSettings = (new Settings())->settings;

        $this->config = array_merge($defaultSettings['config'] ?? [], $instanceSpecificSettings['config'] ?? []);
        $this->order = $defaultSettings['order'];
        $this->porini = $defaultSettings['porini'];

        // Configurar parámetros específicos y asegurar valores por defecto
        $this->config['algorit'] = $this->algorithm;
        $this->symbol = $this->config['symbol'] ?? 'DOTUSDT';
        $this->currency = $this->config['currency'] ?? 'USDT';

        $this->logger->info('Configuración cargada exitosamente', [
            'instance' => $this->instanceName,
            'algorithm' => $this->algorithm,
            'symbol' => $this->symbol,
            'currency' => $this->currency,
            'config' => $this->config,
            'order' => $this->order,
            'porini' => $this->porini
        ]);
    }

    /**
     * Configura el cliente HTTP Guzzle
     */
    private function initializeHttpClient(): void
    {
        $guzzleConfig = [
            'base_uri' => self::BINANCE_BASE_URI,
            'timeout' => self::GUZZLE_TIMEOUT,
            'connect_timeout' => self::GUZZLE_TIMEOUT,
            'http_errors' => true,
            'verify' => true,
            'headers' => [
                'User-Agent' => self::USER_AGENT,
                'Accept' => 'application/json',
                'Content-Type' => 'application/json'
            ]
        ];

        $this->httpClient = new GuzzleClient($guzzleConfig);
        $this->logger->debug('Cliente HTTP Guzzle configurado', [
            'base_uri' => $guzzleConfig['base_uri'],
            'timeout' => $guzzleConfig['timeout']
        ]);
    }

    /**
     * Inicializa VaultCredentialsManager
     * @throws \RuntimeException Si faltan variables de entorno de Vault
     */
    private function initializeVaultCredentialsManager(): void
    {
        $vaultAddr = getenv('VAULT_ADDR');
        $vaultRoleName = getenv('VAULT_APPROLE_ROLE_NAME'); // Este es el role_id de TradingBot para Vault
        $vaultNamespace = getenv('VAULT_NAMESPACE') ?: ''; // Usar cadena vacía por defecto si no hay namespace
        $vaultCaCertPath = getenv('VAULT_CA_CERT_PATH') ?: null;

        // Asumiendo que el SecretID inicial del VCM se pasa como variable de entorno
        $vcmSecretId = getenv('VAULT_APPROLE_SECRET_ID') ?: null;

        // Nombre del AppRole para los bots de trading (si es diferente al del VCM)
        $tradingBotAppRoleName = getenv('VAULT_TRADING_BOT_APPROLE_NAME') ?: 'trading-bot-approle';

        // Directorio donde se inyectan los .env de los bots (ej: /var/www/html/bots)
        $botEnvFileDir = getenv('BOT_ENV_FILE_DIR') ?: self::ENV_FILE_DIR;
        $botEnvFilePrefix = getenv('BOT_ENV_FILE_PREFIX') ?: 'bot'; // Ej: 'bot-instance1.env'

        // Lista de bots a gestionar (ej: ['bitradenes', 'bitradeali'])
        // Esto podría venir de un archivo de configuración o una variable de entorno más compleja
        $botsToManage = json_decode(getenv('BOTS_TO_MANAGE') ?: '[]', true);
        if (!is_array($botsToManage)) {
            $botsToManage = []; // Asegurar que sea un array
        }
        // Transformar a formato esperado por VCM: [['instance' => 'bot1'], ...]
        $botsToManageFormatted = array_map(fn($name) => ['instance' => $name], $botsToManage);


        if (!$vaultAddr || !$vaultRoleName) {
            $this->logger->critical('Variables de entorno de Vault cruciales no configuradas', [
                'vault_addr_set' => !empty($vaultAddr),
                'vault_role_name_set' => !empty($vaultRoleName),
            ]);
            throw new \RuntimeException(
                "CRÍTICO: Variables de entorno de Vault no configuradas.\n" .
                "Requeridas: VAULT_ADDR, VAULT_APPROLE_ROLE_NAME"
            );
        }

        // Asegurarse de que el directorio de runtime existe
        if (!is_dir(self::RUNTIME_DIR) && !mkdir(self::RUNTIME_DIR, 0700, true) && !is_dir(self::RUNTIME_DIR)) {
            throw new \RuntimeException(sprintf('Directory "%s" was not created', self::RUNTIME_DIR));
        }

        // Configuración completa para VaultCredentialsManager
        $vaultConfig = [
            'vault_url' => $vaultAddr,
            'role_id' => $vaultRoleName, // Role ID de este VCM
            'monitor_app_role_name' => $vaultRoleName, // El mismo AppRole para el VCM
            'secret_id' => $vcmSecretId, // SecretID inicial del VCM (si se proporciona)
            'namespace' => $vaultNamespace,
            'mount_path' => 'auth/approle', // Valor por defecto
            'secret_path' => 'login', // Valor por defecto
            'runtime_dir' => self::RUNTIME_DIR, // Directorio para tokens/secret_id cifrados
            'log_file' => self::LOG_DIR . '/vault_manager.log', // Log separado para Vault
            'log_level' => Logger::DEBUG, // Usar la constante de Monolog
            'log_retention_days' => self::LOG_RETENTION_DAYS,
            'ssl_ca_file' => $vaultCaCertPath,
            'ssl_verify_peer' => !empty($vaultCaCertPath), // Verificar si se proporciona CA

            // Configuración para la gestión de SecretID de bots de trading
            'trading_bot_app_role_name' => $tradingBotAppRoleName,
            'bot_env_file_dir' => $botEnvFileDir,
            'bot_env_file_prefix' => $botEnvFilePrefix,
            'bots_to_manage' => $botsToManageFormatted,
            'bot_vault_secret_id_max_ttl_seconds' => VaultCredentialsManager::SECRET_ID_DEFAULT_TTL,
            'bot_rotation_threshold_seconds' => VaultCredentialsManager::VCM_SECRET_ID_RENEWAL_BUFFER,
            // Asumiendo que TelegramBotMonitorDaemon::BOT_SECRET_ID_INJECTION_TIMESTAMP_FILE es una constante pública
            // o se define aquí si no está disponible globalmente.
            'bot_secret_id_injection_timestamp_file' => self::RUNTIME_DIR . '/.bot_secret_id_injection_timestamp',
        ];

        // ** CORRECCIÓN AQUÍ: Pasar $this->loop como segundo argumento **
        $this->vaultCredentialsManager = new VaultCredentialsManager($vaultConfig, $this->loop);
        $this->logger->info('VaultCredentialsManager inicializado', [
            'vault_url' => $vaultAddr,
            'role_id' => $vaultRoleName,
            'runtime_dir' => self::RUNTIME_DIR
        ]);
    }

    /**
     * Configura el servicio de autenticación (Auth)
     * Auth ahora es "ciego" a Vault y solo recibe las credenciales de Binance
     */
    private function initializeAuthService(): void
    {
        // ** CAMBIO CLAVE **
        // Auth ahora recibe la instancia de VaultCredentialsManager y el instanceName
        $this->authService = new Auth(
            $this->vaultCredentialsManager, // Pasamos la instancia de VCM
            $this->instanceName // Pasamos el nombre de la instancia para la ruta de secretos
        );

        $this->logger->info('Servicio de autenticación Auth configurado (sin credenciales aún).');
    }

    /**
     * Inicializa los servicios de API
     */
    private function initializeApiServices(): void
    {
        $this->validResponse = new ValidResponse();

        $this->responseApi = new ResponseApi(
            $this->httpClient,
            $this->loop,
            $this->authService, // Auth ya tiene el acceso a las credenciales
            $this->validResponse
        );

        $this->webSocketApi = new WebSocketApi(
            $this->loop,
            $this->authService, // Auth ya tiene el acceso a las credenciales
            $this->httpClient,
            $this->responseApi
        );

        $this->apiSetOrdersBuySel = new SetOrdersBuySel(
            $this->config,
            $this->responseApi,
            $this->webSocketApi,
            $this->loop
        );

        $this->apiAlgoritmo = new AlgoritApi($this->config, $this->responseApi);

        $this->logger->debug('Servicios de API inicializados', [
            'services' => ['ValidResponse', 'ResponseApi', 'WebSocketApi', 'SetOrdersBuySel', 'AlgoritApi']
        ]);
    }

    /**
     * Inicia el bot de trading
     *
     * @return PromiseInterface
     */
    public function start(): PromiseInterface
    {
        $this->logger->info('Iniciando bot de trading', [
            'algorithm' => $this->algorithm,
            'symbol' => $this->symbol,
            'instance' => $this->instanceName
        ]);

        // Paso 1: Asegurar la autenticación con Vault y obtener las credenciales de Binance
        // Ahora Auth se encarga de esto con su propio método initializeTradingSecrets
        return $this->authService->initializeTradingSecrets()
            ->then(function () {
                $this->logger->info('[TradingBot] Credenciales de Binance cargadas y Auth actualizado. Iniciando configuración del bot.');
                return $this->performInitialSetup();
            })
            ->then(function () {
                $this->setupPeriodicTimers();
                $this->logger->info('Bot completamente inicializado y en funcionamiento');
            })
            ->otherwise(function (\Throwable $e) {
                $this->handleCriticalError($e);
                throw $e;
            });
    }

    /**
     * Obtiene las credenciales de Binance de Vault y las actualiza en AuthService.
     * Esta función ya no es necesaria con la refactorización de Auth.
     * Su lógica ha sido movida a Auth::initializeTradingSecrets().
     * Sin embargo, la mantenemos para el timer periódico, llamando al método de Auth.
     * @return PromiseInterface
     */
    private function updateBinanceCredentialsFromVault(): PromiseInterface
    {
        $this->logger->info('Asegurando autenticación con Vault y obteniendo credenciales de Binance a través de Auth...');
        return $this->authService->initializeTradingSecrets()
            ->then(function () {
                $this->logger->info('Credenciales de Binance refrescadas en Auth por timer.');
                return true; // Resolver con true para indicar éxito
            })
            ->otherwise(function (\Throwable $e) {
                $this->logger->error('Fallo al refrescar credenciales de Binance en Auth por timer.', [
                    'error' => $e->getMessage(),
                    'trace' => $e->getTraceAsString()
                ]);
                return \React\Promise\reject($e); // Rechazar la promesa en caso de error
            });
    }


    /**
     * Realiza la configuración inicial del bot
     *
     * @return PromiseInterface
     */
    private function performInitialSetup(): PromiseInterface
    {
        $this->logger->info('Iniciando configuración inicial del bot');

        return $this->responseApi->checkActivePosition()
            ->then(function (bool $hasActivePosition) {
                $this->apiSetOrdersBuySel->setOrderPending($hasActivePosition);
                $this->logger->info('Estado inicial de posición verificado', ['has_active_position' => $hasActivePosition]);
                return $this->responseApi->getAccount(['symbol' => $this->symbol, 'currency' => $this->currency]);
            })
            ->then(function (array $accountData) {
                $this->porini['porLe'] = $accountData['leverage'] ?? $this->porini['porLe'] ?? null;
                $this->logger->info('Datos de cuenta obtenidos', ['leverage' => $this->porini['porLe'], 'symbol' => $this->symbol]);
                return $this->webSocketApi->connect();
            })
            ->then(function () {
                $this->logger->info('Conexión WebSocket establecida exitosamente');
                return \React\Promise\resolve();
            })
            ->otherwise(function (\Throwable $e) {
                $this->logger->error('Fallo en configuración inicial', [
                    'error' => $e->getMessage(),
                    'trace' => $e->getTraceAsString()
                ]);
                return \React\Promise\reject($e);
            });
    }

    /**
     * Configura todos los temporizadores periódicos
     */
    private function setupPeriodicTimers(): void
    {
        $this->scheduleNextCheck(self::CANDLE_INTERVAL_MINUTES); // Primera ejecución programada

        $this->setupListenKeyRenewalTimer();
        $this->setupPositionCheckTimer();

        // Nuevo timer para refrescar credenciales de Vault y Binance API Keys periódicamente
        // Se refrescará con una frecuencia segura, por ejemplo, la mitad del TTL por defecto del secret_id
        // Asumimos que SECRET_ID_DEFAULT_TTL es una constante pública en VaultCredentialsManager
        $refreshInterval = (defined('App\VaultCredentialsManager::SECRET_ID_DEFAULT_TTL') ? VaultCredentialsManager::SECRET_ID_DEFAULT_TTL : 86400) / 2;
        $this->loop->addPeriodicTimer($refreshInterval, function () {
            $this->logger->info('Intentando refrescar credenciales de Vault y Binance API Keys.');
            $this->updateBinanceCredentialsFromVault() // Llamamos al método que delega en Auth
                ->then(function () {
                    $this->logger->info('Credenciales de Vault y Binance API Keys refrescadas exitosamente por timer.');
                })
                ->otherwise(function (\Throwable $e) {
                    $this->logger->error('Error al refrescar credenciales de Vault y Binance API Keys por timer', [
                        'error' => $e->getMessage(),
                        'trace' => $e->getTraceAsString()
                    ]);
                });
        });

        $this->logger->info('Temporizadores periódicos configurados', [
            'strategy_scheduling' => 'Aligned with ' . self::CANDLE_INTERVAL_MINUTES . ' min candles',
            'listen_key_renewal_interval' => self::LISTEN_KEY_RENEWAL_INTERVAL,
            'position_check_interval' => self::POSITION_CHECK_INTERVAL,
            'vault_binance_refresh_interval' => $refreshInterval . ' seconds'
        ]);
    }

    /**
     * Ejecuta la evaluación de estrategias de trading y programa la próxima ejecución.
     */
    private function processTrading(): void
    {
        $this->logger->debug('Iniciando nuevo ciclo de evaluación de estrategia (programado)');

        // Paso crucial: Refrescar las credenciales de Binance y Vault token antes de CADA ciclo de trading
        // Esto asegura que siempre operamos con credenciales frescas y válidas.
        $this->updateBinanceCredentialsFromVault()
            ->then(function () {
                $this->logger->debug('Credenciales de Binance y Vault refrescadas antes del ciclo de trading.');

                if ($this->apiSetOrdersBuySel->hasPendingOrder()) {
                    $this->logger->debug('Orden pendiente detectada, saltando evaluación.');
                } else {
                    try {
                        $startTime = microtime(true);
                        $algorithmResult = $this->runSelectedAlgorithm();
                        $executionTime = round((microtime(true) - $startTime) * 1000, 2);

                        $this->logger->debug('Algoritmo ejecutado', [
                            'algorithm' => $this->algorithm,
                            'execution_time_ms' => $executionTime,
                            'result_keys' => array_keys($algorithmResult)
                        ]);

                        $this->processAlgorithmResult($algorithmResult);
                    } catch (\Throwable $e) {
                        $this->logger->error('Fallo en evaluación de estrategia', [
                            'algorithm' => $this->algorithm,
                            'error' => $e->getMessage(),
                            'trace' => $e->getTraceAsString()
                        ]);
                    }
                }
                // Después de procesar, programar la próxima ejecución para el inicio del siguiente intervalo de vela.
                $this->scheduleNextCheck(self::CANDLE_INTERVAL_MINUTES);
            })
            ->otherwise(function (\Throwable $e) {
                $this->logger->error('Error al refrescar credenciales de Vault o Binance antes del ciclo de trading. El ciclo se saltará.', [
                    'error' => $e->getMessage(),
                    'trace' => $e->getTraceAsString()
                ]);
                // Si falla el refresh de credenciales, programar el siguiente chequeo para no detener el bot
                $this->scheduleNextCheck(self::CANDLE_INTERVAL_MINUTES);
            });
    }

    /**
     * Calcula el tiempo de espera hasta el inicio del próximo intervalo de vela.
     *
     * @param int $intervalMinutes El intervalo de tiempo de la vela en minutos (ej. 15, 30, 60).
     * @return int Los segundos a esperar.
     */
    private function calculateSleepTime(int $intervalMinutes): int
    {
        $currentTime = time();
        $currentMinute = (int) date('i', $currentTime);
        $currentSecond = (int) date('s', $currentTime);

        $minutesIntoInterval = $currentMinute % $intervalMinutes;

        $secondsToNextInterval = ($intervalMinutes - $minutesIntoInterval - 1) * 60 + (60 - $currentSecond);

        if ($secondsToNextInterval >= ($intervalMinutes * 60)) {
            $secondsToNextInterval = 0;
        }
        if ($secondsToNextInterval <= 0) {
            $secondsToNextInterval = $intervalMinutes * 60;
        }

        return max(1, $secondsToNextInterval);
    }

    /**
     * Programa la próxima verificación de la estrategia para el inicio del siguiente intervalo de vela.
     *
     * @param int $intervalMinutes El intervalo de tiempo de la vela en minutos (ej. 15, 30, 60).
     */
    private function scheduleNextCheck(int $intervalMinutes): void
    {
        $sleepSeconds = $this->calculateSleepTime($intervalMinutes);

        $this->loop->addTimer($sleepSeconds, function() {
            $this->processTrading();
        });

        $sleepMinutes = round($sleepSeconds / 60, 1);
        $this->logger->info('Próximo ciclo de estrategia programado', [
            'interval_minutes' => $intervalMinutes,
            'actual_wait_minutes' => $sleepMinutes,
            'actual_wait_seconds' => $sleepSeconds
        ]);
    }

    /**
     * Ejecuta el algoritmo seleccionado
     *
     * @return array Resultado del algoritmo
     */
    private function runSelectedAlgorithm(): array
    {
        $this->logger->debug('Ejecutando algoritmo', ['algorithm' => $this->algorithm]);

        $method = match ($this->algorithm) {
            'RsiMacd' => 'runAlgoritRsiMacd',
            'ImpulseMacd' => 'runAlgorithmAnalysis',
            'ImpulseMacdEma100' => 'runAlgorithmImpulseMacdEMA100',
            'ImpulseMacdBreakPro' => 'runAlgorithmImpulseMacdBreakPro',
            default => 'runAlgorithmImpulseMacdEMA100', // Algoritmo por defecto
        };

        if ($method === 'runAlgorithmImpulseMacdEMA100' && $this->algorithm !== 'ImpulseMacdEma100') {
            $this->logger->warning('Algoritmo desconocido, usando por defecto', [
                'requested_algorithm' => $this->algorithm,
                'default_algorithm' => 'ImpulseMacdEma100'
            ]);
        }

        return $this->apiAlgoritmo->$method();
    }

    /**
     * Procesa el resultado del algoritmo y ejecuta trades si es necesario
     *
     * @param array $algorithmResult Resultado del algoritmo
     */
    private function processAlgorithmResult(array $algorithmResult): void
    {
        $this->updateConfigurationFromAlgorithm($algorithmResult);

        $shouldEmitOrder = (bool)($algorithmResult['emitir'] ?? false);

        if ($shouldEmitOrder) {
            $this->logger->info('Condiciones de trading cumplidas, intentando ejecutar trade');
            $this->attemptTrade($algorithmResult);
        } else {
            $this->logger->debug('Condiciones de trading no cumplidas', [
                'emit_signal' => $shouldEmitOrder
            ]);
        }
    }

    /**
     * Actualiza la configuración basada en los resultados del algoritmo
     *
     * @param array $algorithmResult Resultado del algoritmo
     */
    private function updateConfigurationFromAlgorithm(array $algorithmResult): void
    {
        $tokentel = $algorithmResult["tokentel"] ?? [];

        $this->porini['porTp'] = $tokentel["potp"] ?? $this->porini['porTp'];
        $this->porini['porSm'] = $tokentel["posl"] ?? $this->porini['porSm'];
        $this->config['cv'] = $algorithmResult['CV'] ?? $this->config['cv'] ?? 0;

        $this->logger->debug('Configuración actualizada desde algoritmo', [
            'take_profit' => $this->porini['porTp'],
            'stop_loss' => $this->porini['porSm'],
            'confidence_value' => $this->config['cv']
        ]);
    }

    /**
     * Intenta ejecutar un trade basado en las señales del algoritmo
     *
     * @param array $algorithmResult Resultado del algoritmo
     */
    private function attemptTrade(array $algorithmResult): void
    {
        $CV = $algorithmResult['CV'] ?? 0;

        $tradeData = $this->prepareTradeData($algorithmResult, $CV);
        $this->apiSetOrdersBuySel->setData($tradeData, $CV);

        $balance = $this->apiSetOrdersBuySel->getBalanceAva();

        $this->logger->info('Verificando balance para trade', [
            'available_balance' => $balance,
            'confidence_value' => $CV
        ]);

        if ($balance > 0) {
            $this->logger->info('Ejecutando trade', [
                'balance' => $balance,
                'symbol' => $this->symbol,
                'algorithm' => $this->algorithm
            ]);

            $tradeResult = $this->apiSetOrdersBuySel->optTrade($CV);

            if ($tradeResult) {
                $this->logger->info('Trade ejecutado exitosamente', [
                    'result' => $tradeResult,
                    'symbol' => $this->symbol,
                    'confidence_value' => $CV
                ]);
                $this->apiSetOrdersBuySel->setOrderPending(true);
            } else {
                $this->logger->error('Fallo al ejecutar trade', [
                    'symbol' => $this->symbol,
                    'balance' => $balance,
                    'confidence_value' => $CV
                ]);
            }
        } else {
            $this->logger->warning('Balance insuficiente para ejecutar trade', [
                'available_balance' => $balance,
                'symbol' => $this->symbol
            ]);
        }
    }

    /**
     * Prepara los datos necesarios para el trade
     *
     * @param array $algorithmResult Resultado del algoritmo
     * @param float $CV Valor de confianza
     * @return array Datos del trade
     */
    private function prepareTradeData(array $algorithmResult, float $CV): array
    {
        $tradeData = [
            'algorithm' => $this->algorithm,
            'symbol' => $this->symbol,
            'currency' => $this->currency,
            'cv' => $CV,
            'timestamp' => time(),
            'porini' => $this->porini,
            'order' => $this->order,
            'result' => $algorithmResult
        ];

        $this->logger->debug('Datos de trade preparados', [
            'algorithm' => $tradeData['algorithm'],
            'symbol' => $tradeData['symbol'],
            'confidence_value' => $tradeData['cv']
        ]);

        return $tradeData;
    }

    /**
     * Configura el temporizador de renovación de ListenKey
     */
    private function setupListenKeyRenewalTimer(): void
    {
        $this->loop->addPeriodicTimer(self::LISTEN_KEY_RENEWAL_INTERVAL, function () {
            if ($this->apiSetOrdersBuySel->hasPendingOrder()) {
                $this->logger->debug('Renovando ListenKey - orden pendiente detectada');
                $this->responseApi->getListenKey()
                    ->then(function (string $listenKey) {
                        $this->webSocketApi->updateListenKey($listenKey);
                        $this->logger->info('ListenKey renovada exitosamente', [
                            'listen_key_preview' => substr($listenKey, 0, 8) . '...'
                        ]);
                    })
                    ->otherwise(function (\Throwable $e) {
                        $this->logger->error('Error al renovar ListenKey', [
                            'error' => $e->getMessage(),
                            'trace' => $e->getTraceAsString()
                        ]);
                    });
            } else {
                $this->logger->debug('Renovación de ListenKey omitida - no hay órdenes pendientes');
            }
        });
    }

    /**
     * Configura el temporizador de verificación de posición
     */
    private function setupPositionCheckTimer(): void
    {
        $this->loop->addPeriodicTimer(self::POSITION_CHECK_INTERVAL, function () {
            \React\Promise\resolve($this->performPositionCheck())
                ->otherwise(function (\Throwable $e) {
                    $this->logger->error('Error capturado por el timer de verificación de posición', [
                        'error' => $e->getMessage(),
                        'trace' => $e->getTraceAsString()
                    ]);
                });
        });
    }

    /**
     * Verifica el estado de la posición vía API (respaldo)
     * @return PromiseInterface
     */
    private function performPositionCheck(): PromiseInterface
    {
        $this->logger->debug('Verificando estado de posición vía API');

        return $this->responseApi->checkActivePosition()
            ->then(function (bool $hasActivePosition) {
                $currentPendingState = $this->apiSetOrdersBuySel->hasPendingOrder();

                if ($currentPendingState !== $hasActivePosition) {
                    $this->apiSetOrdersBuySel->setOrderPending($hasActivePosition);
                    $this->logger->warning('Estado de posición actualizado', [
                        'previous_state' => $currentPendingState ? 'PENDING' : 'INACTIVE',
                        'new_state' => $hasActivePosition ? 'ACTIVE' : 'INACTIVE',
                        'symbol' => $this->symbol
                    ]);
                } else {
                    $this->logger->debug('Estado de posición consistente', [
                        'state' => $hasActivePosition ? 'ACTIVE' : 'INACTIVE',
                        'symbol' => $this->symbol
                    ]);
                }
            })
            ->otherwise(function (\Throwable $e) {
                $this->logger->error('Error en verificación de posición', [
                    'error' => $e->getMessage(),
                    'symbol' => $this->symbol,
                    'trace' => $e->getTraceAsString()
                ]);
                return \React\Promise\reject($e);
            });
    }

    /**
     * Maneja errores críticos del sistema.
     * Detiene el loop de eventos y registra el error.
     *
     * @param \Throwable $e Excepción crítica
     */
    private function handleCriticalError(\Throwable $e): void
    {
        $this->logger->critical('Error crítico en el bot de trading', [
            'instance' => $this->instanceName ?? 'N/A',
            'error' => $e->getMessage(),
            'file' => $e->getFile(),
            'line' => $e->getLine(),
            'trace' => $e->getTraceAsString()
        ]);

        error_log("[CRITICAL] Bot '{$this->instanceName}' falló: " . $e->getMessage());

        $this->shutdown();
    }

    /**
     * Apaga el bot de forma controlada.
     * Cierra conexiones y detiene el event loop.
     */
    public function shutdown(): void
    {
        $this->logger->info('Iniciando cierre controlado del bot', [
            'instance' => $this->instanceName ?? 'N/A'
        ]);

        try {
            if (isset($this->webSocketApi)) {
                $this->webSocketApi->disconnect();
                $this->logger->debug('Conexión WebSocket cerrada');
            }

            if (isset($this->loop) && $this->loop->isRunning()) {
                $this->loop->stop();
                $this->logger->debug('Event Loop detenido');
            }

            $this->logger->info('Bot cerrado correctamente', [
                'instance' => $this->instanceName ?? 'N/A'
            ]);
        } catch (\Throwable $e) {
            $this->logger->error('Error durante el cierre del bot', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
        }
    }

    /**
     * Registra manejadores de señales para un cierre elegante.
     */
    private function registerSignalHandlers(): void
    {
        // No implementado directamente aquí en ReactPHP para SIGTERM/SIGINT,
        // ya que ReactPHP maneja su propio loop de eventos.
        // Se puede añadir con pcntl_signal si se desea un manejo más granular
        // en un entorno CLI, pero la detención del loop es la forma común.

        $this->logger->debug('Manejadores de señales registrados (para cierre controlado).');
    }

    /**
     * Ejecuta el bot (punto de entrada principal)
     */
    public function run(): void
    {
        $this->registerSignalHandlers();

        $this->logger->info('Iniciando ejecución del bot', [
            'instance' => $this->instanceName,
            'algorithm' => $this->algorithm,
            'symbol' => $this->symbol
        ]);

        $this->start()
            ->then(function () {
                $this->logger->info('Bot iniciado exitosamente, entrando en loop principal');
                $this->loop->run(); // Iniciar el loop de ReactPHP
            })
            ->otherwise(function (\Throwable $e) {
                $this->handleCriticalError($e);
            });
    }
}

// ============================================================================
// PUNTO DE ENTRADA PRINCIPAL
// ============================================================================
/*
try {
    $tradingBot = new TradingBot();
    $tradingBot->run();
} catch (\Throwable $e) {
    $errorMessage = "[FATAL] Error crítico al inicializar el bot: " . $e->getMessage();
    error_log($errorMessage);
    echo $errorMessage . "\n";
    exit(1);
}
*/