<?php

namespace App;

use App\Trader; // Asegúrate de que la clase Trader está definida y accesible
use App\SmartMoneyVolume; // Asegúrate de que la clase SmartMoneyVolume está definida y accesible

/**
 * Estrategia de Trading basada en Fusión de Volumen y Smart Money Concepts.
 *
 * Esta estrategia combina análisis de volumen avanzado con filtros de tendencia
 * y gestión dinámica de riesgo para identificar y ejecutar operaciones de alta probabilidad.
 *
 * Características clave:
 * - Detección de señales basada en patrones de volumen de Smart Money (absorción, acumulación/distribución, spikes).
 * - Operaciones solo en dirección de la tendencia mayor (filtradas con SMA).
 * - Niveles dinámicos de Take Profit (TP) y Stop Loss (SL), y Trailing Stop basados en ATR.
 * - Ajuste automático del tamaño de posición según la confianza de la señal.
 * - Evita operar en horas de baja liquidez configurables.
 * - Parámetros configurables para ajustar la sensibilidad de la estrategia.
 */
class SmartMoneyFusionStrategy
{
    /**
     * @var SmartMoneyVolume Instancia del analizador de volumen.
     */
    private SmartMoneyVolume $volumeAnalyzer;

    /**
     * @var Trader Motor de trading para ejecutar órdenes y obtener el saldo de la cuenta.
     */
    private Trader $trader;

    // Parámetros de Configuración (pueden hacerse públicos o pasarse al constructor)

    /**
     * @var float Porcentaje del capital a arriesgar por operación en el nivel base de confianza (1.0 = 1%).
     */
    private float $baseRiskPercent;

    /**
     * @var array Horas permitidas para operar en formato 24h (e.g., [9, 10, ..., 17]). Evita baja liquidez.
     */
    private array $tradingHours;

    /**
     * @var int Período para el cálculo de la Media Móvil Simple (SMA) usada como filtro de tendencia.
     */
    private int $smaPeriod;

    /**
     * @var float Multiplicador del volumen promedio (en filtros de entrada) para considerar que el volumen actual es significativo.
     */
    private float $volumeFilterMultiplier;

    /**
     * @var int Lookback para determinar la tendencia del volumen (en filtros de entrada y señales).
     */
    private int $volumeTrendLookback;

    /**
     * @var float Porcentaje de proximidad a un nivel de liquidez/perfil de volumen para considerarlo relevante para la señal/filtro.
     */
    private float $liquidityProximityPercentage;

    /**
     * @var int Período para el cálculo del Average True Range (ATR).
     */
    private int $atrPeriod;

    /**
     * @var float Multiplicador del ATR para calcular el Stop Loss inicial.
     */
    private float $slMultiplier;

    /**
     * @var float Multiplicador del ATR para calcular el Take Profit inicial.
     */
    private float $tpMultiplier;

    /**
     * @var float Multiplicador del ATR para calcular el Trailing Stop inicial.
     */
    private float $tsMultiplier;

     /**
      * @var int Umbral mínimo de confianza (0-100) para ejecutar una operación.
      */
     private int $minConfidenceToExecute;

    /**
     * Constructor de la estrategia.
     *
     * @param SmartMoneyVolume $volumeAnalyzer Analizador de volumen con datos cargados.
     * @param Trader $trader Motor de trading para ejecutar órdenes.
     * @param array $config Array opcional de configuración para sobrescribir parámetros por defecto.
     * Keys reconocidas: 'baseRiskPercent', 'tradingHours', 'smaPeriod',
     * 'volumeFilterMultiplier', 'volumeTrendLookback', 'liquidityProximityPercentage',
     * 'atrPeriod', 'slMultiplier', 'tpMultiplier', 'tsMultiplier', 'minConfidenceToExecute'.
     */
    public function __construct(SmartMoneyVolume $volumeAnalyzer, Trader $trader, array $config = [])
    {
        $this->volumeAnalyzer = $volumeAnalyzer;
        $this->trader = $trader;

        // Configuración por defecto y sobreescritura con $config
        $defaultConfig = [
            'baseRiskPercent' => 1.0, // 1% base riesgo
            'tradingHours' => [9, 10, 11, 12, 13, 14, 15, 16, 17], // Horario de mercado típico (ejemplo)
            'smaPeriod' => 20,
            'volumeFilterMultiplier' => 1.5, // Requiere 1.5x el volumen promedio
            'volumeTrendLookback' => 7, // Tendencia de volumen en los últimos 7 periodos
            'liquidityProximityPercentage' => 0.2, // Precio debe estar a 0.2% de un nivel clave
            'atrPeriod' => 14,
            'slMultiplier' => 1.5, // SL a 1.5 * ATR
            'tpMultiplier' => 3.0, // TP a 3.0 * ATR
            'tsMultiplier' => 1.2, // Trailing Stop a 1.2 * ATR
            'minConfidenceToExecute' => 60, // Requiere al menos 60 de confianza
        ];

        $config = array_merge($defaultConfig, $config);

        $this->baseRiskPercent = $config['baseRiskPercent'];
        $this->tradingHours = $config['tradingHours'];
        $this->smaPeriod = $config['smaPeriod'];
        $this->volumeFilterMultiplier = $config['volumeFilterMultiplier'];
        $this->volumeTrendLookback = $config['volumeTrendLookback'];
        $this->liquidityProximityPercentage = $config['liquidityProximityPercentage'];
        $this->atrPeriod = $config['atrPeriod'];
        $this->slMultiplier = $config['slMultiplier'];
        $this->tpMultiplier = $config['tpMultiplier'];
        $this->tsMultiplier = $config['tsMultiplier'];
        $this->minConfidenceToExecute = $config['minConfidenceToExecute'];
    }

    //region Evaluación y Ejecución

    /**
     * Evalúa si se debe entrar en una operación basándose en los datos actuales
     * y los criterios de la estrategia.
     *
     * @return array|null Información detallada de la señal (dirección, entrada, SL, TP, tamaño, confianza)
     * o null si no hay señal válida.
     */
    public function evaluate(): ?array
    {
        // 1. Verificar hora y filtros generales de entrada
        if (!$this->isTradingHour()) {
            // Log::info("Fuera del horario de trading."); // Ejemplo de log
            return null;
        }

        // Calculamos valores comunes una vez
        $currentPrice = end($this->volumeAnalyzer->prices);
        if ($currentPrice === false) { // Asegurarse de que hay precios
             // Log::error("No hay datos de precio disponibles.");
             return null;
        }

         try {
             $atr = $this->volumeAnalyzer->getATR($this->atrPeriod);
         } catch (\InvalidArgumentException $e) {
             // Log::error("Error calculando ATR: " . $e->getMessage());
             return null; // No podemos operar sin ATR válido
         }

        if ($atr <= 0) {
             // Log::info("ATR es cero o negativo, volatilidad insuficiente o datos inválidos.");
             return null; // No operar con volatilidad cero
        }

        // Pasar ATR y precio actual a los filtros para consistencia
        if (!$this->passEntryFilters($currentPrice, $atr)) {
            // Log::info("Filtros de entrada no pasados.");
            return null;
        }

        // 2. Determinar tendencia principal
        $trend = $this->getTrendDirection();
        if ($trend === 'neutral') {
            // Log::info("Tendencia neutral, no se opera.");
            return null;
        }

        // 3. Detectar señales específicas (requiere suficientes datos en volumeAnalyzer)
        try {
            $buySignal = $this->isBuySignal($currentPrice, $atr);
            $sellSignal = $this->isSellSignal($currentPrice, $atr);
        } catch (\InvalidArgumentException $e) {
            // Log::error("Error detectando señales: " . $e->getMessage());
            return null; // No podemos continuar sin señales válidas
        }


        // 4. Solo operar a favor de la tendencia principal
        if ($buySignal && $trend !== 'long') {
            // Log::info("Señal de compra en tendencia bajista/neutral, se ignora.");
            return null;
        }
        if ($sellSignal && $trend !== 'short') {
            // Log::info("Señal de venta en tendencia alcista/neutral, se ignora.");
            return null;
        }

        // 5. Si hay una señal válida a favor de la tendencia
        if ($buySignal || $sellSignal) {
             // Log::info(($buySignal ? "Compra" : "Venta") . " detectada.");

            // 6. Calcular niveles y tamaño de posición
            $confidence = $this->calculateConfidence();
            $adjustedRisk = $this->adjustRiskByConfidence($confidence);

             // Calcular niveles de TP/SL/TS dinámicamente
             $levels = $this->calculateTradeLevels($currentPrice, $atr, $buySignal ? 'buy' : 'sell');
             $stopLoss = $levels['stop_loss'];
             $takeProfit = $levels['take_profit'];
             $trailingStop = $levels['trailing_stop'];

            // Asegurarse de que el stop loss es válido (no igual a la entrada, evita división por cero)
            if (abs($currentPrice - $stopLoss) < 1e-9) {
                 // Log::warning("Stop loss demasiado cerca o igual al precio de entrada. Señal ignorada.");
                 return null;
            }

            try {
                $positionSize = $this->calculatePositionSize($currentPrice, $stopLoss, $adjustedRisk);
            } catch (\RuntimeException $e) {
                // Log::error("Error calculando tamaño de posición: " . $e->getMessage());
                 return null; // No podemos operar si no podemos calcular el tamaño
            }


            return [
                'signal' => $buySignal ? 'buy' : 'sell',
                'entry' => $currentPrice,
                'stop_loss' => $stopLoss,
                'take_profit' => $takeProfit,
                'trailing_stop' => $trailingStop,
                'position_size' => $positionSize,
                'confidence' => $confidence,
                'timestamp' => date('Y-m-d H:i:s'), // Usar timestamp real si está disponible en los datos
            ];
        }

        // Log::info("No se detectó ninguna señal válida.");
        return null; // No hay señal de entrada
    }

    /**
     * Ejecuta la operación si hay una señal válida y la confianza es suficiente.
     *
     * @return array|null Información de la señal y el resultado de la orden si se ejecutó,
     * o null si no se cumplieron los criterios de ejecución.
     */
    public function execute(): ?array
    {
        $signal = $this->evaluate();

        if ($signal && $signal['confidence'] >= $this->minConfidenceToExecute) {
            // Log::info("Confianza suficiente (" . $signal['confidence'] . ") para ejecutar " . $signal['signal'] . ".");
            try {
                 $order = $this->trader->placeOrder(
                     $signal['signal'],
                     $signal['entry'],
                     $signal['position_size'],
                     $signal['stop_loss'],
                     $signal['take_profit']
                     // Considerar pasar trailing_stop si el motor de trading lo soporta directamente al abrir
                 );
                 // Log::info("Orden colocada: ", $order);
                 return array_merge($signal, ['order' => $order]);

            } catch (\Exception $e) {
                 // Capturar excepciones específicas del motor de trading si es posible
                 // Log::error("Error al colocar la orden: " . $e->getMessage());
                 // Podríamos decidir si reintentar, notificar, etc.
                 return null; // La orden no se pudo colocar
            }
        }

        // Log::info("Señal detectada pero confianza (" . ($signal['confidence'] ?? 'N/A') . ") insuficiente (Min: " . $this->minConfidenceToExecute . ").");
        return null; // No se cumplen los criterios de ejecución
    }

    //endregion

    //region Filtros de Entrada

    /**
     * Aplica filtros generales de entrada que deben cumplirse para cualquier operación.
     *
     * @param float $currentPrice El precio actual.
     * @param float $atr El Average True Range actual.
     * @return bool True si se pasan todos los filtros, false en caso contrario.
     * @throws \InvalidArgumentException Si no hay suficientes datos en SmartMoneyVolume para los cálculos.
     */
    private function passEntryFilters(float $currentPrice, float $atr): bool
    {
        // 1. Volumen superior a un promedio reciente
        try {
            $avgVolume = $this->volumeAnalyzer->getAverageVolume($this->volumeAnalyzer->defaultVolumeLookback); // Usar lookback por defecto de SmartMoneyVolume
            $currentVolume = end($this->volumeAnalyzer->volumes);

             if ($currentVolume === false || $avgVolume <= 0) {
                  // Log::debug("Filtro de volumen fallido: Datos insuficientes o volumen promedio <= 0.");
                  return false; // Datos insuficientes o volumen promedio no válido
             }

            if ($currentVolume < $avgVolume * $this->volumeFilterMultiplier) {
                // Log::debug("Filtro de volumen fallido: Volumen actual (" . $currentVolume . ") < promedio * " . $this->volumeFilterMultiplier . " (" . ($avgVolume * $this->volumeFilterMultiplier) . ")");
                return false;
            }
        } catch (\InvalidArgumentException $e) {
             // Log::error("Error en filtro de volumen: " . $e->getMessage());
             return false; // Error de datos
        }


        // 2. Precio actual cerca de un nivel de liquidez (perfil de volumen significativo)
        try {
            $volumeProfile = $this->volumeAnalyzer->getVolumeProfile(); // Usa top 5 niveles por defecto
            $isNearLiquidityLevel = false;

            // Verificar si el precio actual está dentro de un pequeño porcentaje de cualquiera de los niveles clave
            if ($currentPrice > 0) { // Evitar división por cero
                 foreach ($volumeProfile as $levelPrice) {
                     $percentageDifference = abs($currentPrice - $levelPrice) / $currentPrice * 100;
                     if ($percentageDifference <= $this->liquidityProximityPercentage) {
                         $isNearLiquidityLevel = true;
                         break;
                     }
                 }
            }


            if (!$isNearLiquidityLevel) {
                // Log::debug("Filtro de liquidez fallido: Precio actual (" . $currentPrice . ") no está cerca (" . $this->liquidityProximityPercentage . "%) de los niveles clave (" . implode(', ', $volumeProfile) . ").");
                return false;
            }
        } catch (\Exception $e) { // Capturar posibles errores de getVolumeProfile
             // Log::error("Error en filtro de liquidez: " . $e->getMessage());
             return false;
        }


        // 3. Tendencia de volumen alcista (generalmente deseable para cualquier movimiento fuerte)
        // Nota: Esto filtra tanto compras como ventas. Si solo quieres volumen alcista para compras
        // y bajista para ventas, mueve este filtro a isBuySignal/isSellSignal.
        // Basado en la descripción original "Tendencia de volumen alcista", lo mantenemos aquí,
        // pero solo permitiremos 'rising'. Si 'neutral' también es válido, cambiar la condición.
        try {
             $volumeTrend = $this->volumeAnalyzer->getVolumeTrend($this->volumeTrendLookback);
             if ($volumeTrend !== 'rising') {
                 // Log::debug("Filtro de tendencia de volumen fallido: Tendencia es '" . $volumeTrend . "' (se requiere 'rising').");
                 return false;
             }
        } catch (\InvalidArgumentException $e) {
             // Log::error("Error en filtro de tendencia de volumen: " . $e->getMessage());
             return false; // Error de datos
        }


        // Si pasa todos los filtros
        // Log::debug("Todos los filtros de entrada pasados.");
        return true;
    }

    //endregion

    //region Señales de Compra y Venta

    /**
     * Detecta una señal de compra basándose en patrones de volumen/precio y niveles clave.
     *
     * @param float $currentPrice El precio actual.
     * @param float $atr El Average True Range actual.
     * @return bool True si se detecta una señal de compra, false en caso contrario.
     * @throws \InvalidArgumentException Si no hay suficientes datos en SmartMoneyVolume para los cálculos.
     */
    private function isBuySignal(float $currentPrice, float $atr): bool
    {
        // Criterios para señal de compra (usando métodos de SmartMoneyVolume)
        try {
            // Requerir Absorción (puede indicar que la venta está siendo absorbida)
            if (!$this->volumeAnalyzer->detectAbsorption()) {
                // Log::debug("Señal de compra: No hay absorción.");
                return false;
            }

            // Requerir que el último spike de volumen significativo haya sido alcista
            if ($this->volumeAnalyzer->getLastSpikeDirection() !== 'up') {
                // Log::debug("Señal de compra: Último spike no fue alcista.");
                return false;
            }

            // Requerir que el mercado esté en fase de Acumulación (usando múltiples criterios internos de SmartMoneyVolume)
            // Pasar parámetros de configuración para isAccumulation para consistencia
             if (!$this->volumeAnalyzer->isAccumulation(
                 5, // topN para VolumeProfile (defecto)
                 5, // volumeTrendLookback (defecto)
                 $this->liquidityProximityPercentage, // porcentaje de proximidad
                 $this->atrPeriod // periodo ATR
             )) {
                // Log::debug("Señal de compra: No hay acumulación.");
                return false;
            }

            // Criterio adicional: Precio actual está cerca de un nivel de liquidez detectado
            // calculateLiquidityLevels puede ser costoso, tal vez sólo comprobar si el precio
            // está cerca de CUALQUIER nivel de perfil de volumen ya es suficiente (como en passEntryFilters).
            // Mantendremos la lógica original de verificar cerca de niveles *detectados específicamente como liquidez*
            // pero siendo conscientes del costo.
             $liquidityLevels = $this->volumeAnalyzer->calculateLiquidityLevels(
                 $this->volumeAnalyzer->liquidityPriceThresholdATR, // usar defecto de SmartMoneyVolume
                 $this->volumeAnalyzer->liquidityVolumeRatioThreshold, // usar defecto de SmartMoneyVolume
                 $this->atrPeriod // periodo ATR
             );

            $isNearDetectedLiquidity = false;
             if ($currentPrice > 0) {
                 foreach ($liquidityLevels as $level) {
                      // Verificar si el precio actual está dentro de un pequeño porcentaje del nivel de liquidez
                      $percentageDifference = abs($currentPrice - $level['price']) / $currentPrice * 100;
                      if ($percentageDifference <= $this->liquidityProximityPercentage) {
                          $isNearDetectedLiquidity = true;
                          break;
                      }
                 }
             }


            if (!$isNearDetectedLiquidity) {
                // Log::debug("Señal de compra: Precio actual (" . $currentPrice . ") no está cerca (" . $this->liquidityProximityPercentage . "%) de niveles de liquidez detectados.");
                return false;
            }

        } catch (\InvalidArgumentException $e) {
             // Log::error("Error en detección de señal de compra: " . $e->getMessage());
             throw $e; // Relanzar para ser manejado en evaluate()
        }


        // Si todos los criterios se cumplen
        return true;
    }

    /**
     * Detecta una señal de venta basándose en patrones de volumen/precio y niveles clave.
     *
     * @param float $currentPrice El precio actual.
     * @param float $atr El Average True Range actual.
     * @return bool True si se detecta una señal de venta, false en caso contrario.
     * @throws \InvalidArgumentException Si no hay suficientes datos en SmartMoneyVolume para los cálculos.
     */
    private function isSellSignal(float $currentPrice, float $atr): bool
    {
        // Criterios para señal de venta (usando métodos de SmartMoneyVolume)
        try {
             // Requerir Absorción (puede indicar que la compra está siendo absorbida por vendedores)
            if (!$this->volumeAnalyzer->detectAbsorption()) {
                // Log::debug("Señal de venta: No hay absorción.");
                return false;
            }

            // Requerir que el último spike de volumen significativo haya sido bajista
            if ($this->volumeAnalyzer->getLastSpikeDirection() !== 'down') {
                // Log::debug("Señal de venta: Último spike no fue bajista.");
                return false;
            }

            // Requerir que el mercado esté en fase de Distribución (usando múltiples criterios internos de SmartMoneyVolume)
             // Pasar parámetros de configuración para isDistribution para consistencia
             if (!$this->volumeAnalyzer->isDistribution(
                 5, // topN para VolumeProfile (defecto)
                 5, // volumeTrendLookback (defecto)
                 $this->liquidityProximityPercentage, // porcentaje de proximidad
                 $this->atrPeriod // periodo ATR
             )) {
                // Log::debug("Señal de venta: No hay distribución.");
                return false;
            }

            // Criterio adicional: Precio actual está cerca de un nivel de liquidez detectado
             $liquidityLevels = $this->volumeAnalyzer->calculateLiquidityLevels(
                 $this->volumeAnalyzer->liquidityPriceThresholdATR, // usar defecto de SmartMoneyVolume
                 $this->volumeAnalyzer->liquidityVolumeRatioThreshold, // usar defecto de SmartMoneyVolume
                 $this->atrPeriod // periodo ATR
             );

            $isNearDetectedLiquidity = false;
             if ($currentPrice > 0) {
                 foreach ($liquidityLevels as $level) {
                      // Verificar si el precio actual está dentro de un pequeño porcentaje del nivel de liquidez
                      $percentageDifference = abs($currentPrice - $level['price']) / $currentPrice * 100;
                      if ($percentageDifference <= $this->liquidityProximityPercentage) {
                          $isNearDetectedLiquidity = true;
                          break;
                      }
                 }
             }

            if (!$isNearDetectedLiquidity) {
                 // Log::debug("Señal de venta: Precio actual (" . $currentPrice . ") no está cerca (" . $this->liquidityProximityPercentage . "%) de niveles de liquidez detectados.");
                 return false;
            }

        } catch (\InvalidArgumentException $e) {
             // Log::error("Error en detección de señal de venta: " . $e->getMessage());
             throw $e; // Relanzar para ser manejado en evaluate()
        }


        // Si todos los criterios se cumplen
        return true;
    }

    //endregion

    //region Análisis Técnico

    /**
     * Calcula la Media Móvil Simple (SMA) para un periodo dado.
     *
     * @param int $period El número de periodos para el cálculo.
     * @return float|null El valor de la SMA, o null si no hay suficientes datos.
     * @throws \InvalidArgumentException Si el período es inválido.
     */
    private function getSMA(int $period): ?float
    {
        if ($period <= 0) {
             throw new \InvalidArgumentException("El período de la SMA debe ser un número positivo.");
        }
        $prices = $this->volumeAnalyzer->prices;
        if ($period > count($prices)) {
            return null; // No hay suficientes datos para calcular la SMA
        }

        $recentPrices = array_slice($prices, -$period);
        return array_sum($recentPrices) / $period;
    }

    /**
     * Determina la dirección de la tendencia principal utilizando una Media Móvil Simple (SMA).
     * La tendencia es alcista ('long') si el precio actual está por encima de la SMA,
     * bajista ('short') si está por debajo, y neutral si no hay suficientes datos o está sobre/cerca de la SMA.
     *
     * @return string 'long', 'short', o 'neutral'.
     */
    private function getTrendDirection(): string
    {
        try {
            $sma = $this->getSMA($this->smaPeriod);
        } catch (\InvalidArgumentException $e) {
            // Log::error("Error calculando SMA para tendencia: " . $e->getMessage());
            return 'neutral'; // No se puede determinar la tendencia sin SMA válida
        }


        if ($sma === null) {
            return 'neutral'; // No hay suficientes datos para la SMA
        }

        $price = end($this->volumeAnalyzer->prices);
         if ($price === false) {
             return 'neutral'; // No hay datos de precio
         }

        // Se podría añadir una pequeña tolerancia alrededor de la SMA si se desea un estado 'neutral'
        // más amplio cuando el precio está muy cerca de la SMA.
        if ($price > $sma) {
            return 'long';
        }
        if ($price < $sma) {
            return 'short';
        }

        return 'neutral';
    }

    //endregion

    //region Gestión de Riesgo y Posición

    /**
     * Calcula los niveles de Stop Loss (SL), Take Profit (TP) y Trailing Stop (TS)
     * basados en el precio de entrada y el ATR actual.
     *
     * @param float $entryPrice El precio al que se entrará en la operación.
     * @param float $atr El valor actual del Average True Range.
     * @param string $direction La dirección de la operación ('buy' o 'sell').
     * @return array Un array asociativo con 'stop_loss', 'take_profit', y 'trailing_stop'.
     * @throws \InvalidArgumentException Si la dirección no es 'buy' ni 'sell'.
     */
    private function calculateTradeLevels(float $entryPrice, float $atr, string $direction): array
    {
        if (!in_array($direction, ['buy', 'sell'])) {
             throw new \InvalidArgumentException("Dirección de operación inválida: " . $direction);
        }

        $slDistance = $atr * $this->slMultiplier;
        $tpDistance = $atr * $this->tpMultiplier;
        $tsDistance = $atr * $this->tsMultiplier; // Distancia para el trailing stop

        if ($direction === 'buy') {
            $stopLoss = $entryPrice - $slDistance;
            $takeProfit = $entryPrice + $tpDistance;
             // El trailing stop para una compra sigue el precio hacia arriba
             // Se inicializa a la misma distancia que el SL inicial desde la entrada
             $trailingStop = $entryPrice - $tsDistance; // Distancia *desde* el precio actual/de entrada
        } else { // sell
            $stopLoss = $entryPrice + $slDistance;
            $takeProfit = $entryPrice - $tpDistance;
             // El trailing stop para una venta sigue el precio hacia abajo
             // Se inicializa a la misma distancia que el SL inicial desde la entrada
             $trailingStop = $entryPrice + $tsDistance; // Distancia *desde* el precio actual/de entrada
        }

        // Asegurarse de que TP/SL tienen sentido básico (TP > SL para buy, TP < SL para sell)
        // y que SL no cruza la entrada inmediatamente (aunque abs($entry - SL) check en evaluate ya lo cubre)
         if ($direction === 'buy' && ($stopLoss >= $entryPrice || $takeProfit <= $entryPrice)) {
              // Esto debería ser manejado por el cálculo de ATR y multiplicadores, pero es una verificación de seguridad
              // Log::warning("Niveles de compra inválidos calculados: SL=" . $stopLoss . ", TP=" . $takeProfit . ", Entry=" . $entryPrice);
              // Dependiendo de la lógica deseada, podrías lanzar una excepción o ajustar los niveles.
              // Aquí, asumimos que los multiplicadores y ATR son razonables y procedemos.
         }
          if ($direction === 'sell' && ($stopLoss <= $entryPrice || $takeProfit >= $entryPrice)) {
               // Lo mismo para ventas
              // Log::warning("Niveles de venta inválidos calculados: SL=" . $stopLoss . ", TP=" . $takeProfit . ", Entry=" . $entryPrice);
          }


        return [
            'stop_loss' => $stopLoss,
            'take_profit' => $takeProfit,
            'trailing_stop' => $trailingStop,
        ];
    }


    /**
     * Calcula una puntuación de confianza para la señal detectada (0-100).
     * La confianza se basa en la fuerza del volumen y la volatilidad.
     *
     * @return int La puntuación de confianza calculada.
     * @throws \InvalidArgumentException Si no hay suficientes datos en SmartMoneyVolume para los cálculos.
     */
    private function calculateConfidence(): int
    {
        $confidence = 50; // Punto de partida neutral

        try {
             // Factores que aumentan la confianza:
             // Volumen muy alto comparado con el promedio reciente
             $avgVolume = $this->volumeAnalyzer->getAverageVolume($this->volumeAnalyzer->defaultVolumeLookback);
             $currentVolume = end($this->volumeAnalyzer->volumes);

             if ($currentVolume !== false && $avgVolume > 0) {
                  // Si el volumen actual es más de 3 veces el promedio, gran impulso
                  if ($currentVolume > $avgVolume * 3) {
                      $confidence += 20;
                  }
                  // Si detectamos un cluster de volumen (spike significativo)
                  if ($this->volumeAnalyzer->detectVolumeCluster()) { // Usa los defectos de SmartMoneyVolume
                      $confidence += 15;
                  }
             }


             // Factores que disminuyen la confianza:
             // Volatilidad (medida por ATR) demasiado baja en relación con el precio.
             // Una volatilidad muy baja podría indicar falta de interés o consolidación sin dirección clara.
             $atr = $this->volumeAnalyzer->getATR($this->atrPeriod);
             $currentPrice = end($this->volumeAnalyzer->prices);

             if ($currentPrice !== false && $atr > 0) {
                  // Umbral: ATR es menor que 0.2% del precio actual (configurable, 0.002 como fracción)
                  $atrPercentageOfPrice = ($atr / $currentPrice);
                  if ($atrPercentageOfPrice < 0.002) { // 0.2%
                      $confidence -= 20;
                  }
             } else if ($atr <= 0) {
                 // Si el ATR es cero o negativo, la volatilidad es nula o inválida, lo que disminuye la confianza.
                 $confidence -= 30; // Penalización mayor
             }


        } catch (\InvalidArgumentException $e) {
             // Log::error("Error en cálculo de confianza: " . $e->getMessage());
             // Un error en el cálculo de un factor debería penalizar la confianza, no detener todo.
             // Podríamos restar puntos y continuar, o propagar la excepción si es crítica.
             // Restemos un valor fijo y registremos el error.
             $confidence -= 25; // Penalización por error en el cálculo
             // throw $e; // Si decides que un error aquí es fatal
        } catch (\Exception $e) {
             // Log::error("Error inesperado en cálculo de confianza: " . $e->getMessage());
             $confidence -= 25;
        }


        // Asegurarse de que la confianza esté entre 0 y 100
        return max(0, min(100, $confidence));
    }

    /**
     * Ajusta el porcentaje de riesgo base por operación según la puntuación de confianza de la señal.
     * Una mayor confianza resulta en un mayor porcentaje de riesgo (hasta el riesgo base).
     *
     * @param int $confidence La puntuación de confianza (0-100).
     * @return float El porcentaje de riesgo ajustado (e.g., 0.5 para 0.5%).
     */
    private function adjustRiskByConfidence(int $confidence): float
    {
        // Escala lineal: 0 confianza -> 0% riesgo ajustado, 100 confianza -> baseRiskPercent riesgo ajustado.
        // Asegurarse de que la confianza esté dentro del rango esperado
        $clampedConfidence = max(0, min(100, $confidence));
        return $this->baseRiskPercent * ($clampedConfidence / 100.0);
    }

    /**
     * Calcula el tamaño de la posición (en unidades del activo base) basándose en
     * el riesgo ajustado por operación, el saldo de la cuenta y la distancia al Stop Loss.
     *
     * Tamaño = (Saldo Total * Porcentaje de Riesgo) / (Distancia al Stop Loss * Valor por unidad)
     * Asumimos que el valor por unidad es 1 si se opera el activo base directamente (e.g., cantidad de BTC en BTC/USD).
     * Si se opera con margen o derivados, el cálculo podría ser diferente (e.g., contratos).
     * Aquí, se calcula la cantidad de activo base que puedes comprar/vender.
     *
     * @param float $entry El precio de entrada.
     * @param float $stopLoss El precio del Stop Loss.
     * @param float $adjustedRisk El porcentaje de riesgo ajustado (e.g., 0.01 para 1%).
     * @return float El tamaño de la posición (cantidad de activo base).
     * @throws \RuntimeException Si el saldo de la cuenta es cero o negativo, o la distancia al SL es cero.
     */
    private function calculatePositionSize(float $entry, float $stopLoss, float $adjustedRisk): float
    {
        try {
             $accountBalance = $this->trader->getBalance();
        } catch (\Exception $e) {
             // Log::error("Error obteniendo saldo de la cuenta: " . $e->getMessage());
             throw new \RuntimeException("No se pudo obtener el saldo de la cuenta.", 0, $e);
        }


        if ($accountBalance <= 0) {
            // Log::error("Saldo de la cuenta cero o negativo: " . $accountBalance);
            throw new \RuntimeException("Saldo de la cuenta cero o negativo.");
        }

        // Cantidad de dinero a arriesgar en esta operación
        $riskAmount = $accountBalance * ($adjustedRisk / 100.0);

        // Distancia en precio desde la entrada al Stop Loss
        $riskPerUnit = abs($entry - $stopLoss);

        if ($riskPerUnit <= 1e-9) { // Usar una pequeña tolerancia para comparar con cero
            // Log::error("Distancia al Stop Loss es cero o insignificante: " . $riskPerUnit);
             // Esto también podría indicar un problema con los niveles de TP/SL calculados o datos inválidos.
            throw new \RuntimeException("Distancia al Stop Loss es cero, no se puede calcular el tamaño de la posición.");
        }

        // Tamaño de la posición = Cantidad a arriesgar / Riesgo por unidad
        // Asegurarse de que el tamaño de la posición sea positivo
        return max(0.0, $riskAmount / $riskPerUnit);
    }

    /**
     * Verifica si la hora actual está dentro de las horas de trading permitidas.
     *
     * @return bool True si es hora de operar, false en caso contrario.
     */
    private function isTradingHour(): bool
    {
        $hour = (int) date('G'); // Hora actual en formato 24h (0-23)
        return in_array($hour, $this->tradingHours);
    }

    //endregion

     //region Setters (opcional, si quieres cambiar la config después de la construcción)
     // public function setBaseRiskPercent(float $percent): void { $this->baseRiskPercent = $percent; }
     // public function setTradingHours(array $hours): void { $this->tradingHours = $hours; }
     // ...etc.
     //endregion
}