<?php

namespace App; // Asegúrate que el namespace sea correcto

use App\Trader; // Usado implícitamente si se reemplaza getATR o para SMA en umbrales estáticos
use InvalidArgumentException;
use RuntimeException; // Podríamos usarlo para errores internos

/**
 * Analiza la relación entre el precio y el volumen para detectar patrones significativos.
 *
 * Refactorizada para calcular y devolver SERIES de resultados para análisis barra a barra.
 */
class SmartMoneyVolumeSeries
{
    //region Propiedades de Configuración e Input
    private array $data; // Contiene ['high', 'low', 'close', 'volume']
    private int $volumeLookback = 20;
    private float $priceSensitivity = 0.15;
    private int $atrPeriod = 14;
    private int $volumeClusterLookback = 5; // Lookback específico para cluster
    private int $volumeTrendLookback = 14; // Lookback específico para tendencia volumen
    //endregion

    //region Propiedades para Series Calculadas (Base)
    private ?array $atrSeries = null;
    private ?array $avgVolumeSeries = null; // Promedio móvil simple del volumen
    private ?array $priceChangeSeries = null; // Cambio absoluto Cierre[i] - Cierre[i-1]
    private ?array $volumeChangeRatioSeries = null; // Ratio Volumen[i] / Volumen[i-1]
    //endregion

    //region Propiedades para Series de Análisis
    private ?array $volumeClusterSeries = null;
    private ?array $absorptionSeries = null;
    private ?array $anomalySpikeSeries = null; // Serie booleana de spikes
    private ?array $lastSpikeDirectionSeries = null; // Serie 'up'/'down'/null
    private ?array $volumeTrendSeries = null; // Serie 'rising'/'falling'/'neutral'
    private ?array $liquidityLevelSeries = null; // Serie booleana
    private ?array $isAccumulationSeries = null;
    private ?array $isDistributionSeries = null;
    //endregion

    private bool $seriesCalculated = false;
    private array $keys = []; // Almacena las claves originales (timestamps/índices)

    /**
     * Constructor.
     *
     * @param array $data Array asociativo con 'high', 'low', 'close', 'volume'.
     * @throws InvalidArgumentException Si los datos son inválidos.
     */
    public function __construct(array $data)
    {
        if (!isset($data['high'], $data['low'], $data['close'], $data['volume']) ||
            empty($data['close']) || empty($data['volume']) ||
            count($data['close']) !== count($data['volume']) ||
            count($data['close']) !== count($data['high']) ||
            count($data['close']) !== count($data['low'])) {
            throw new InvalidArgumentException("Los arrays de datos HLCV deben existir, contener datos y tener la misma longitud.");
        }

        // Validar que los datos internos sean numéricos o null
        foreach(['high', 'low', 'close', 'volume'] as $key) {
             $data[$key] = array_map(function($v) { return is_numeric($v) ? (float)$v : null; }, $data[$key]);
        }

        $this->data = $data;
        $this->keys = array_keys($this->data['close']); // Guardar claves para alinear resultados
    }

    /**
     * Método principal para calcular todas las series analizables.
     * Debe llamarse antes de usar los getters de series.
     */
    public function calculateAllSeries(): void
    {
        if ($this->seriesCalculated) {
            return; // Ya calculado
        }

        $this->calculateBaseSeries(); // Calcular ATR, AvgVol, etc.

        if ($this->atrSeries === null || $this->avgVolumeSeries === null) {
             // Fallo en cálculos base, no se puede continuar
             // Opcional: throw new RuntimeException("Fallo al calcular series base (ATR/AvgVol).");
             return;
        }

        // Calcular series de análisis que dependen de las base
        $this->volumeClusterSeries = $this->calculateVolumeClusterSeries($this->volumeClusterLookback);
        $this->absorptionSeries = $this->calculateAbsorptionSeries();
        $this->anomalySpikeSeries = $this->calculateAnomalySpikeSeries();
        $this->lastSpikeDirectionSeries = $this->calculateLastSpikeDirectionSeries($this->anomalySpikeSeries);
        $this->volumeTrendSeries = $this->calculateVolumeTrendSeries($this->volumeTrendLookback);
        $this->liquidityLevelSeries = $this->calculateLiquidityLevelSeries();

        // Calcular A/D (dependen de otras series)
        $this->isAccumulationSeries = $this->calculateIsAccumulationSeries();
        $this->isDistributionSeries = $this->calculateIsDistributionSeries();

        $this->seriesCalculated = true;
    }

    //region Getters para Series Calculadas
    public function getAtrSeries(): ?array { return $this->atrSeries; }
    public function getAvgVolumeSeries(): ?array { return $this->avgVolumeSeries; }
    public function getVolumeClusterSeries(): ?array { return $this->volumeClusterSeries; }
    public function getAbsorptionSeries(): ?array { return $this->absorptionSeries; }
    public function getAnomalySpikeSeries(): ?array { return $this->anomalySpikeSeries; }
    public function getLastSpikeDirectionSeries(): ?array { return $this->lastSpikeDirectionSeries; }
    public function getVolumeTrendSeries(): ?array { return $this->volumeTrendSeries; }
    public function getLiquidityLevelSeries(): ?array { return $this->liquidityLevelSeries; }
    public function getIsAccumulationSeries(): ?array { return $this->isAccumulationSeries; }
    public function getIsDistributionSeries(): ?array { return $this->isDistributionSeries; }
    //endregion

    //region Cálculo de Series Base
    /** Calcula las series base necesarias para otros análisis. */
    private function calculateBaseSeries(): void
    {
        $n = count($this->keys);
        if ($n === 0) return;

        // Calcular ATR Series (usando la lógica original de SMA de TR)
        $this->atrSeries = $this->calculateATRSeries_SimpleSMA($this->atrPeriod);

        // Calcular Volumen Promedio Móvil Simple Series
        $this->avgVolumeSeries = $this->calculateRollingAverageSeries($this->data['volume'], $this->volumeLookback);

        // Calcular Cambios de Precio (Close[i] - Close[i-1]) Series
        $this->priceChangeSeries = $this->calculateDifferenceSeries($this->data['close']);

        // Calcular Ratio de Cambio de Volumen (Vol[i] / Vol[i-1]) Series
        $this->volumeChangeRatioSeries = $this->calculateRatioSeries($this->data['volume']);
    }

    /** Calcula ATR usando SMA simple de True Ranges (NO es el ATR estándar suavizado). */
    private function calculateATRSeries_SimpleSMA(int $period): ?array
    {
        $n = count($this->keys);
        if ($n <= 1 || $period <= 0 || $period >= $n) return array_fill_keys($this->keys, null);

        $trueRanges = [];
        foreach($this->keys as $i => $key) {
             if ($i === 0) continue; // Necesita precio previo
             $prevKey = $this->keys[$i-1];
             $high = $this->data['high'][$key];
             $low = $this->data['low'][$key];
             $closePrev = $this->data['close'][$prevKey];

             if ($high === null || $low === null || $closePrev === null) {
                 $trueRanges[$key] = null; // No se puede calcular TR
                 continue;
             }
             $tr = max(($high - $low), abs($high - $closePrev), abs($low - $closePrev));
             $trueRanges[$key] = $tr;
        }

        // Calcular SMA de los True Ranges
        // Necesitamos pasar solo los valores numéricos a la función de promedio móvil
        $numericTRs = array_filter($trueRanges, 'is_numeric');
        if (count($numericTRs) < $period) {
             // No hay suficientes TRs para calcular la primera media
             return array_fill_keys($this->keys, null);
        }

        // Usar una función genérica de promedio móvil simple para series
        $atrSmaSeries = $this->calculateRollingAverageSeries($trueRanges, $period); // Usará las claves de trueRanges

        // Rellenar nulos iniciales y alinear con las claves originales
        $alignedAtr = array_fill_keys($this->keys, null);
        foreach($atrSmaSeries as $key => $value) {
            if(isset($alignedAtr[$key])) { // Asegurar que la clave existe
                $alignedAtr[$key] = $value;
            }
        }
        return $alignedAtr;
        // NOTA: Para ATR estándar suavizado, la lógica es más compleja (EMA/Wilder's).
    }

     /** Calcula una serie de promedio móvil simple. */
    private function calculateRollingAverageSeries(array $sourceSeries, int $period): ?array
    {
        $n = count($sourceSeries);
        $keys = array_keys($sourceSeries);
        if ($n < $period || $period <= 0) return array_fill_keys($keys, null);

        $resultSeries = array_fill_keys($keys, null);
        $rollingSum = 0.0;
        $rollingCount = 0; // Cuenta de valores numéricos en la ventana

        // Convertir la serie fuente a un array indexado numéricamente para facilitar slicing
        $sourceValues = array_values($sourceSeries);

        for ($i = 0; $i < $n; $i++) {
            $currentKey = $keys[$i];
            $currentValue = $sourceValues[$i];

            // Añadir valor actual si es numérico
            if (is_numeric($currentValue)) {
                $rollingSum += $currentValue;
                $rollingCount++;
            }

            // Restar valor que sale de la ventana si estamos más allá del período inicial
            if ($i >= $period) {
                $valueToRemove = $sourceValues[$i - $period];
                if (is_numeric($valueToRemove)) {
                    $rollingSum -= $valueToRemove;
                    $rollingCount--;
                }
            }

            // Calcular y guardar la media si hemos completado el primer período y hay valores válidos
            if ($i >= $period - 1 && $rollingCount > 0) {
                $resultSeries[$currentKey] = $rollingSum / $rollingCount;
            } else {
                $resultSeries[$currentKey] = null; // Aún no hay suficientes datos o todos son null
            }
        }
        return $resultSeries;
    }

     /** Calcula la diferencia entre elemento[i] y elemento[i-1]. */
    private function calculateDifferenceSeries(array $sourceSeries): ?array
    {
        $n = count($sourceSeries);
        $keys = array_keys($sourceSeries);
        if ($n <= 1) return array_fill_keys($keys, null);
        $resultSeries = array_fill_keys($keys, null);
        $sourceValues = array_values($sourceSeries);

        for ($i = 1; $i < $n; $i++) {
             $currentKey = $keys[$i];
             $prevValue = $sourceValues[$i - 1];
             $currentValue = $sourceValues[$i];
             if (is_numeric($currentValue) && is_numeric($prevValue)) {
                 $resultSeries[$currentKey] = $currentValue - $prevValue;
             }
        }
        return $resultSeries;
    }

     /** Calcula el ratio elemento[i] / elemento[i-1]. */
    private function calculateRatioSeries(array $sourceSeries): ?array
    {
        $n = count($sourceSeries);
        $keys = array_keys($sourceSeries);
        if ($n <= 1) return array_fill_keys($keys, null);
        $resultSeries = array_fill_keys($keys, null);
        $sourceValues = array_values($sourceSeries);

        for ($i = 1; $i < $n; $i++) {
             $currentKey = $keys[$i];
             $prevValue = $sourceValues[$i - 1];
             $currentValue = $sourceValues[$i];
             if (is_numeric($currentValue) && is_numeric($prevValue) && $prevValue != 0) {
                 $resultSeries[$currentKey] = $currentValue / $prevValue;
             }
        }
        return $resultSeries;
    }

    //endregion

    //region Cálculo de Series de Análisis
    private function calculateVolumeClusterSeries(int $lookback = 5): ?array
    {
         if ($lookback <= 0) return array_fill_keys($this->keys, false);
         // Necesita el promedio móvil de volumen
         $avgVolSeries = $this->calculateRollingAverageSeries($this->data['volume'], $lookback);
         if($avgVolSeries === null) return array_fill_keys($this->keys, false);

         $resultSeries = array_fill_keys($this->keys, false);
         foreach($this->keys as $key) {
              $currentVolume = $this->data['volume'][$key] ?? null;
              $avgVolume = $avgVolSeries[$key] ?? null;
              if (is_numeric($currentVolume) && is_numeric($avgVolume) && $avgVolume > 0) {
                   if ($currentVolume > ($avgVolume * 2.5)) { // Umbral del método original
                        $resultSeries[$key] = true;
                   }
              }
         }
         return $resultSeries;
    }

    private function calculateAbsorptionSeries(): ?array
    {
         // Necesita: Cambio de precio, Volumen, Volumen Promedio, ATR
         if ($this->priceChangeSeries === null || $this->avgVolumeSeries === null || $this->atrSeries === null) {
             return array_fill_keys($this->keys, false);
         }

         $resultSeries = array_fill_keys($this->keys, false);
         foreach($this->keys as $key) {
              $priceChange = $this->priceChangeSeries[$key] ?? null;
              $currentVolume = $this->data['volume'][$key] ?? null;
              $avgVolume = $this->avgVolumeSeries[$key] ?? null; // Usa el promedio calculado con this->volumeLookback
              $atr = $this->atrSeries[$key] ?? null;

              if (is_numeric($priceChange) && is_numeric($currentVolume) && is_numeric($avgVolume) && $avgVolume > 0 && is_numeric($atr) && $atr > 0) {
                   if (($currentVolume > $avgVolume * 3) && (abs($priceChange) < ($atr * $this->priceSensitivity))) {
                        $resultSeries[$key] = true;
                   }
              }
         }
         return $resultSeries;
    }

    /** Calcula una serie booleana indicando si la barra es un spike de volumen. */
    private function calculateAnomalySpikeSeries(): ?array
    {
         $numericVolumes = array_filter($this->data['volume'], 'is_numeric');
         if (count($numericVolumes) < 2) return array_fill_keys($this->keys, false);

         $mean = array_sum($numericVolumes) / count($numericVolumes);
         $stdDev = $this->calculateStdDev($numericVolumes); // StdDev de toda la serie
         $threshold = $mean + (2 * $stdDev);

         $resultSeries = array_fill_keys($this->keys, false);
         foreach($this->keys as $key) {
              $volume = $this->data['volume'][$key] ?? null;
              if (is_numeric($volume) && $volume > $threshold) {
                   $resultSeries[$key] = true;
              }
         }
         return $resultSeries;
    }

     /** Calcula la dirección ('up'/'down') del precio en las barras que son spikes. */
    private function calculateLastSpikeDirectionSeries(?array $spikeSeries): ?array
    {
        if($spikeSeries === null) return array_fill_keys($this->keys, null);

        $resultSeries = array_fill_keys($this->keys, null);
        $closeValues = array_values($this->data['close']); // Indexado numéricamente

        foreach($this->keys as $i => $key) {
            if (($spikeSeries[$key] ?? false) === true) { // Si es un spike
                 if ($i > 0) { // Necesita barra anterior para dirección
                      $priceNow = $closeValues[$i];
                      $priceBefore = $closeValues[$i - 1];
                      if (is_numeric($priceNow) && is_numeric($priceBefore)) {
                           $resultSeries[$key] = ($priceNow > $priceBefore) ? 'up' : (($priceNow < $priceBefore) ? 'down' : 'neutral'); // Añadido neutral
                      }
                 }
            }
        }
        return $resultSeries;
    }

    /** Calcula la tendencia del volumen para cada barra usando regresión lineal en ventana móvil. */
    private function calculateVolumeTrendSeries(int $lookback = 14): ?array
    {
        $n = count($this->keys);
        if ($n < $lookback || $lookback < 2) return array_fill_keys($this->keys, 'neutral');

        $resultSeries = array_fill_keys($this->keys, 'neutral');
        $volumeValues = array_values($this->data['volume']); // Indexado numéricamente

        for ($i = $lookback - 1; $i < $n; $i++) {
            $currentKey = $this->keys[$i];
            // Extraer la ventana de volúmenes (solo numéricos)
            $window = array_slice($volumeValues, $i - $lookback + 1, $lookback);
            $numericWindow = array_filter($window, 'is_numeric');

            // Realizar regresión solo si hay suficientes puntos en la ventana
            if (count($numericWindow) >= 2) { // Necesita al menos 2 puntos para regresión
                 $regression = $this->linearRegression(array_values($numericWindow)); // Pasar solo valores
                 if ($regression['slope'] > 1e-9) { // Umbral pequeño para evitar ruido flotante
                      $resultSeries[$currentKey] = 'rising';
                 } elseif ($regression['slope'] < -1e-9) {
                      $resultSeries[$currentKey] = 'falling';
                 } else {
                     $resultSeries[$currentKey] = 'neutral';
                 }
            } else {
                // No hay suficientes datos en la ventana, mantener neutral
                 $resultSeries[$currentKey] = 'neutral';
            }
        }
        return $resultSeries;
    }

     /** Calcula serie booleana indicando posible nivel de liquidez (bajo mov precio, alto ratio vol). */
    private function calculateLiquidityLevelSeries(): ?array
    {
        if ($this->priceChangeSeries === null || $this->volumeChangeRatioSeries === null || $this->atrSeries === null) {
            return array_fill_keys($this->keys, false);
        }
        $resultSeries = array_fill_keys($this->keys, false);
        foreach($this->keys as $key) {
            $priceDelta = $this->priceChangeSeries[$key] ?? null;
            $volumeRatio = $this->volumeChangeRatioSeries[$key] ?? null; // Usamos el ratio calculado
            $atr = $this->atrSeries[$key] ?? null;

            if (is_numeric($priceDelta) && is_numeric($volumeRatio) && is_numeric($atr) && $atr > 0) {
                 // Condición original: abs(PrecioDelta) < ATR*0.3 Y RatioVolumen > 1.8
                 // El ratio > 1.8 significa Vol[i] > 1.8 * Vol[i-1]
                 if (abs($priceDelta) < ($atr * 0.3) && $volumeRatio > 1.8) {
                      $resultSeries[$key] = true;
                 }
            }
        }
        return $resultSeries;
    }

    /** Calcula serie booleana para fase de Acumulación (Tendencia Vol 'rising' Y Absorción). */
    private function calculateIsAccumulationSeries(): ?array
    {
         if ($this->volumeTrendSeries === null || $this->absorptionSeries === null) {
             return array_fill_keys($this->keys, false);
         }
         $resultSeries = array_fill_keys($this->keys, false);
         foreach($this->keys as $key) {
              $volTrend = $this->volumeTrendSeries[$key] ?? 'neutral';
              $isAbsorption = $this->absorptionSeries[$key] ?? false;
              // Lógica adaptada: Tendencia volumen 'rising' Y hay absorción
              if ($volTrend === 'rising' && $isAbsorption === true) {
                   $resultSeries[$key] = true;
              }
         }
         return $resultSeries;
    }

     /** Calcula serie booleana para fase de Distribución (Tendencia Vol 'falling' Y Cluster Volumen). */
    private function calculateIsDistributionSeries(): ?array
    {
         if ($this->volumeTrendSeries === null || $this->volumeClusterSeries === null) {
             return array_fill_keys($this->keys, false);
         }
          $resultSeries = array_fill_keys($this->keys, false);
         foreach($this->keys as $key) {
              $volTrend = $this->volumeTrendSeries[$key] ?? 'neutral';
              $isCluster = $this->volumeClusterSeries[$key] ?? false;
               // Lógica adaptada: Tendencia volumen 'falling' Y hay cluster
              if ($volTrend === 'falling' && $isCluster === true) {
                   $resultSeries[$key] = true;
              }
         }
         return $resultSeries;
    }
    //endregion

    //region Métodos de Utilidad y Cálculo Estático (no modificados o auxiliares)

    /** Obtiene el perfil de volumen ESTÁTICO sobre todos los datos. */
    public function getVolumeProfile(): array {
        $profile = [];
        // Asegurar que los precios son floats antes de redondear
        $numericPrices = array_filter($this->data['close'], 'is_numeric');
        if(empty($numericPrices)) return [];

        $strPrices = array_map(function($price) { return strval(round($price, 2)); }, $numericPrices);
        $priceCounts = array_count_values($strPrices);
        arsort($priceCounts);
        $topPriceStrings = array_keys(array_slice($priceCounts, 0, 5, true));
        // Convertir de nuevo a float
        $topPrices = array_map('floatval', $topPriceStrings);
        return $topPrices;
    }

    private function calculateStdDev(array $data): float {
        $count = count($data); if ($count === 0) return 0.0;
        $mean = array_sum($data) / $count;
        $variance = array_sum(array_map(function ($x) use ($mean) { return pow($x - $mean, 2); }, $data)) / $count;
        return sqrt($variance);
    }

    private function linearRegression(array $data): array {
        $n = count($data); if ($n < 2) return ['slope' => 0, 'intercept' => 0];
        $x = range(1, $n); $sumX = array_sum($x); $sumY = array_sum($data);
        $sumXY = 0; $sumXX = 0;
        for ($i = 0; $i < $n; $i++) { $sumXX += $x[$i] * $x[$i]; $sumXY += $x[$i] * $data[$i]; }
        $denominator = ($n * $sumXX - $sumX * $sumX);
        if (abs($denominator) < 1e-9) return ['slope' => 0, 'intercept' => 0]; // Evitar división por cero
        $slope = ($n * $sumXY - $sumX * $sumY) / $denominator;
        $intercept = ($sumY - $slope * $sumX) / $n;
        return ['slope' => $slope, 'intercept' => $intercept];
    }

    // Incluir helpers de alineación por si se usan internamente o para debug
    private function cleanIndicatorOutput($indicatorOutput): array { if ($indicatorOutput === false || !is_array($indicatorOutput)) return []; $cleaned = []; foreach ($indicatorOutput as $key => $value) { if ($value === false || !is_numeric($value)) { $cleaned[$key] = null; } else { $cleaned[$key] = (float)$value; } } return $cleaned; }
    private function alignAndCleanWithReference(?array $resultArray, ?array $referenceArray): ?array { if ($resultArray === null || $referenceArray === null) return null; $cleanedResult = $this->cleanIndicatorOutput($resultArray); if (empty($referenceArray)) return []; if (empty($cleanedResult)) return array_fill_keys(array_keys($referenceArray), null); $alignedArray = []; $refKeys = array_keys($referenceArray); $refCount = count($refKeys); $resValues = array_values($cleanedResult); $resCount = count($resValues); $initialNulls = $refCount - $resCount; if ($initialNulls < 0) return null; foreach ($refKeys as $index => $key) { if ($index < $initialNulls) { $alignedArray[$key] = null; } else { $resIndex = $index - $initialNulls; $alignedArray[$key] = $resValues[$resIndex] ?? null; } } return $alignedArray; }

    //endregion

} // Fin Clase SmartMoneyVolume (Refactorizada para Series)