<?php
namespace App;
use App\Trader;

/**
 * Analiza la relación entre el precio y el volumen para detectar patrones significativos.
 *
 * Esta clase proporciona métodos para identificar anomalías de volumen, tendencias y niveles de liquidez,
 * así como para determinar si el mercado se encuentra en una fase de acumulación o distribución.
 */
class SmartMoneyVolume070425 {
    /**
     * @var array Array de precios.
     */
    private $prices;

    /**
     * @var array Array de volúmenes.
     */
    private $volumes;

    /**
     * @var array All Data.
     */
    private $data;

    /**
     * @var int Número de periodos a considerar para el análisis de volumen.
     *
     * El valor por defecto de 20 se utiliza en varios cálculos, como el de la media del volumen.
     * Este valor puede ser ajustado, pero se recomienda mantenerlo en un rango razonable (10-30)
     * para equilibrar la sensibilidad del análisis con la suavidad de los datos.
     */
    private $volumeLookback = 20;

    /**
     * @var float Sensibilidad del precio utilizada en la detección de absorción.
     *
     * Este valor representa el porcentaje de Average True Range (ATR) que se considera un
     * movimiento de precio "pequeño". Un valor más alto hará que la detección de absorción sea
     * menos sensible, mientras que un valor más bajo la hará más sensible.
     */
    private $priceSensitivity = 0.15;

    /**
     * @var int Período para el cálculo del Average True Range (ATR).
     */
    private $atrPeriod = 14;

    /**
     * Constructor de la clase SmartMoneyVolume.
     *
     * @param array $prices Array de precios.
     * @param array $volumes Array de volúmenes.
     *
     * @throws \InvalidArgumentException Si los arrays de precios y volúmenes están vacíos o tienen una longitud diferente.
     */
    public function __construct(array $data) {
        if (empty($data['close']) || empty($data['volume']) || count($data['close']) !== count($data['volume'])) {
            throw new \InvalidArgumentException("Los arrays de precios y volúmenes deben contener datos y tener la misma longitud.");
        }

        $this->prices = $data['close'];
        $this->volumes = $data['volume'];
        $this->data = $data;
    }

    //region Métodos de Detección de Patrones

    /**
     * Detecta un clúster de volumen, indicativo de una actividad significativa.
     *
     * @param int $lookback Número de periodos a considerar para el cálculo del volumen promedio.
     * Un valor más pequeño hace que la detección sea más sensible a los picos de volumen recientes,
     * mientras que un valor más grande la hace menos sensible. Se recomienda un valor entre 3 y 7.
     *
     * @return bool True si se detecta un clúster de volumen, false en caso contrario.
     */
    public function detectVolumeCluster(int $lookback = 5): bool {
        if ($lookback > count($this->volumes)) {
            throw new \InvalidArgumentException("El lookback no puede ser mayor que el número de volúmenes disponibles.");
        }
        $recentVolumes = array_slice($this->volumes, -$lookback);
        $averageVolume = array_sum($recentVolumes) / $lookback;
        $currentVolume = end($this->volumes);

        return $currentVolume > ($averageVolume * 2.5);
    }

    /**
     * Detecta absorción, un patrón donde un volumen alto acompaña a un pequeño movimiento de precio.
     *
     * @return bool True si se detecta absorción, false en caso contrario.
     */
    public function detectAbsorption(): bool {
        $lastPriceChange = $this->getPriceChange(1);
        $lastVolume = end($this->volumes);
        $volumeAvg = $this->getAverageVolume($this->volumeLookback); // Usa el valor por defecto de la clase.
        $atr = $this->getATR($this->atrPeriod);

        return ($lastVolume > $volumeAvg * 3) && (abs($lastPriceChange) < ($atr * $this->priceSensitivity));
    }

    /**
     * Obtiene el perfil de volumen, identificando los niveles de precio con mayor actividad.
     *
     * @return array Array de los 5 niveles de precio más significativos, ordenados por volumen.
     */
    public function getVolumeProfile(): array {
        $profile = [];
        // Usa round() directamente en el array_map.
        $strPrices = array_map(function($price) {
            return strval(round($price, 2));
        }, $this->prices);
        $priceCounts = array_count_values($strPrices);
        arsort($priceCounts);
        $slice = array_keys(array_slice($priceCounts, 0, 5, true));
        // Usa round() directamente en el array_map.
        $roundedPrices = array_map(function($price) {
            return round($price, 2);
        }, $slice);
        //return array_keys(array_slice($priceCounts, 0, 5, true)); //Retiene las claves
        return $roundedPrices;
    }

    /**
     * Detecta anomalías de volumen, específicamente spikes.
     *
     * @return array Array de anomalías detectadas, donde cada anomalía es un array asociativo
     * con las claves 'index', 'price', 'volume' y 'type' ('spike').
     */
    public function detectAnomalies(): array {
        $anomalies = [];
        $mean = $this->getAverageVolume(count($this->volumes));
        $stdDev = $this->calculateStdDev($this->volumes);

        foreach ($this->volumes as $index => $volume) {
            if ($volume > ($mean + (2 * $stdDev))) {
                $anomalies[] = [
                    'index' => $index,
                    'price' => $this->prices[$index],
                    'volume' => $volume,
                    'type' => 'spike',
                ];
            }
        }
        return $anomalies;
    }

    /**
     * Obtiene la tendencia del volumen.
     *
     * @param int $lookback Número de periodos a considerar para el cálculo de la tendencia.
     * Un valor más pequeño hace que la tendencia sea más sensible a los cambios recientes,
     * mientras que un valor más grande la hace menos sensible. Se recomienda un valor entre 10 y 20.
     *
     * @return string 'rising' si el volumen tiene tendencia alcista, 'falling' si es bajista,
     * o 'neutral' si no se puede determinar una tendencia clara.
     */
    public function getVolumeTrend(int $lookback = 14): string {
        if ($lookback > count($this->volumes)) {
            throw new \InvalidArgumentException("El lookback no puede ser mayor que el número de volúmenes disponibles.");
        }
        $recentVolumes = array_slice($this->volumes, -$lookback);
        $regression = $this->linearRegression($recentVolumes);
        if(abs($regression['slope']) < 1e-5){
            return 'neutral';
        }
        return ($regression['slope'] > 0) ? 'rising' : 'falling';
    }

    /**
     * Calcula los niveles de liquidez, identificando áreas donde se acumula el volumen.
     *
     * @return array Array de niveles de liquidez, donde cada nivel es un array asociativo
     * con las claves 'price', 'volume' y 'type' ('liquidity_pool').
     */
    public function calculateLiquidityLevels(): array {
        $levels = [];
        $priceChanges = $this->calculatePriceChanges();
        $volumeChanges = $this->calculateVolumeChanges();
        $atr = $this->getATR($this->atrPeriod); //Calcula el ATR una vez

        foreach ($this->prices as $index => $price) {
            if ($index < 1) continue;

            $priceDelta = $priceChanges[$index - 1];
            $volumeDelta = $volumeChanges[$index - 1];

            if (abs($priceDelta) < ($atr * 0.3) && $volumeDelta > 1.8) {
                $levels[] = [
                    'price' => $price,
                    'volume' => $this->volumes[$index],
                    'type' => 'liquidity_pool',
                ];
            }
        }
        return $levels;
    }

    /**
     * Determina si el mercado se encuentra en una fase de acumulación.
     *
     * @return bool True si hay acumulación, false en caso contrario.
     */
    public function isAccumulation(): bool {
        $volumeProfile = $this->getVolumeProfile();
        $currentPrice = end($this->prices);
        $currentPriceRounded = round($currentPrice, 2);
        return in_array($currentPriceRounded, $volumeProfile)
            && $this->getVolumeTrend(5) === 'rising'
            && $this->detectAbsorption();
    }

    /**
     * Determina si el mercado se encuentra en una fase de distribución.
     *
     * @return bool True si hay distribución, false en caso contrario.
     */
    public function isDistribution(): bool {
        $volumeProfile = $this->getVolumeProfile();
        $currentPrice = end($this->prices);
        $currentPriceRounded = round($currentPrice, 2);
        return in_array($currentPriceRounded, $volumeProfile)
            && $this->getVolumeTrend(5) === 'falling'
            && $this->detectVolumeCluster();
    }
    //endregion

    //region Métodos de Utilidad y Cálculo

    /**
     * Calcula el cambio de precio porcentual.
     *
     * @param int $lookback Número de periodos a considerar para el cálculo.
     * Un valor de 1 calcula el cambio de precio del período anterior al actual.
     *
     * @return float Cambio de precio porcentual.
     *
     * @throws \InvalidArgumentException Si el lookback es mayor o igual que el número de precios disponibles.
     */
    private function getPriceChange(int $lookback = 1): float {
        if ($lookback >= count($this->prices)) {
            throw new \InvalidArgumentException("El lookback no puede ser mayor o igual que el número de precios disponibles.");
        }
        $currentIndex = count($this->prices) - 1;
        $previousIndex = $currentIndex - $lookback;

        // Verificar si el índice anterior es válido
        if ($previousIndex < 0) {
            throw new \InvalidArgumentException("El lookback es demasiado grande y excede los datos disponibles.");
        }
        $currentPrice = $this->prices[$currentIndex];
        $previousPrice = $this->prices[$previousIndex];

        return (($currentPrice - $previousPrice) / $previousPrice) * 100;
    }

    /**
     * Calcula la desviación estándar de un array de datos.
     *
     * @param array $data Array de datos.
     * @return float Desviación estándar.
     */
    private function calculateStdDev(array $data): float {
        $count = count($data);
        if ($count === 0) {
            return 0;
        }
        $mean = array_sum($data) / $count;
        $squaredDifferencesSum = array_sum(array_map(function ($value) use ($mean) {
            return pow($value - $mean, 2);
        }, $data));
        return sqrt($squaredDifferencesSum / $count);
    }

    /**
     * Realiza una regresión lineal sobre un array de datos.
     *
     * @param array $data Array de datos.
     * @return array Array asociativo con la pendiente ('slope') y la intersección ('intercept').
     */
    private function linearRegression(array $data): array {
        $n = count($data);
        if ($n < 2) {
            return ['slope' => 0, 'intercept' => 0]; // No se puede calcular la regresión con menos de 2 puntos.
        }

        $xSum = 0;
        $ySum = 0;
        $xxSum = 0;
        $xySum = 0;

        foreach ($data as $i => $y) {
            $x = $i + 1; // x values: [1, 2, 3, ...]
            $xSum += $x;
            $ySum += $y;
            $xxSum += $x * $x;
            $xySum += $x * $y;
        }

        $slope = ($n * $xySum - $xSum * $ySum) / ($n * $xxSum - $xSum * $xSum);
        $intercept = ($ySum - $slope * $xSum) / $n;

        return ['slope' => $slope, 'intercept' => $intercept];
    }

    /**
     * Calcula el Average True Range (ATR).
     *
     * @param int $period Período para el cálculo del ATR.
     * @return float Valor del ATR.
     *
     * @throws \InvalidArgumentException Si el período es mayor que el número de precios disponibles.
     */
    private function getATR(int $period = 14): float {
        $priceCount = count($this->prices);
        if ($period > $priceCount) {
            throw new \InvalidArgumentException("El período del ATR no puede ser mayor que el número de precios disponibles.");
        }
        $trueRanges = [];
        for ($i = 1; $i < $priceCount; $i++) {
            $high = $this->data['high'][$i];  // Accede al 'high'
            $low = $this->data['low'][$i];    // Accede al 'low'
            $closePrev = $this->data['close'][$i - 1]; // Accede al 'close' del día anterior
            $trueRange = max(
                ($high - $low),
                abs($high - $closePrev),
                abs($low - $closePrev)
            );
            $trueRanges[] = $trueRange;
        }

        // Si el número de trues ranges es menor que el periodo, retorna 0.
        if (count($trueRanges) < $period) {
            return 0;
        }
        $atrSum = array_sum(array_slice($trueRanges, -$period));


        return $atrSum / $period;
    }


    /**
     * Calcula los cambios de precio absolutos.
     *
     * @return array Array de cambios de precio.
     */
    private function calculatePriceChanges(): array {
        $priceCount = count($this->prices);
        $changes = array_fill(0, $priceCount, 0); // Inicializa el array con 0s
        for ($i = 1; $i < $priceCount; $i++) {
             // Accede al 'close' del día actual y anterior
            $changes[$i] = $this->prices[$i]['close'] - $this->prices[$i - 1]['close'];
        }
        return $changes;
    }

    /**
     * Calcula los cambios de volumen relativos.
     *
     * @return array Array de cambios de volumen.
     */
    private function calculateVolumeChanges(): array {
        $volumeCount = count($this->volumes);
        $changes = array_fill(0, $volumeCount, 0); // Inicializa el array con 0s
        for ($i = 1; $i < $volumeCount; $i++) {
            $changes[$i] = $this->volumes[$i] / $this->volumes[$i - 1];
        }
        return $changes;
    }

    /**
     * Calcula el volumen promedio de los últimos n periodos.
     *
     * @param int $lookback El número de periodos a considerar.
     * @return float El volumen promedio.
     */
    private function getAverageVolume(int $lookback): float {
          if ($lookback > count($this->volumes)) {
            throw new \InvalidArgumentException("El lookback no puede ser mayor que el número de volúmenes disponibles.");
        }
        $recentVolumes = array_slice($this->volumes, -$lookback);
        return array_sum($recentVolumes) / $lookback;
    }

    /**
     * Determina la dirección del spike de volumen.
     *
     * @return bool String si hay un spike, up down.
     */    
        public function getLastSpikeDirection(): ?string {
            $anomalies = $this->detectAnomalies();

            if (empty($anomalies)) {
                return null;
            }

            $lastSpike = end($anomalies);
            $index = $lastSpike['index'];

            if ($index < 1 || $index >= count($this->prices)) {
                return null;
            }

            $priceNow = is_array($this->prices[$index]) ? $this->prices[$index]['close'] : $this->prices[$index];
            $priceBefore = is_array($this->prices[$index - 1]) ? $this->prices[$index - 1]['close'] : $this->prices[$index - 1];


            return ($priceNow > $priceBefore) ? 'up' : 'down';
        }


    /**
     * Determina si hay un spike de volumen.
     *
     * @return bool True si hay un spike, false si no.
     */
    public function hasSpike(): bool {
        $anomalies = $this->detectAnomalies();
        foreach ($anomalies as $anomaly) {
            if ($anomaly['type'] === 'spike') {
                return true;
            }
        }
        return false;
    }
    //endregion
}

