<?php

namespace App;
use App\Trader;
// Importa las clases necesarias si las hubiera (Trader no es necesaria aquí)

/**
 * Analiza la relación entre el precio y el volumen para detectar patrones significativos
 * en el contexto de "Smart Money" (dinero institucional).
 *
 * Esta clase proporciona métodos para identificar anomalías de volumen, tendencias,
 * niveles de liquidez clave basados en el perfil de volumen, y determinar si el mercado
 * se encuentra en una fase de acumulación o distribución.
 */
class SmartMoneyVolume
{
    /**
     * @var array Array de precios de cierre.
     */
    public array $prices;

    /**
     * @var array Array de volúmenes.
     */
    public array $volumes;

    /**
     * @var array Datos completos del OHLCV (Open, High, Low, Close, Volume).
     */
    private array $data;

    /**
     * @var int Número de periodos a considerar por defecto para el cálculo de la media del volumen.
     */
    private int $defaultVolumeLookback = 20;

    /**
     * @var int Número de periodos por defecto para el cálculo de la tendencia del volumen.
     */
    private int $defaultVolumeTrendLookback = 14;

    /**
     * @var int Período por defecto para el cálculo del Average True Range (ATR).
     */
    private int $defaultAtrPeriod = 14;

    /**
     * @var float Sensibilidad del precio utilizada en la detección de absorción,
     * como porcentaje del ATR. Un valor más alto (e.g., 0.2) permite mayores
     * movimientos de precio con volumen alto; un valor más bajo (e.g., 0.05)
     * requiere movimientos de precio muy pequeños para considerar absorción.
     */
    private float $absorptionPriceSensitivity = 0.15; // 15% del ATR

    /**
     * @var float Multiplicador del volumen promedio para detectar un cluster de volumen.
     * Un volumen actual debe ser X veces el volumen promedio reciente.
     */
    private float $volumeClusterMultiplier = 2.5;

    /**
     * @var float Multiplicador del volumen promedio para la detección de absorción.
     * Un volumen actual debe ser Y veces el volumen promedio.
     */
    private float $absorptionVolumeMultiplier = 3.0;

    /**
     * @var float Umbral del cambio de volumen relativo para identificar niveles de liquidez.
     * Representa cuántas veces el volumen actual es mayor que el anterior.
     */
    private float $liquidityVolumeRatioThreshold = 1.8;

    /**
     * @var float Umbral del cambio de precio absoluto para identificar niveles de liquidez,
     * como porcentaje del ATR. Indica cuán pequeño debe ser el movimiento de precio
     * para un determinado aumento de volumen.
     */
    private float $liquidityPriceThresholdATR = 0.3; // 30% del ATR

    /**
     * Constructor de la clase SmartMoneyVolume.
     *
     * @param array $data Array asociativo con los datos históricos del precio y volumen.
     * Debe contener las claves 'open', 'high', 'low', 'close' y 'volume',
     * cada una siendo un array de floats/integers de la misma longitud.
     *
     * @throws \InvalidArgumentException Si los datos proporcionados son inválidos (faltan claves, arrays vacíos o longitud diferente).
     */
    public function __construct(array $data)
    {
        $requiredKeys = ['open', 'high', 'low', 'close', 'volume'];
        foreach ($requiredKeys as $key) {
            if (!isset($data[$key]) || !is_array($data[$key])) {
                throw new \InvalidArgumentException("Falta la clave '$key' o no es un array en los datos proporcionados.");
            }
        }

        $count = count($data['close']);
        if ($count === 0) {
            throw new \InvalidArgumentException("Los datos de precio 'close' están vacíos.");
        }

        foreach ($requiredKeys as $key) {
            if (count($data[$key]) !== $count) {
                throw new \InvalidArgumentException("Los arrays de datos no tienen la misma longitud.");
            }
        }

        $this->data = $data;
        // Solo almacenamos close y volume directamente para acceso frecuente
        $this->prices = $data['close'];
        $this->volumes = $data['volume'];
    }

    //region Métodos de Detección de Patrones

    /**
     * Detecta un clúster de volumen, indicativo de una actividad significativa
     * que excede considerablemente el volumen promedio reciente.
     *
     * @param int|null $lookback Número de periodos a considerar para el cálculo del volumen promedio.
     * Si es null, usa el valor por defecto de la clase ($defaultVolumeLookback).
     * @param float|null $multiplier Multiplicador del volumen promedio para definir el umbral del clúster.
     * Si es null, usa el valor por defecto de la clase ($volumeClusterMultiplier).
     *
     * @return bool True si se detecta un clúster de volumen en el último periodo, false en caso contrario.
     * @throws \InvalidArgumentException Si el lookback es mayor que el número de volúmenes disponibles.
     */
    public function detectVolumeCluster(?int $lookback = null, ?float $multiplier = null): bool
    {
        $lookback = $lookback ?? $this->defaultVolumeLookback;
        $multiplier = $multiplier ?? $this->volumeClusterMultiplier;

        if ($lookback > count($this->volumes)) {
            throw new \InvalidArgumentException("El lookback ($lookback) no puede ser mayor que el número de volúmenes disponibles (" . count($this->volumes) . ").");
        }
        if (count($this->volumes) < $lookback) {
             // No hay suficientes datos para calcular el promedio en el lookback
             return false;
        }

        $recentVolumes = array_slice($this->volumes, -$lookback);
        $averageVolume = array_sum($recentVolumes) / $lookback;
        $currentVolume = end($this->volumes);

        return $currentVolume > ($averageVolume * $multiplier);
    }

    /**
     * Detecta absorción en el último periodo, un patrón donde un volumen alto
     * acompaña a un movimiento de precio relativamente pequeño, sugiriendo que una
     * parte del volumen está siendo "absorbida" por participantes más fuertes
     * en el lado opuesto del movimiento.
     *
     * @param int|null $volumeLookback Período para el volumen promedio. Null usa el defecto.
     * @param float|null $priceSensitivity Porcentaje del ATR para definir un movimiento de precio "pequeño". Null usa el defecto.
     * @param int|null $atrPeriod Período para el cálculo del ATR. Null usa el defecto.
     *
     * @return bool True si se detecta absorción en el último periodo, false en caso contrario.
     * @throws \InvalidArgumentException Si no hay suficientes datos para calcular el ATR o el promedio de volumen.
     */
    public function detectAbsorption(?int $volumeLookback = null, ?float $priceSensitivity = null, ?int $atrPeriod = null): bool
    {
        $volumeLookback = $volumeLookback ?? $this->defaultVolumeLookback;
        $priceSensitivity = $priceSensitivity ?? $this->absorptionPriceSensitivity;
        $atrPeriod = $atrPeriod ?? $this->defaultAtrPeriod;

        // Necesitamos al menos 2 periodos para cambio de precio y ATR
        if (count($this->prices) < 2 || count($this->volumes) < $volumeLookback || count($this->data['high']) < $atrPeriod || count($this->data['low']) < $atrPeriod || count($this->data['close']) < $atrPeriod + 1) {
             return false; // No hay suficientes datos
        }

        $lastPriceChangeAbs = abs($this->getPriceChange(1)); // Cambio absoluto del último periodo
        $lastVolume = end($this->volumes);
        $volumeAvg = $this->getAverageVolume($volumeLookback);
        $atr = $this->getATR($atrPeriod);

        // Si ATR es muy pequeño o cero, esta condición podría ser problemática.
        // Considerar un umbral mínimo absoluto o manejar el caso ATR=0.
        // Por ahora, si atr es 0, priceSensitivity * atr es 0, lo que significa abs($lastPriceChangeAbs) < 0, siempre falso.
        // Esto es razonable: sin volatilidad (ATR=0), ¿puede haber absorción significativa?
        if ($atr <= 0) {
             return false;
        }

        return ($lastVolume > $volumeAvg * $this->absorptionVolumeMultiplier) &&
               ($lastPriceChangeAbs < ($atr * $priceSensitivity));
    }

    /**
     * Obtiene el perfil de volumen, identificando los niveles de precio con mayor actividad
     * durante todo el historial de datos cargado.
     *
     * @param int $topN El número de niveles de precio más significativos a retornar.
     * @param int $roundingDecimalPlaces Número de decimales para redondear los precios antes de contarlos.
     *
     * @return array Array de los N niveles de precio más significativos, ordenados por volumen (descendente).
     */
    public function getVolumeProfile(int $topN = 5, int $roundingDecimalPlaces = 2): array
    {
        if (empty($this->prices)) {
            return [];
        }

        $profile = [];
        // Redondea cada precio y lo convierte a string para usar como clave
        $strPrices = array_map(function($price) use ($roundingDecimalPlaces) {
            return strval(round($price, $roundingDecimalPlaces));
        }, $this->prices);

        // Cuenta cuántas veces aparece cada precio redondeado
        $priceCounts = array_count_values($strPrices);

        // Ordena los precios por la cuenta (volumen en este contexto) de forma descendente
        arsort($priceCounts);

        // Extrae las claves (los precios redondeados) de los N elementos superiores
        $topPricesStr = array_keys(array_slice($priceCounts, 0, $topN, true));

        // Convierte de nuevo a float
        $topPrices = array_map('floatval', $topPricesStr);

        return $topPrices;
    }

    /**
     * Detecta anomalías de volumen (spikes) comparando el volumen actual con la
     * desviación estándar del volumen histórico.
     *
     * @param float $stdDevMultiplier El número de desviaciones estándar por encima de la media
     * para considerar un volumen como una anomalía.
     * @param int|null $lookback Número de periodos para calcular la media y desviación estándar.
     * Si es null, usa todos los datos disponibles. Usar un lookback
     * más corto puede detectar spikes relativos a la volatilidad reciente del volumen.
     *
     * @return array Array de anomalías detectadas, donde cada anomalía es un array asociativo
     * con las claves 'index', 'price', 'volume' y 'type' ('spike'). Retorna un
     * array vacío si no hay anomalías o no hay suficientes datos.
     * @throws \InvalidArgumentException Si el lookback es mayor que el número de volúmenes disponibles.
     */
    public function detectAnomalies(float $stdDevMultiplier = 2.0, ?int $lookback = null): array
    {
        $lookback = $lookback ?? count($this->volumes); // Usa todos los datos por defecto
        if ($lookback > count($this->volumes)) {
            throw new \InvalidArgumentException("El lookback ($lookback) no puede ser mayor que el número de volúmenes disponibles (" . count($this->volumes) . ").");
        }
         if (count($this->volumes) < $lookback) {
             return []; // No hay suficientes datos
         }

        $recentVolumes = array_slice($this->volumes, -$lookback);

        // Necesitamos al menos 2 puntos para calcular la desviación estándar
        if (count($recentVolumes) < 2) {
            return [];
        }

        $mean = array_sum($recentVolumes) / count($recentVolumes);
        $stdDev = $this->calculateStdDev($recentVolumes);

        $anomalies = [];
        $startIndex = count($this->volumes) - count($recentVolumes); // Índice de inicio en el array original

        foreach ($recentVolumes as $i => $volume) {
            $originalIndex = $startIndex + $i;
            // Evita la división por cero si la desviación estándar es 0 (volumen constante)
            // En ese caso, cualquier volumen diferente a la media sería una anomalía,
            // pero la condición $volume > ($mean + ...) no se cumpliría a menos que el multiplicador sea <= 0.
            // Si stdDev es 0, solo consideramos anomalía si el volumen es mayor estricto que la media (y el multiplicador > 0)
            if (($stdDev > 0 && $volume > ($mean + ($stdDevMultiplier * $stdDev))) || ($stdDev <= 0 && $volume > $mean && $stdDevMultiplier > 0)) {
                // Asegurarse de que el índice existe en el array de precios
                 if (isset($this->prices[$originalIndex])) {
                     $anomalies[] = [
                         'index' => $originalIndex,
                         'price' => $this->prices[$originalIndex],
                         'volume' => $volume,
                         'type' => 'spike',
                     ];
                 }
            }
        }
        return $anomalies;
    }

    /**
     * Obtiene la dirección de la tendencia del volumen basándose en una regresión lineal
     * sobre los últimos periodos.
     *
     * @param int|null $lookback Número de periodos a considerar para el cálculo de la tendencia.
     * Si es null, usa el valor por defecto de la clase ($defaultVolumeTrendLookback).
     * @param float $slopeTolerance Una pequeña tolerancia para considerar la pendiente como "neutral".
     * Una pendiente con valor absoluto menor que esta tolerancia se considera neutral.
     *
     * @return string 'rising' si el volumen tiene tendencia alcista, 'falling' si es bajista,
     * o 'neutral' si no se puede determinar una tendencia clara.
     * @throws \InvalidArgumentException Si el lookback es mayor que el número de volúmenes disponibles.
     */
    public function getVolumeTrend(?int $lookback = null, float $slopeTolerance = 1e-5): string
    {
        $lookback = $lookback ?? $this->defaultVolumeTrendLookback;
        if ($lookback > count($this->volumes)) {
            throw new \InvalidArgumentException("El lookback ($lookback) no puede ser mayor que el número de volúmenes disponibles (" . count($this->volumes) . ").");
        }
        if (count($this->volumes) < $lookback) {
             return 'neutral'; // No hay suficientes datos para determinar la tendencia
        }

        $recentVolumes = array_slice($this->volumes, -$lookback);

        // Necesitamos al menos 2 puntos para la regresión lineal
        if (count($recentVolumes) < 2) {
             return 'neutral';
        }

        $regression = $this->linearRegression($recentVolumes);

        if (abs($regression['slope']) < $slopeTolerance) {
            return 'neutral';
        }
        return ($regression['slope'] > 0) ? 'rising' : 'falling';
    }

     /**
      * Calcula los niveles de liquidez, identificando áreas donde el volumen
      * aumenta significativamente mientras el precio se mueve poco. Esto sugiere
      * actividad de grandes participantes absorbiendo oferta o demanda.
      *
      * @param float|null $priceThresholdATR Umbral del cambio de precio absoluto como % del ATR. Null usa el defecto.
      * @param float|null $volumeRatioThreshold Umbral del cambio de volumen relativo. Null usa el defecto.
      * @param int|null $atrPeriod Periodo para el cálculo del ATR. Null usa el defecto.
      *
      * @return array Array de niveles de liquidez detectados, donde cada nivel es un array asociativo
      * con las claves 'index', 'price', 'volume' y 'type' ('liquidity_pool').
      * Retorna un array vacío si no se detectan niveles o no hay suficientes datos.
      * @throws \InvalidArgumentException Si no hay suficientes datos para los cálculos necesarios.
      */
    public function calculateLiquidityLevels(?float $priceThresholdATR = null, ?float $volumeRatioThreshold = null, ?int $atrPeriod = null): array
    {
        $priceThresholdATR = $priceThresholdATR ?? $this->liquidityPriceThresholdATR;
        $volumeRatioThreshold = $volumeRatioThreshold ?? $this->liquidityVolumeRatioThreshold;
        $atrPeriod = $atrPeriod ?? $this->defaultAtrPeriod;

        // Necesitamos al menos 2 periodos para calcular cambios y ATR
        if (count($this->prices) < 2 || count($this->volumes) < 2 || count($this->data['high']) < $atrPeriod || count($this->data['low']) < $atrPeriod || count($this->data['close']) < $atrPeriod + 1) {
             return []; // No hay suficientes datos
        }

        $levels = [];
        $atr = $this->getATR($atrPeriod); // Calcula el ATR una vez

         // Si ATR es muy pequeño o cero, los umbrales de precio serán cero.
         // Esto podría hacer que casi cualquier movimiento de precio sea "pequeño".
         // Podríamos añadir un umbral mínimo absoluto si es necesario.
         if ($atr <= 0) {
             return $levels; // No se pueden detectar niveles de liquidez basados en ATR si la volatilidad es cero
         }

        // Calcula los cambios una vez
        $priceChanges = $this->calculatePriceChanges();
        $volumeChanges = $this->calculateVolumeChanges();

        // Iterar desde el segundo elemento (índice 1) ya que los cambios se calculan respecto al anterior
        for ($i = 1; $i < count($this->prices); $i++) {
            $priceDeltaAbs = abs($priceChanges[$i]); // Usamos el cambio absoluto del periodo actual
            $volumeDeltaRatio = $volumeChanges[$i]; // Usamos el cambio relativo del periodo actual

            // Condición: Pequeño cambio de precio (en relación con ATR) Y Gran aumento de volumen (relativo)
            if ($priceDeltaAbs < ($atr * $priceThresholdATR) && $volumeDeltaRatio > $volumeRatioThreshold) {
                 // Asegurarse de que el índice existe en el array de precios y volumen
                 if (isset($this->prices[$i]) && isset($this->volumes[$i])) {
                     $levels[] = [
                         'index' => $i,
                         'price' => $this->prices[$i],
                         'volume' => $this->volumes[$i],
                         'type' => 'liquidity_pool',
                     ];
                 }
            }
        }
        return $levels;
    }

    /**
     * Determina si el mercado en el último periodo se encuentra en una fase de acumulación
     * basándose en un conjunto de criterios: precio cerca de un nivel de volumen significativo,
     * tendencia de volumen alcista reciente y detección de absorción.
     *
     * @param int|null $volumeProfileTopN Número de niveles del perfil de volumen a considerar. Null usa defecto.
     * @param int|null $volumeTrendLookback Lookback para la tendencia de volumen. Null usa defecto.
     * @param float|null $liquidityProximityPercentage Porcentaje de proximidad a los niveles de liquidez. Null usa defecto.
     * @param int|null $atrPeriod Periodo para el ATR usado en la proximidad. Null usa defecto.
     *
     * @return bool True si hay indicación de acumulación en el último periodo, false en caso contrario.
     * @throws \InvalidArgumentException Si no hay suficientes datos para los cálculos necesarios.
     */
    public function isAccumulation(?int $volumeProfileTopN = null, ?int $volumeTrendLookback = null, ?float $liquidityProximityPercentage = null, ?int $atrPeriod = null): bool
    {
        $volumeProfileTopN = $volumeProfileTopN ?? 5; // Usar el defecto del método getVolumeProfile
        $volumeTrendLookback = $volumeTrendLookback ?? 5; // Lookback más corto para tendencia reciente
        $atrPeriod = $atrPeriod ?? $this->defaultAtrPeriod;

         if (count($this->prices) < 2 || count($this->volumes) < $volumeTrendLookback || count($this->data['high']) < $atrPeriod || count($this->data['low']) < $atrPeriod || count($this->data['close']) < $atrPeriod + 1) {
             return false; // No hay suficientes datos
         }

        $volumeProfile = $this->getVolumeProfile($volumeProfileTopN);
        $currentPrice = end($this->prices);

        // Criterio 1: Precio actual cerca de un nivel del perfil de volumen
        $isNearVolumeLevel = false;
         // Usar el mismo umbral de proximidad que en la estrategia si se define, o un defecto aquí
         $liquidityProximityPercentage = $liquidityProximityPercentage ?? 0.5; // Default 0.5%
         if ($currentPrice > 0) { // Evitar división por cero
             foreach ($volumeProfile as $levelPrice) {
                 $percentageDifference = abs($currentPrice - $levelPrice) / $currentPrice * 100;
                 if ($percentageDifference <= $liquidityProximityPercentage) {
                     $isNearVolumeLevel = true;
                     break;
                 }
             }
         }


        // Criterio 2: Tendencia de volumen alcista reciente
        $isVolumeTrendRising = $this->getVolumeTrend($volumeTrendLookback) === 'rising';

        // Criterio 3: Detección de absorción
        $isAbsorptionDetected = $this->detectAbsorption(
             $this->defaultVolumeLookback, // Usar defecto de la clase para absorción
             $this->absorptionPriceSensitivity, // Usar defecto de la clase para absorción
             $atrPeriod // Usar el periodo ATR proporcionado o defecto
         );


        return $isNearVolumeLevel && $isVolumeTrendRising && $isAbsorptionDetected;
    }

    /**
     * Determina si el mercado en el último periodo se encuentra en una fase de distribución
     * basándose en un conjunto de criterios: precio cerca de un nivel de volumen significativo,
     * tendencia de volumen bajista reciente y detección de un clúster de volumen.
     *
     * @param int|null $volumeProfileTopN Número de niveles del perfil de volumen a considerar. Null usa defecto.
     * @param int|null $volumeTrendLookback Lookback para la tendencia de volumen. Null usa defecto.
     * @param float|null $liquidityProximityPercentage Porcentaje de proximidad a los niveles de liquidez. Null usa defecto.
     * @param int|null $atrPeriod Periodo para el ATR usado en la proximidad. Null usa defecto.
     *
     * @return bool True si hay indicación de distribución en el último periodo, false en caso contrario.
     * @throws \InvalidArgumentException Si no hay suficientes datos para los cálculos necesarios.
     */
    public function isDistribution(?int $volumeProfileTopN = null, ?int $volumeTrendLookback = null, ?float $liquidityProximityPercentage = null, ?int $atrPeriod = null): bool
    {
        $volumeProfileTopN = $volumeProfileTopN ?? 5; // Usar el defecto del método getVolumeProfile
        $volumeTrendLookback = $volumeTrendLookback ?? 5; // Lookback más corto para tendencia reciente
        $atrPeriod = $atrPeriod ?? $this->defaultAtrPeriod;

        if (count($this->prices) < 2 || count($this->volumes) < $volumeTrendLookback || count($this->data['high']) < $atrPeriod || count($this->data['low']) < $atrPeriod || count($this->data['close']) < $atrPeriod + 1) {
             return false; // No hay suficientes datos
         }

        $volumeProfile = $this->getVolumeProfile($volumeProfileTopN);
        $currentPrice = end($this->prices);

        // Criterio 1: Precio actual cerca de un nivel del perfil de volumen
        $isNearVolumeLevel = false;
         // Usar el mismo umbral de proximidad que en la estrategia si se define, o un defecto aquí
         $liquidityProximityPercentage = $liquidityProximityPercentage ?? 0.5; // Default 0.5%
         if ($currentPrice > 0) { // Evitar división por cero
             foreach ($volumeProfile as $levelPrice) {
                 $percentageDifference = abs($currentPrice - $levelPrice) / $currentPrice * 100;
                 if ($percentageDifference <= $liquidityProximityPercentage) {
                     $isNearVolumeLevel = true;
                     break;
                 }
             }
         }

        // Criterio 2: Tendencia de volumen bajista reciente
        $isVolumeTrendFalling = $this->getVolumeTrend($volumeTrendLookback) === 'falling';

        // Criterio 3: Detección de un clúster de volumen
        $isVolumeClusterDetected = $this->detectVolumeCluster(
             $this->defaultVolumeLookback, // Usar defecto de la clase para clúster
             $this->volumeClusterMultiplier // Usar defecto de la clase para clúster
         );

        return $isNearVolumeLevel && $isVolumeTrendFalling && $isVolumeClusterDetected;
    }

     /**
      * Determina la dirección del último spike de volumen detectado.
      *
      * @return string|null 'up' si el precio subió durante el spike, 'down' si bajó,
      * o null si no se detectaron spikes o no hay suficientes datos.
      * @throws \InvalidArgumentException Si no hay suficientes datos para los cálculos internos (detectAnomalies).
      */
    public function getLastSpikeDirection(): ?string
    {
        $anomalies = $this->detectAnomalies(); // Usa los parámetros por defecto de detectAnomalies

        if (empty($anomalies)) {
            return null;
        }

        // El último spike detectado (detectAnomalies retorna en orden ascendente por índice)
        $lastSpike = end($anomalies);
        $index = $lastSpike['index'];

        // Necesitamos al menos el periodo anterior para comparar el precio
        if ($index < 1 || $index >= count($this->prices)) {
            return null;
        }

        // Corregido: Acceder directamente al precio por índice
        $priceNow = $this->prices[$index];
        $priceBefore = $this->prices[$index - 1];

        if ($priceNow > $priceBefore) {
            return 'up';
        } elseif ($priceNow < $priceBefore) {
            return 'down';
        } else {
             // Precio no cambió durante el spike
             return null; // O podrías definir un 'neutral' o 'unchanged'
        }
    }

    /**
     * Determina si hay al menos un spike de volumen en los datos analizados.
     *
     * @return bool True si hay al menos un spike, false si no.
     * @throws \InvalidArgumentException Si no hay suficientes datos para los cálculos internos (detectAnomalies).
     */
    public function hasSpike(): bool
    {
        return !empty($this->detectAnomalies());
    }

    //endregion

    //region Métodos de Utilidad y Cálculo

    /**
     * Calcula el cambio de precio absoluto en el último periodo o durante un lookback.
     *
     * @param int $lookback Número de periodos a considerar para el cálculo del cambio (desde $i - $lookback hasta $i).
     * Si $lookback es 1, calcula el cambio del último periodo.
     * @param int|null $currentIndex El índice del periodo actual. Si es null, usa el último periodo disponible.
     *
     * @return float El cambio de precio absoluto. Retorna 0 si no hay suficientes datos.
     * @throws \InvalidArgumentException Si el lookback o currentIndex son inválidos.
     */
    public function getPriceChange(int $lookback = 1, ?int $currentIndex = null): float
    {
        $priceCount = count($this->prices);
        if ($priceCount < 2) {
             return 0.0; // No hay suficientes datos para un cambio
        }

        $currentIndex = $currentIndex ?? ($priceCount - 1);
        $previousIndex = $currentIndex - $lookback;

        if ($currentIndex >= $priceCount || $previousIndex < 0) {
             throw new \InvalidArgumentException("Índice actual ($currentIndex) o lookback ($lookback) inválido para los datos disponibles.");
        }

        // Corregido: Acceder directamente al precio por índice
        $currentPrice = $this->prices[$currentIndex];
        $previousPrice = $this->prices[$previousIndex];

        return $currentPrice - $previousPrice; // Retorna el cambio absoluto, no porcentual como el nombre podría sugerir
    }
    public function getDefaultVolumeLookback(): float
    {
        return $this->defaultVolumeLookback;
    }
    /**
     * Calcula la desviación estándar de un array de datos.
     *
     * @param array $data Array de datos numéricos.
     * @return float Desviación estándar. Retorna 0 si el array está vacío o tiene un solo elemento.
     */
    private function calculateStdDev(array $data): float
    {
        $count = count($data);
        if ($count < 2) {
            return 0.0;
        }

        $mean = array_sum($data) / $count;
        // Cálculo de la varianza poblacional (dividir por N)
        $variance = array_sum(array_map(function ($value) use ($mean) {
            return pow($value - $mean, 2);
        }, $data)) / $count;

        return sqrt($variance);
    }

    /**
     * Realiza una regresión lineal sobre un array de datos para encontrar la pendiente
     * e intersección de la línea de mejor ajuste. Se utiliza para determinar la tendencia.
     *
     * @param array $data Array de datos (valores Y). Los valores X implícitos son [1, 2, 3, ...].
     * @return array Array asociativo con la pendiente ('slope') y la intersección ('intercept').
     * Retorna ['slope' => 0, 'intercept' => 0] si no hay suficientes datos (menos de 2 puntos).
     */
    private function linearRegression(array $data): array
    {
        $n = count($data);
        if ($n < 2) {
            return ['slope' => 0.0, 'intercept' => 0.0]; // No se puede calcular la regresión con menos de 2 puntos.
        }

        $xSum = 0.0;
        $ySum = 0.0;
        $xxSum = 0.0;
        $xySum = 0.0;

        foreach ($data as $i => $y) {
            $x = $i + 1; // x values: [1, 2, 3, ...]
            $xSum += $x;
            $ySum += $y;
            $xxSum += $x * $x;
            $xySum += $x * $y;
        }

        $denominator = ($n * $xxSum - $xSum * $xSum);
        if ($denominator == 0) {
             // Todos los puntos tienen la misma coordenada X, es una línea vertical (o todos los puntos son el mismo)
             return ['slope' => 0.0, 'intercept' => $ySum / $n]; // Pendiente infinita, pero para tendencia consideramos 0
        }

        $slope = ($n * $xySum - $xSum * $ySum) / $denominator;
        $intercept = ($ySum - $slope * $xSum) / $n;

        return ['slope' => (float) $slope, 'intercept' => (float) $intercept];
    }

    /**
     * Calcula el Average True Range (ATR) sobre los datos históricos.
     *
     * @param int|null $period Período para el cálculo del ATR. Si es null, usa el valor por defecto de la clase ($defaultAtrPeriod).
     * @return float Valor del ATR para el último periodo disponible. Retorna 0.0 si no hay suficientes datos.
     * @throws \InvalidArgumentException Si el período es mayor que el número de datos disponibles.
     */
    public function getATR(?int $period = null): float
    {
        $period = $period ?? $this->defaultAtrPeriod;
        $dataCount = count($this->data['close']); // Usamos el conteo total de datos

        // Necesitamos al menos $period + 1 barras para calcular el ATR inicial
        if ($dataCount <= $period) {
            return 0.0;
        }

        $trueRanges = [];
        // Calculamos el True Range para cada barra desde la segunda
        for ($i = 1; $i < $dataCount; $i++) {
            $high = $this->data['high'][$i];
            $low = $this->data['low'][$i];
            $closePrev = $this->data['close'][$i - 1];

            $trueRange = max(
                ($high - $low),
                abs($high - $closePrev),
                abs($low - $closePrev)
            );
            $trueRanges[] = $trueRange;
        }

        // Si el número de true ranges es menor que el periodo, no podemos calcular el ATR
         if (count($trueRanges) < $period) {
             return 0.0;
         }

        // Calcula el ATR (generalmente se usa EMA o SMA, aquí usamos SMA simple por simplicidad)
        // Tomamos las últimas 'period' true ranges
        $recentTrueRanges = array_slice($trueRanges, -$period);
        $trader_atr = Trader::ATR($this->data['high'], $this->data['low'], $this->data['close'], 14);
        //var_dump(end($trader_atr));
        //var_dump(array_sum($recentTrueRanges) / $period);
        return array_sum($recentTrueRanges) / $period;
    }

    /**
     * Calcula los cambios de precio absolutos entre periodos consecutivos.
     *
     * @return array Array de cambios de precio. El índice $i contiene el cambio de precio
     * entre el periodo $i-1 y $i. El primer elemento es 0.0.
     */
    private function calculatePriceChanges(): array
    {
        $priceCount = count($this->prices);
        $changes = array_fill(0, $priceCount, 0.0); // Inicializa el array con 0.0s

        // Iterar desde el segundo elemento
        for ($i = 1; $i < $priceCount; $i++) {
            // Corregido: Acceder directamente al precio por índice
            $changes[$i] = $this->prices[$i] - $this->prices[$i - 1];
        }
        return $changes;
    }

    /**
     * Calcula los cambios de volumen relativos (ratio) entre periodos consecutivos.
     *
     * @return array Array de cambios de volumen relativos. El índice $i contiene
     * el ratio del volumen del periodo $i respecto al $i-1. El primer
     * elemento es 0.0.
     */
    private function calculateVolumeChanges(): array
    {
        $volumeCount = count($this->volumes);
        $changes = array_fill(0, $volumeCount, 0.0); // Inicializa el array con 0.0s

        // Iterar desde el segundo elemento
        for ($i = 1; $i < $volumeCount; $i++) {
            // Manejar el caso donde el volumen anterior es cero para evitar división por cero
            if ($this->volumes[$i - 1] > 0) {
                $changes[$i] = $this->volumes[$i] / $this->volumes[$i - 1];
            } else {
                // Si el volumen anterior era cero, el ratio es infinito o indefinido.
                // Un ratio muy alto (e.g., PHP_FLOAT_MAX) o 0.0 podría usarse.
                // 0.0 implica que no hubo aumento significativo relativo a la nada.
                // Usar un valor grande puede ser más indicativo de un cambio.
                // Decidimos usar un valor muy grande para representar un salto desde cero.
                $changes[$i] = ($this->volumes[$i] > 0) ? PHP_FLOAT_MAX : 0.0;
            }
        }
        return $changes;
    }

    /**
     * Calcula el volumen promedio de los últimos n periodos.
     *
     * @param int $lookback El número de periodos a considerar.
     * @return float El volumen promedio. Retorna 0.0 si no hay suficientes datos.
     * @throws \InvalidArgumentException Si el lookback es mayor que el número de volúmenes disponibles.
     */
    public function getAverageVolume(int $lookback): float
    {
        if ($lookback <= 0) {
             throw new \InvalidArgumentException("El lookback debe ser un número positivo.");
        }
        if ($lookback > count($this->volumes)) {
             throw new \InvalidArgumentException("El lookback ($lookback) no puede ser mayor que el número de volúmenes disponibles (" . count($this->volumes) . ").");
        }
        if (count($this->volumes) < $lookback) {
             return 0.0; // No hay suficientes datos
        }

        $recentVolumes = array_slice($this->volumes, -$lookback);
        $count = count($recentVolumes);
        return ($count > 0) ? (float) array_sum($recentVolumes) / $count : 0.0;
    }

    //endregion

    //region Getters para propiedades (opcional, si quieres exponer valores de configuración)
    // public function getDefaultVolumeLookback(): int { return $this->defaultVolumeLookback; }
    // public function getAbsorptionPriceSensitivity(): float { return $this->absorptionPriceSensitivity; }
    // ...etc.
    //endregion
}