<?php

namespace App;

use App\SmartMoneyVolume;
/**
 ******* Esta versión esta bien, la hemos mejorado con las ponderaciones el 080525, se usara la version actual *****
*/
// Puedes añadir aquí los 'use' de tus otras clases de indicadores
// si la estrategia los usa internamente para sus cálculos o filtros.
// use App\RSI;
// use App\BreakOutProbability;
// use App\ImpulseMACD;

/**
 * Estrategia de Trading basada en Fusión de Volumen y Smart Money Concepts.
 *
 * Esta clase detecta y evalúa oportunidades de trading basándose en análisis
 * de volumen avanzado, filtros y gestión dinámica de riesgo. Calcula una puntuación
 * de confianza interna. Proporciona un método público que devuelve la señal completa
 * (incluyendo niveles de SL/TP sugeridos, riesgo ajustado y confianza) **solo si
 * cumple el umbral mínimo de confianza interna configurado en la estrategia**.
 *
 * NOTA IMPORTANTE: Esta estrategia NO ejecuta la orden directamente. La ejecución
 * final debe ser manejada por una capa superior (ej: una clase Algoritmo) que
 * reciba la señal sugerida y decida ejecutarla a través de tu clase centralizada
 * de gestión de órdenes (ej: setOrdersBuySel).
 *
 * Características clave:
 * - Detección de señales basadas en patrones de volumen de Smart Money.
 * - Verificación interna de la confianza de la señal contra un umbral configurado.
 * - Cálculo de parámetros de operación sugeridos (SL, TP, riesgo ajustado).
 * - Proporciona la señal completa solo si es considerada "ejecutable" por la propia estrategia.
 * - Depende de una instancia de SmartMoneyVolume para el análisis de datos de precio y volumen.
 */
class SmartMoneyFusionStrategy080525
{
    /**
     * @var SmartMoneyVolume Instancia del analizador de volumen con datos cargados.
     */
    private SmartMoneyVolume $volumeAnalyzer;

    // No hay dependencia de una clase Trader para ejecución de órdenes en esta versión.

    // --- Parámetros de Configuración ---
    /**
     * @var float Porcentaje del capital a arriesgar por operación en el nivel base de confianza (1.0 = 1%). Usado para calcular el riesgo ajustado sugerido.
     */
    private float $baseRiskPercent;

    /**
     * @var array Horas permitidas para operar en formato 24h. La estrategia solo sugerirá operaciones durante estas horas.
     */
    private array $tradingHours;

    /**
     * @var int Período para el cálculo de la Media Móvil Simple (SMA) usada como filtro de tendencia.
     */
    private int $smaPeriod;

    /**
     * @var float Multiplicador del volumen promedio (en filtros de entrada) para considerar que el volumen actual es significativo.
     */
    private float $volumeFilterMultiplier;

    /**
     * @var int Lookback para determinar la tendencia del volumen (en filtros de entrada y señales).
     */
    private int $volumeTrendLookback;

    /**
     * @var float Porcentaje de proximidad a un nivel de liquidez/perfil de volumen para considerarlo relevante para la señal/filtro.
     */
    private float $liquidityProximityPercentage;

    /**
     * @var int Período para el cálculo del Average True Range (ATR).
     */
    private int $atrPeriod;

    /**
     * @var float Multiplicador del ATR para calcular el Stop Loss inicial sugerido.
     */
    private float $slMultiplier;

    /**
     * @var float Multiplicador del ATR para calcular el Take Profit inicial sugerido.
     */
    private float $tpMultiplier;

    /**
     * @var float Multiplicador del ATR para calcular el Trailing Stop inicial sugerido.
     */
    private float $tsMultiplier;

    /**
     * @var int Umbral mínimo de confianza (0-100) que la propia estrategia considera suficiente para sugerir la ejecución.
     */
    private int $minConfidenceToSuggestExecution;


    /**
     * Constructor de la estrategia.
     *
     * @param SmartMoneyVolume $volumeAnalyzer Analizador de volumen con datos cargados.
     * @param array $config Array opcional de configuración para sobrescribir parámetros por defecto.
     * Keys reconocidas: 'baseRiskPercent', 'tradingHours', 'smaPeriod',
     * 'volumeFilterMultiplier', 'volumeTrendLookback', 'liquidityProximityPercentage',
     * 'atrPeriod', 'slMultiplier', 'tpMultiplier', 'tsMultiplier', 'minConfidenceToSuggestExecution'.
     * // Puedes añadir aquí argumentos para tus otras clases de indicadores si la estrategia los usa internamente
     * // @param RSI|null $rsiAnalyzer
     * // @param BreakOutProbability|null $probabilityAnalyzer
     * // @param ImpulseMACD|null $impulseMacd
     */
    public function __construct(
        SmartMoneyVolume $volumeAnalyzer,
        array $config = []
        // Añadir aquí los argumentos opcionales para otros indicadores
        // ?RSI $rsiAnalyzer = null,
        // ?BreakOutProbability $probabilityAnalyzer = null,
        // ?ImpulseMACD $impulseMacd = null
    ) {
        $this->volumeAnalyzer = $volumeAnalyzer;
        // Asignar otras dependencias si las hay
        // $this->rsiAnalyzer = $rsiAnalyzer;
        // $this->probabilityAnalyzer = $probabilityAnalyzer;
        // $this->impulseMacd = $impulseMacd;


        // Configuración por defecto y sobreescritura con $config
        $defaultConfig = [
            'baseRiskPercent' => 1.0, // 1% base riesgo
            'tradingHours' => [9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20], // Horario de mercado típico (ejemplo)
            'smaPeriod' => 20,
            'volumeFilterMultiplier' => 1.5, // Requiere 1.5x el volumen promedio
            'volumeTrendLookback' => 7, // Tendencia de volumen en los últimos 7 periodos
            'liquidityProximityPercentage' => 0.2, // Precio debe estar a 0.2% de un nivel clave
            'atrPeriod' => 14,
            'slMultiplier' => 1.5, // SL a 1.5 * ATR
            'tpMultiplier' => 3.0, // TP a 3.0 * ATR
            'tsMultiplier' => 1.2, // Trailing Stop a 1.2 * ATR
            'minConfidenceToSuggestExecution' => 60, // Mínima confianza interna para sugerir ejecución
        ];

        $config = array_merge($defaultConfig, $config);

        $this->baseRiskPercent = $config['baseRiskPercent'];
        $this->tradingHours = $config['tradingHours'];
        $this->smaPeriod = $config['smaPeriod'];
        $this->volumeFilterMultiplier = $config['volumeFilterMultiplier'];
        $this->volumeTrendLookback = $config['volumeTrendLookback'];
        $this->liquidityProximityPercentage = $config['liquidityProximityPercentage'];
        $this->atrPeriod = $config['atrPeriod'];
        $this->slMultiplier = $config['slMultiplier'];
        $this->tpMultiplier = $config['tpMultiplier'];
        $this->tsMultiplier = $config['tsMultiplier'];
        $this->minConfidenceToSuggestExecution = $config['minConfidenceToSuggestExecution'];
    }

    //region Evaluación Interna de la Señal (MÉTODO PRIVADO)

    /**
     * Evalúa internamente si se ha detectado una posible señal de trading
     * basándose en los datos y los criterios de la estrategia.
     * Calcula todos los parámetros de la señal y la confianza, pero no verifica
     * la confianza mínima ni sugiere la ejecución externa.
     *
     * @return array|null Información detallada de la señal (sin verificación de confianza mínima)
     * o null si no hay señal válida que cumpla los filtros básicos.
     */
    private function evaluateSignal(): ?array
    {
        // 1. Verificar hora y filtros generales de entrada
        if (!$this->isTradingHour()) {
             return null;
        }

        // Asegurarse de que hay datos suficientes antes de acceder al último precio
        if (count($this->volumeAnalyzer->prices) < 1) {
             return null;
        }
        $currentPrice = end($this->volumeAnalyzer->prices);
 
        if ($currentPrice === false) { // Debería ser redundante si el count es >= 1, pero seguro.
             return null;
        }

         // Asegurarse de que hay suficientes datos para ATR y otros cálculos
         if (count($this->volumeAnalyzer->prices) < $this->atrPeriod || count($this->volumeAnalyzer->volumes) < $this->volumeTrendLookback || count($this->volumeAnalyzer->prices) < $this->smaPeriod) {
              return null; // No hay suficientes datos para indicadores y filtros
         }
         try {
             $atr = $this->volumeAnalyzer->getATR($this->atrPeriod);
         } catch (\InvalidArgumentException $e) {
             // Log::error("Error calculando ATR en evaluateSignal: " . $e->getMessage());
             return null;
         }

        if ($atr <= 0) {
             return null; // No sugerir operación con volatilidad cero
        }
        if (!$this->passEntryFilters($currentPrice, $atr)) {
            return null;
        }
        // 2. Determinar tendencia principal
        $trend = $this->getTrendDirection();

        if ($trend === 'neutral') {
            return null;
        }

        // 3. Detectar señales específicas
        try {
            $buySignal = $this->isBuySignal($currentPrice, $atr);
            $sellSignal = $this->isSellSignal($currentPrice, $atr);
        } catch (\InvalidArgumentException $e) {
            // Log::error("Error detectando señales en evaluateSignal: " . $e->getMessage());
            return null;
        }


        // 4. Solo sugerir operación a favor de la tendencia principal
        if ($buySignal && $trend !== 'long') {
            return null;
        }
        if ($sellSignal && $trend !== 'short') {
            return null;
        }

        // 5. Si hay una señal válida a favor de la tendencia
        if ($buySignal || $sellSignal) {
            // 6. Calcular niveles y riesgo ajustado sugeridos
            $confidence = $this->calculateConfidence();
            $adjustedRisk = $this->adjustRiskByConfidence($confidence);

             // Calcular niveles de TP/SL/TS dinámicamente
             $levels = $this->calculateTradeLevels($currentPrice, $atr, $buySignal ? 'buy' : 'sell');
             $stopLoss = $levels['stop_loss'];
             $takeProfit = $levels['take_profit'];
             $trailingStop = $levels['trailing_stop'];

            // Asegurarse de que el stop loss es válido (evita división por cero en cálculo de tamaño externo)
            if (abs($currentPrice - $stopLoss) < 1e-9) {
                 return null;
            }

            // Devolver todos los datos de la señal detectada
            return [
                'signal' => $buySignal ? 'buy' : 'sell',
                'entry' => $currentPrice,
                'stop_loss' => $stopLoss,
                'take_profit' => $takeProfit,
                'trailing_stop' => $trailingStop,
                'confidence' => $confidence, // Devolver la confianza calculada
                'adjusted_risk_percent' => $adjustedRisk, // Devolver el riesgo ajustado sugerido
                'timestamp' => date('Y-m-d H:i:s'), // Usar timestamp real si está disponible
                'strategy' => 'SmartMoneyFusionStrategy', // Identificar la estrategia
            ];
        }

        return null; // No hay señal de entrada que cumpla los filtros básicos
    }

    //endregion

    //region Método Público para Obtener Señal Ejecutable Sugerida (Punto de Entrada para Algoritmo)

    /**
     * Evalúa la estrategia y devuelve la información completa de la señal
     * solo si detecta una señal válida que cumple con el umbral mínimo de confianza
     * configurado en la propia estrategia.
     *
     * Este es el método que la clase AlgoritApi debería llamar. La señal devuelta
     * incluye todos los parámetros necesarios (dirección, entrada, SL, TP, riesgo ajustado, confianza)
     * para que una capa superior (AlgoritApi) decida y ejecute la orden real a través
     * de la clase centralizada (setOrdersBuySel).
     *
     * @return array|null Un array con la información de la señal sugerida si cumple
     * la confianza mínima interna, o null en caso contrario. El array incluirá:
     * 'signal', 'entry', 'stop_loss', 'take_profit', 'trailing_stop',
     * 'confidence', 'adjusted_risk_percent', 'timestamp', 'strategy'.
     */
    public function getExecutableSignalSuggestion(): ?array
    {
        // 1. Realizar la evaluación interna para obtener la señal potencial
        $signal = $this->evaluateSignal();

        // 2. Verificar si se detectó una señal y si tiene suficiente confianza INTERNA
        if ($signal !== null && $signal['confidence'] >= $this->minConfidenceToSuggestExecution) {
            // Puedes añadir logging aquí si la estrategia detecta una señal "ejecutable" según sus criterios
            //echo "SmartMoneyFusionStrategy: Señal detectada y confianza (" . $signal['confidence'] . ") >= umbral interno (" . $this->minConfidenceToSuggestExecution . "). Sugiriendo ejecución.\n";
            return $signal; // Devolver la señal completa
        }

        // Si no hay señal o la confianza es insuficiente para la estrategia
        // echo "SmartMoneyFusionStrategy: No se detectó señal ejecutable sugerida (signal=" . ($signal ? "presente" : "null") . ", confidence=" . ($signal['confidence'] ?? 'N/A') . ").\n";
        return null;
    }

    //endregion

    //region Filtros de Entrada (Usan $this->volumeAnalyzer y posibles otras dependencias inyectadas)

    /**
     * Aplica filtros generales de entrada que deben cumplirse para cualquier operación.
     *
     * @param float $currentPrice El precio actual.
     * @param float $atr El Average True Range actual.
     * @return bool True si se pasan todos los filtros, false en caso contrario.
     * @throws \InvalidArgumentException Si no hay suficientes datos en SmartMoneyVolume para los cálculos.
     */
    private function passEntryFilters(float $currentPrice, float $atr): bool
    {
        // 1. Volumen superior a un promedio reciente
        try {
            $loopback = $this->volumeAnalyzer->getDefaultVolumeLookback();
            $avgVolume = $this->volumeAnalyzer->getAverageVolume($loopback); // Usar lookback por defecto de SmartMoneyVolume
            $currentVolume = end($this->volumeAnalyzer->volumes) * 1;

             if ($currentVolume === false || $avgVolume <= 0) {
                  return false; // Datos insuficientes o volumen promedio no válido
             }

            if ($currentVolume < $avgVolume * $this->volumeFilterMultiplier) {
                return false;
            }

        } catch (\InvalidArgumentException $e) {
             // Log::error("Error en filtro de volumen: " . $e->getMessage());
             return false; // Error de datos
        }

        // 2. Precio actual cerca de un nivel de liquidez (perfil de volumen significativo)
        try {
            $volumeProfile = $this->volumeAnalyzer->getVolumeProfile(); // Usa top 5 niveles por defecto
            $isNearLiquidityLevel = false;

            if ($currentPrice > 0) {
                 foreach ($volumeProfile as $levelPrice) {
                     $percentageDifference = abs($currentPrice - $levelPrice) / $currentPrice * 100;
                     if ($percentageDifference <= $this->liquidityProximityPercentage) {
                         $isNearLiquidityLevel = true;
                         break;
                     }
                 }
            }

            if (!$isNearLiquidityLevel) {
                return false;
            }
        } catch (\Exception $e) { // Capturar posibles errores de getVolumeProfile
             // Log::error("Error en filtro de liquidez: " . $e->getMessage());
             return false;
        }


        // 3. Tendencia de volumen alcista (requiere 'rising')
        try {
             $volumeTrend = $this->volumeAnalyzer->getVolumeTrend($this->volumeTrendLookback);
             if ($volumeTrend !== 'rising') {
                 return false;
             }
        } catch (\InvalidArgumentException $e) {
             // Log::error("Error en filtro de tendencia de volumen: " . $e->getMessage());
             return false; // Error de datos
        }

        // Puedes añadir aquí filtros basados en tus otras dependencias (RSI, MACD, Probabilidad)
        // Ejemplo:
        // if ($this->rsiAnalyzer !== null && $this->rsiAnalyzer->getValue() < 50) { return false; }


        return true; // Pasa todos los filtros
    }

    //endregion

    //region Señales de Compra y Venta (Usan $this->volumeAnalyzer y posibles otras dependencias inyectadas)

    /**
     * Detecta una señal de compra basándose en patrones de volumen/precio y niveles clave.
     *
     * @param float $currentPrice El precio actual.
     * @param float $atr El Average True Range actual.
     * @return bool True si se detecta una señal de compra, false en caso contrario.
     * @throws \InvalidArgumentException Si no hay suficientes datos en SmartMoneyVolume para los cálculos.
     */
    private function isBuySignal(float $currentPrice, float $atr): bool
    {
        try {
            // Criterios basados en SmartMoneyVolume
            if (!$this->volumeAnalyzer->detectAbsorption()) { return false; }
            if ($this->volumeAnalyzer->getLastSpikeDirection() !== 'up') { return false; }
             if (!$this->volumeAnalyzer->isAccumulation(5, 5, $this->liquidityProximityPercentage, $this->atrPeriod)) { return false; }

             // Criterio adicional: Precio cerca de un nivel de liquidez detectado específicamente
             $liquidityLevels = $this->volumeAnalyzer->calculateLiquidityLevels($this->volumeAnalyzer->liquidityPriceThresholdATR, $this->volumeAnalyzer->liquidityVolumeRatioThreshold, $this->atrPeriod);
            $isNearDetectedLiquidity = false;
             if ($currentPrice > 0) {
                 foreach ($liquidityLevels as $level) {
                      $percentageDifference = abs($currentPrice - $level['price']) / $currentPrice * 100;
                      if ($percentageDifference <= $this->liquidityProximityPercentage) {
                          $isNearDetectedLiquidity = true;
                          break;
                      }
                 }
             }
            if (!$isNearDetectedLiquidity) { return false; }

            // Puedes añadir aquí confirmaciones basadas en tus otras dependencias (RSI, MACD)
            // Ejemplo:
            // if ($this->impulseMacd !== null && !$this->impulseMacd->isBullishCrossover()) { return false; }

        } catch (\InvalidArgumentException $e) {
             // Log::error("Error en detección de señal de compra: " . $e->getMessage());
             throw $e; // Relanzar para ser manejado en evaluateSignal()
        }
        return true;
    }

    /**
     * Detecta una señal de venta basándose en patrones de volumen/precio y niveles clave.
     *
     * @param float $currentPrice El precio actual.
     * @param float $atr El Average True Range actual.
     * @return bool True si se detecta una señal de venta, false en caso contrario.
     * @throws \InvalidArgumentException Si no hay suficientes datos en SmartMoneyVolume para los cálculos.
     */
    private function isSellSignal(float $currentPrice, float $atr): bool
    {
        try {
            // Criterios basados en SmartMoneyVolume
            if (!$this->volumeAnalyzer->detectAbsorption()) { return false; }
            if ($this->volumeAnalyzer->getLastSpikeDirection() !== 'down') { return false; }
             if (!$this->volumeAnalyzer->isDistribution(5, 5, $this->liquidityProximityPercentage, $this->atrPeriod)) { return false; }

            // Criterio adicional: Precio cerca de un nivel de liquidez detectado específicamente
             $liquidityLevels = $this->volumeAnalyzer->calculateLiquidityLevels($this->volumeAnalyzer->liquidityPriceThresholdATR, $this->volumeAnalyzer->liquidityVolumeRatioThreshold, $this->atrPeriod);
            $isNearDetectedLiquidity = false;
             if ($currentPrice > 0) {
                 foreach ($liquidityLevels as $level) {
                      $percentageDifference = abs($currentPrice - $level['price']) / $currentPrice * 100;
                      if ($percentageDifference <= $this->liquidityProximityPercentage) {
                          $isNearDetectedLiquidity = true;
                          break;
                      }
                 }
             }
            if (!$isNearDetectedLiquidity) { return false; }

            // Puedes añadir aquí confirmaciones basadas en tus otras dependencias (RSI, MACD)
            // Ejemplo:
            // if ($this->impulseMacd !== null && !$this->impulseMacd->isBearishCrossover()) { return false; }

        } catch (\InvalidArgumentException $e) {
             // Log::error("Error en detección de señal de venta: " . $e->getMessage());
             throw $e; // Relanzar para ser manejado en evaluateSignal()
        }
        return true;
    }

    //endregion

    //region Análisis Técnico (Usan $this->volumeAnalyzer o dependencias inyectadas para TA)

    /**
     * Calcula la Media Móvil Simple (SMA) para un periodo dado.
     * Nota: Implementado internamente, podría refactorizarse para usar una dependencia externa de TA.
     *
     * @param int $period El número de periodos para el cálculo.
     * @return float|null El valor de la SMA, o null si no hay suficientes datos.
     * @throws \InvalidArgumentException Si el período es inválido.
     */
    private function getSMA(int $period): ?float
    {
        if ($period <= 0) { throw new \InvalidArgumentException("El período de la SMA debe ser un número positivo."); }
        $prices = $this->volumeAnalyzer->prices;
        if ($period > count($prices)) { return null; }
        $recentPrices = array_slice($prices, -$period);
        $sum = array_sum($recentPrices);
        return (count($recentPrices) > 0) ? $sum / count($recentPrices) : null; // Asegurar división por cero
    }

    /**
     * Determina la dirección de la tendencia principal utilizando una Media Móvil Simple (SMA).
     *
     * @return string 'long', 'short', o 'neutral'.
     */
    private function getTrendDirection(): string
    {
        try {
            $sma = $this->getSMA($this->smaPeriod);
        } catch (\InvalidArgumentException $e) {
            // Log::error("Error calculando SMA para tendencia: " . $e->getMessage());
            return 'neutral';
        }
        if ($sma === null) {
            return 'neutral'; // No hay suficientes datos para la SMA
        }
        $price = end($this->volumeAnalyzer->prices);
         if ($price === false) { // Debería ser redundante si evaluateSignal chequea count
             return 'neutral'; // No hay datos de precio
         }


        if ($price > $sma) {
            return 'long';
        }
        if ($price < $sma) {
            return 'short';
        }
        return 'neutral';
    }

    //endregion

    //region Gestión de Riesgo (Cálculo de Confianza y Riesgo Ajustado)

    /**
     * Calcula una puntuación de confianza para la señal detectada (0-100).
     *
     * @return int La puntuación de confianza calculada.
     * @throws \InvalidArgumentException Si no hay suficientes datos en SmartMoneyVolume para los cálculos.
     */
    private function calculateConfidence(): int
    {
         $confidence = 50;
         try {
              // Factores basados en VolumeAnalyzer
              $avgVolume = $this->volumeAnalyzer->getAverageVolume($this->volumeAnalyzer->defaultVolumeLookback);
              $currentVolume = end($this->volumeAnalyzer->volumes);
              if ($currentVolume !== false && $avgVolume > 0) {
                   if ($currentVolume > $avgVolume * 3) { $confidence += 20; }
                   if ($this->volumeAnalyzer->detectVolumeCluster()) { $confidence += 15; }
              }

              // Factor basado en ATR
              $atr = $this->volumeAnalyzer->getATR($this->atrPeriod);
              $currentPrice = end($this->volumeAnalyzer->prices);
              if ($currentPrice !== false && $atr > 0) {
                   $atrPercentageOfPrice = ($atr / $currentPrice);
                   if ($atrPercentageOfPrice < 0.002) { $confidence -= 20; }
              } else if ($atr <= 0) {
                  $confidence -= 30;
              }

              // Puedes añadir aquí factores basados en tus otras dependencias (RSI, Probabilidad, MACD)
              // Ejemplo:
              // if ($this->probabilityAnalyzer !== null && $this->probabilityAnalyzer->getBreakoutProbability() > 0.8) { $confidence += 10; }
              // if ($this->impulseMacd !== null && $this->impulseMacd->isStrongTrendConfirmation()) { $confidence += 10; }

         } catch (\InvalidArgumentException $e) {
              // Log::error("Error en cálculo de confianza (InvalidArgument): " . $e->getMessage());
              $confidence -= 25; // Penalización por error en el cálculo
         } catch (\Exception $e) {
              // Log::error("Error inesperado en cálculo de confianza: " . $e->getMessage());
              $confidence -= 25; // Penalización por error en el cálculo
         }


        // Asegurarse de que la confianza esté entre 0 y 100
        return max(0, min(100, $confidence));
    }

    /**
     * Ajusta el porcentaje de riesgo base según la puntuación de confianza.
     *
     * @param int $confidence La puntuación de confianza (0-100).
     * @return float El porcentaje de riesgo ajustado (e.g., 0.5 para 0.5%).
     */
    private function adjustRiskByConfidence(int $confidence): float
    {
        $clampedConfidence = max(0, min(100, $confidence));
        return $this->baseRiskPercent * ($clampedConfidence / 100.0);
    }

    // Eliminamos calculatePositionSize de la estrategia, ya que el Algoritmo lo hará
    // private function calculatePositionSize(...): float { ... }

    /**
     * Calcula los niveles sugeridos de Stop Loss (SL), Take Profit (TP) y Trailing Stop (TS)
     * basados en el precio de entrada y el ATR actual.
     *
     * @param float $entryPrice El precio al que se entraría en la operación.
     * @param float $atr El valor actual del Average True Range.
     * @param string $direction La dirección de la operación ('buy' o 'sell').
     * @return array Un array asociativo con 'stop_loss', 'take_profit', y 'trailing_stop' sugeridos.
     * @throws \InvalidArgumentException Si la dirección no es 'buy' ni 'sell'.
     */
    private function calculateTradeLevels(float $entryPrice, float $atr, string $direction): array
    {
        if (!in_array($direction, ['buy', 'sell'])) { throw new \InvalidArgumentException("Dirección de operación inválida: " . $direction); }

        $slDistance = $atr * $this->slMultiplier;
        $tpDistance = $atr * $this->tpMultiplier;
        $tsDistance = $atr * $this->tsMultiplier;

        if ($direction === 'buy') {
            $stopLoss = $entryPrice - $slDistance;
            $takeProfit = $entryPrice + $tpDistance;
            $trailingStop = $entryPrice - $tsDistance;
        } else { // sell
            $stopLoss = $entryPrice + $slDistance;
            $takeProfit = $entryPrice - $tpDistance;
            $trailingStop = $entryPrice + $tsDistance;
        }

        // Consideraciones de seguridad: asegurar TP/SL no son inversos o iguales a la entrada (aunque la lógica de evaluateSignal ya chequea SL vs Entry)
        // if ($direction === 'buy' && $takeProfit <= $stopLoss) { $takeProfit = $entryPrice + ($atr * $this->tpMultiplier); } // O manejar como error
        // if ($direction === 'sell' && $takeProfit >= $stopLoss) { $takeProfit = $entryPrice - ($atr * $this->tpMultiplier); } // O manejar como error


        return [
            'stop_loss' => $stopLoss,
            'take_profit' => $takeProfit,
            'trailing_stop' => $trailingStop,
        ];
    }


    /**
     * Verifica si la hora actual está dentro de las horas de trading permitidas.
     *
     * @return bool True si es hora de operar, false en caso contrario.
     */
    private function isTradingHour(): bool
    {
        $hour = (int) date('G');
        return in_array($hour, $this->tradingHours);
    }

    //endregion

    //region Método para actualizar VolumeAnalyzer (útil si el objeto estrategia es de larga vida)
    /**
     * Actualiza la instancia del SmartMoneyVolume. Útil si el objeto estrategia es de larga vida
     * y el algoritmo no recrea la estrategia en cada ciclo.
     *
     * @param SmartMoneyVolume $volumeAnalyzer La nueva instancia con datos actualizados.
     */
    public function setVolumeAnalyzer(SmartMoneyVolume $volumeAnalyzer): void
    {
        $this->volumeAnalyzer = $volumeAnalyzer;
         // Si tus otras dependencias (RSI, MACD, etc.) también necesitan actualizarse con los nuevos datos,
         // deberías añadir métodos similares en esas clases y llamarlos aquí.
         // Ejemplo: $this->rsiAnalyzer->updateData($volumeAnalyzer->prices);
    }
    //endregion

     //region Setters para otras dependencias (opcional, si las inyectas después del constructor)
     // public function setRsiAnalyzer(RSI $rsiAnalyzer): void { $this->rsiAnalyzer = $rsiAnalyzer; }
     // public function setBreakoutProbability(BreakOutProbability $analyzer): void { $this->probabilityAnalyzer = $analyzer; }
     // public function setImpulseMacd(ImpulseMACD $macd): void { $this->impulseMacd = $macd; }
     // ... etc.
     //endregion
}