<?php

namespace App;

use App\SmartMoneyVolume;
// Puedes añadir aquí los 'use' de tus otras clases de indicadores
// si la estrategia los usa internamente para sus cálculos o filtros.
// use App\RSI;
// use App\BreakOutProbability;
// use App\ImpulseMACD;

/**
 * Estrategia de Trading basada en Fusión de Volumen y Smart Money Concepts.
 *
 * Esta clase detecta y evalúa oportunidades de trading basándose en análisis
 * de volumen avanzado, filtros y gestión dinámica de riesgo. Calcula una puntuación
 * de confianza interna y puntuaciones ponderadas. Proporciona un método público
 * que devuelve la información completa de la señal potencial (incluyendo niveles
 * sugeridos, riesgo ajustado, confianza y puntuaciones ponderadas) **siempre que
 * se detecte una señal básica**, e indica con un flag si cumple el umbral interno
 * de confianza para una "sugerencia ejecutable".
 *
 * NOTA IMPORTANTE: Esta estrategia NO ejecuta la orden directamente. La ejecución
 * final debe ser manejada por una capa superior (ej: una clase Algoritmo) que
 * reciba la información de la señal potencial y decida ejecutarla, posiblemente
 * combinándola con análisis de otras estrategias/indicadores.
 *
 * Características clave:
 * - Detección de señales basadas en patrones de volumen de Smart Money.
 * - Evaluación de la confianza de la señal y cálculo de puntuaciones ponderadas.
 * - Cálculo de parámetros de operación sugeridos (SL, TP, riesgo ajustado).
 * - Proporciona información detallada de la señal potencial si se detecta alguna.
 * - Indica si la señal potencial cumple el umbral interno de "ejecutabilidad".
 * - Depende de una instancia de SmartMoneyVolume para el análisis de datos.
 */
class SmartMoneyFusionStrategy
{
    /**
     * @var SmartMoneyVolume Instancia del analizador de volumen con datos cargados.
     */
    private SmartMoneyVolume $volumeAnalyzer;

    // Parámetros de Configuración
    private float $baseRiskPercent;
    private array $tradingHours;
    private int $smaPeriod;
    private float $volumeFilterMultiplier;
    private int $volumeTrendLookback;
    private float $liquidityProximityPercentage;
    private int $atrPeriod;
    private float $slMultiplier;
    private float $tpMultiplier;
    private float $tsMultiplier;

    /**
     * @var int Umbral mínimo de confianza (0-100) que la propia estrategia considera suficiente para marcar la señal como "sugerencia ejecutable".
     */
    private int $minConfidenceToSuggestExecution;

    // Nuevos parámetros para la ponderación (opcional, puedes hardcodear si prefieres)
    private array $scoringWeights;


    /**
     * Constructor de la estrategia.
     *
     * @param SmartMoneyVolume $volumeAnalyzer Analizador de volumen con datos cargados.
     * @param array $config Array opcional de configuración para sobrescribir parámetros por defecto.
     * Keys reconocidas: ... (ver $defaultConfig y $defaultScoringWeights) ...
     * // Puedes añadir aquí argumentos para tus otras clases de indicadores si la estrategia los usa internamente
     */
    public function __construct(
        SmartMoneyVolume $volumeAnalyzer,
        array $config = []
        // Añadir aquí los argumentos opcionales para otros indicadores
    ) {
        $this->volumeAnalyzer = $volumeAnalyzer;
        // Asignar otras dependencias si las hay

        // Configuración por defecto
        $defaultConfig = [
            'baseRiskPercent' => 1.0,
            'tradingHours' => [9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20],
            'smaPeriod' => 20,
            'volumeFilterMultiplier' => 1.5,
            'volumeTrendLookback' => 7,
            'liquidityProximityPercentage' => 0.2,
            'atrPeriod' => 14,
            'slMultiplier' => 1.5,
            'tpMultiplier' => 3.0,
            'tsMultiplier' => 1.2,
            'minConfidenceToSuggestExecution' => 60,
        ];

        // Pesos por defecto para la nueva ponderación de volumen
        $defaultScoringWeights = [
            'spike_direction' => 20,
            'accumulation' => 15, // Aumento peso sugerido
            'distribution' => 15, // Aumento peso sugerido
            'undirected_spike_activity' => 5, // Peso sugerido para spike sin dirección clara (si ocurre en nivel clave, por ejemplo)
            // Puedes añadir más pesos aquí si añades más factores de volumen a la ponderación
        ];

        $config = array_merge($defaultConfig, $config);
        $this->scoringWeights = array_merge($defaultScoringWeights, $config['scoringWeights'] ?? []); // Permite sobrescribir pesos de ponderación

        $this->baseRiskPercent = $config['baseRiskPercent'];
        $this->tradingHours = $config['tradingHours'];
        $this->smaPeriod = $config['smaPeriod'];
        $this->volumeFilterMultiplier = $config['volumeFilterMultiplier'];
        $this->volumeTrendLookback = $config['volumeTrendLookback'];
        $this->liquidityProximityPercentage = $config['liquidityProximityPercentage'];
        $this->atrPeriod = $config['atrPeriod'];
        $this->slMultiplier = $config['slMultiplier'];
        $this->tpMultiplier = $config['tpMultiplier'];
        $this->tsMultiplier = $config['tsMultiplier'];
        $this->minConfidenceToSuggestExecution = $config['minConfidenceToSuggestExecution'];
    }

    //region Evaluación Interna de la Señal Potencial (MÉTODO PRIVADO)

    /**
     * Evalúa internamente si se ha detectado una posible señal de trading
     * basándose en los datos y los criterios de la estrategia.
     * Calcula todos los parámetros de la señal, la confianza y las puntuaciones ponderadas.
     * No verifica la confianza mínima para la sugerencia ejecutable.
     *
     * @return array|null Información detallada de la señal potencial (sin verificación de confianza mínima interna)
     * o null si no hay señal válida que cumpla los filtros básicos. Incluye 'confidence', 'adjusted_risk_percent',
     * 'smartmoney_long_score', 'smartmoney_short_score'.
     */
    private function evaluateSignal(): ?array
    {
        // 1. Verificar hora y filtros generales de entrada
        if (!$this->isTradingHour()) {
             return null;
        }

        if (count($this->volumeAnalyzer->prices) < max($this->atrPeriod, $this->smaPeriod, $this->volumeTrendLookback, ($this->volumeAnalyzer->defaultVolumeLookback ?? 0))) {
             return null; // No hay suficientes datos para indicadores y filtros clave
         }

        $currentPrice = end($this->volumeAnalyzer->prices);
        if ($currentPrice === false) {
             return null;
        }

         try {
             $atr = $this->volumeAnalyzer->getATR($this->atrPeriod);
         } catch (\InvalidArgumentException $e) {
             // Log::error("Error calculando ATR en evaluateSignal: " . $e->getMessage());
             return null;
         }

        if ($atr <= 0) {
             return null; // No sugerir operación con volatilidad cero
        }
        if (!$this->passEntryFilters($currentPrice, $atr)) {
            return null;
        }
        // 2. Determinar tendencia principal
        $trend = $this->getTrendDirection();
        if ($trend === 'neutral') {
            return null;
        }

        // 3. Detectar señales específicas
        try {
            $buySignal = $this->isBuySignal($currentPrice, $atr);
            $sellSignal = $this->isSellSignal($currentPrice, $atr);
        } catch (\InvalidArgumentException $e) {
            // Log::error("Error detectando señales en evaluateSignal: " . $e->getMessage());
            return null;
        }


        // 4. Solo sugerir operación a favor de la tendencia principal
        if ($buySignal && $trend !== 'long') {
            return null;
        }
        if ($sellSignal && $trend !== 'short') {
            return null;
        }

        // 5. Si hay una señal válida a favor de la tendencia
        if ($buySignal || $sellSignal) {
            // 6. Calcular confianza, riesgo ajustado y puntuaciones ponderadas
            $confidence = $this->calculateConfidence();
            $adjustedRisk = $this->adjustRiskByConfidence($confidence);
            $weightedScores = $this->calculateSmartMoneyWeightedScores(); // Calcular las nuevas puntuaciones

             // Calcular niveles de TP/SL/TS dinámicamente
             $levels = $this->calculateTradeLevels($currentPrice, $atr, $buySignal ? 'buy' : 'sell');
             $stopLoss = $levels['stop_loss'];
             $takeProfit = $levels['take_profit'];
             $trailingStop = $levels['trailing_stop'];

            // Asegurarse de que el stop loss es válido (evita división por cero en cálculo de tamaño externo)
            if (abs($currentPrice - $stopLoss) < 1e-9) {
                 return null;
            }

            // Devolver todos los datos de la señal detectada
            return [
                'signal' => $buySignal ? 1:-1, // 'buy' : 'sell',
                'entry' => $currentPrice,
                'stop_loss' => $stopLoss,
                'take_profit' => $takeProfit,
                'trailing_stop' => $trailingStop,
                'confidence' => $confidence, // Confianza original
                'adjusted_risk_percent' => $adjustedRisk, // Riesgo ajustado original
                'smartmoney_long_score' => $weightedScores['long_score'],   // Nuevas puntuaciones ponderadas
                'smartmoney_short_score' => $weightedScores['short_score'], // Nuevas puntuaciones ponderadas
                'timestamp' => date('Y-m-d H:i:s'), // Usar timestamp real si está disponible
                'strategy' => 'SmartMoneyFusionStrategy', // Identificar la estrategia
            ];
        }

        return null; // No hay señal de entrada que cumpla los filtros básicos
    }

    //endregion

    //region Método Público para Obtener Señal Potencial o Sugerencia Ejecutable

    /**
     * Evalúa la estrategia y devuelve la información completa de la señal potencial
     * si se detecta alguna señal básica, incluyendo la confianza y un flag que indica
     * si cumple el umbral interno de confianza de la estrategia para ser considerada "ejecutable".
     *
     * Este es el método que tu clase Algoritmo debería llamar. Proporciona datos
     * incluso para señales con confianza baja, permitiendo que el Algoritmo las combine
     * con otras señales para tomar la decisión final.
     *
     * @return array|null Un array con la información detallada de la señal potencial
     * si se detecta alguna (incluye el flag 'is_executable_suggestion'), o null
     * si no se detectó ninguna señal potencial básica. El array incluirá:
     * 'signal', 'entry', 'stop_loss', 'take_profit', 'trailing_stop',
     * 'confidence', 'adjusted_risk_percent', 'smartmoney_long_score',
     * 'smartmoney_short_score', 'timestamp', 'strategy', 'is_executable_suggestion'.
     */
    public function getPotentialSignal(): ?array // Renombramos el método público para mayor claridad
    {
        // 1. Realizar la evaluación interna para obtener la señal potencial
        $signal = $this->evaluateSignal();

        // 2. Si se detectó una señal potencial (incluso con baja confianza)
        if ($signal !== null) {
            // Añadir el flag que indica si cumple el umbral interno de "ejecutabilidad"
            $signal['is_executable_suggestion'] = $signal['confidence'] >= $this->minConfidenceToSuggestExecution;

            // Puedes añadir logging aquí para ver si la estrategia detectó *algo* y si lo consideró "ejecutable"
            // echo "SmartMoneyFusionStrategy: Señal potencial detectada (Confianza: " . $signal['confidence'] . ", Ejecutable Internamente: " . ($signal['is_executable_suggestion'] ? 'Sí' : 'No') . ").\n";

            return $signal; // Devolver la señal completa con el nuevo flag y las puntuaciones ponderadas
        }

        // Si no se detectó ninguna señal potencial básica (no pasó filtros de entrada, etc.)
        // echo "SmartMoneyFusionStrategy: No se detectó señal potencial básica.\n";
        return null;
    }

    //endregion

    //region Filtros de Entrada (Usan $this->volumeAnalyzer y posibles otras dependencias inyectadas)
    private function passEntryFilters(float $currentPrice, float $atr): bool
    {
        // Implementación existente...
        // 1. Volumen superior a un promedio reciente
        try {
            $loopback = $this->volumeAnalyzer->getDefaultVolumeLookback();
            $avgVolume = $this->volumeAnalyzer->getAverageVolume($loopback); // Usar lookback por defecto de SmartMoneyVolume
            $currentVolume = end($this->volumeAnalyzer->volumes) * 1;
            if ($currentVolume === false || $avgVolume <= 0) { return false; }
            if ($currentVolume < $avgVolume * $this->volumeFilterMultiplier) { return false; }
        } catch (\InvalidArgumentException $e) { return false; }

        // 2. Precio actual cerca de un nivel de liquidez
        try {
            $volumeProfile = $this->volumeAnalyzer->getVolumeProfile();
            $isNearLiquidityLevel = false;
             if ($currentPrice > 0) {
                 foreach ($volumeProfile as $levelPrice) {
                      $percentageDifference = abs($currentPrice - $levelPrice) / $currentPrice * 100;
                      if ($percentageDifference <= $this->liquidityProximityPercentage) { $isNearLiquidityLevel = true; break; }
                 }
             }
            if (!$isNearLiquidityLevel) { return false; }
        } catch (\Exception $e) { return false; }

        // 3. Tendencia de volumen alcista (requiere 'rising')
        try {
             $volumeTrend = $this->volumeAnalyzer->getVolumeTrend($this->volumeTrendLookback);
             if ($volumeTrend !== 'rising') { return false; }
        } catch (\InvalidArgumentException $e) { return false; }

        // Puedes añadir aquí filtros basados en tus otras dependencias (RSI, MACD, Probabilidad)
        // Ejemplo: if ($this->rsiAnalyzer !== null && $this->rsiAnalyzer->getValue() < 50) { return false; }

        return true;
    }

    //endregion

    //region Señales de Compra y Venta (Usan $this->volumeAnalyzer y posibles otras dependencias inyectadas)

    /**
     * Detecta una señal de compra basándose en patrones de volumen/precio y niveles clave.
     *
     * @param float $currentPrice El precio actual.
     * @param float $atr El Average True Range actual.
     * @return bool True si se detecta una señal de compra, false en caso contrario.
     * @throws \InvalidArgumentException Si no hay suficientes datos en SmartMoneyVolume para los cálculos.
     */
    private function isBuySignal(float $currentPrice, float $atr): bool
    {
        try {
            if (!$this->volumeAnalyzer->detectAbsorption()) { return false; }
            if ($this->volumeAnalyzer->getLastSpikeDirection() !== 'up') { return false; }
             if (!$this->volumeAnalyzer->isAccumulation(5, 5, $this->liquidityProximityPercentage, $this->atrPeriod)) { return false; }
             $liquidityLevels = $this->volumeAnalyzer->calculateLiquidityLevels($this->volumeAnalyzer->liquidityPriceThresholdATR, $this->volumeAnalyzer->liquidityVolumeRatioThreshold, $this->atrPeriod);
            $isNearDetectedLiquidity = false;
             if ($currentPrice > 0) {
                 foreach ($liquidityLevels as $level) {
                      $percentageDifference = abs($currentPrice - $level['price']) / $currentPrice * 100;
                      if ($percentageDifference <= $this->liquidityProximityPercentage) { $isNearDetectedLiquidity = true; break; }
                 }
             }
            if (!$isNearDetectedLiquidity) { return false; }
            // Puedes añadir aquí confirmaciones basadas en tus otras dependencias (RSI, MACD)
        } catch (\InvalidArgumentException $e) { throw $e; }
        return true;
    }

    /**
     * Detecta una señal de venta basándose en patrones de volumen/precio y niveles clave.
     *
     * @param float $currentPrice El precio actual.
     * @param float $atr El Average True Range actual.
     * @return bool True si se detecta una señal de venta, false en caso contrario.
     * @throws \InvalidArgumentException Si no hay suficientes datos en SmartMoneyVolume para los cálculos.
     */
    private function isSellSignal(float $currentPrice, float $atr): bool
    {
        try {
            if (!$this->volumeAnalyzer->detectAbsorption()) { return false; }
            if ($this->volumeAnalyzer->getLastSpikeDirection() !== 'down') { return false; }
             if (!$this->volumeAnalyzer->isDistribution(5, 5, $this->liquidityProximityPercentage, $this->atrPeriod)) { return false; }
             $liquidityLevels = $this->volumeAnalyzer->calculateLiquidityLevels($this->volumeAnalyzer->liquidityPriceThresholdATR, $this->volumeAnalyzer->liquidityVolumeRatioThreshold, $this->atrPeriod);
            $isNearDetectedLiquidity = false;
             if ($currentPrice > 0) {
                 foreach ($liquidityLevels as $level) {
                      $percentageDifference = abs($currentPrice - $level['price']) / $currentPrice * 100;
                      if ($percentageDifference <= $this->liquidityProximityPercentage) { $isNearDetectedLiquidity = true; break; }
                 }
             }
            if (!$isNearDetectedLiquidity) { return false; }
            // Puedes añadir aquí confirmaciones basadas en tus otras dependencias (RSI, MACD)
        } catch (\InvalidArgumentException $e) { throw $e; }
        return true;
    }

    //endregion

    //region Análisis Técnico (Usan $this->volumeAnalyzer o dependencias inyectadas para TA)

    /**
     * Calcula la Media Móvil Simple (SMA) para un periodo dado.
     * Nota: Implementado internamente, podría refactorizarse para usar una dependencia externa de TA.
     *
     * @param int $period El número de periodos para el cálculo.
     * @return float|null El valor de la SMA, o null si no hay suficientes datos.
     * @throws \InvalidArgumentException Si el período es inválido.
     */
    private function getSMA(int $period): ?float
    {
        if ($period <= 0) { throw new \InvalidArgumentException("El período de la SMA debe ser un número positivo."); }
        $prices = $this->volumeAnalyzer->prices;
        if ($period > count($prices)) { return null; }
        $recentPrices = array_slice($prices, -$period);
        $sum = array_sum($recentPrices);
        return (count($recentPrices) > 0) ? $sum / count($recentPrices) : null;
    }

    /**
     * Determina la dirección de la tendencia principal utilizando una Media Móvil Simple (SMA).
     *
     * @return string 'long', 'short', o 'neutral'.
     */
    private function getTrendDirection(): string
    {
        try {
            $sma = $this->getSMA($this->smaPeriod);
        } catch (\InvalidArgumentException $e) { return 'neutral'; }
        if ($sma === null) { return 'neutral'; }
        $price = end($this->volumeAnalyzer->prices);
         if ($price === false) { return 'neutral'; }

        if ($price > $sma) { return 'long'; }
        if ($price < $sma) { return 'short'; }
        return 'neutral';
    }

    //endregion

    //region Gestión de Riesgo (Cálculo de Confianza, Riesgo Ajustado y Ponderaciones)

    /**
     * Calcula una puntuación de confianza para la señal detectada (0-100).
     *
     * @return int La puntuación de confianza calculada.
     * @throws \InvalidArgumentException Si no hay suficientes datos en SmartMoneyVolume para los cálculos.
     */
    private function calculateConfidence(): int
    {
         $confidence = 50; // Punto de partida neutral
         try {
              // Factores basados en VolumeAnalyzer (ejemplos, ajustar según pruebas)
              $avgVolume = $this->volumeAnalyzer->getAverageVolume($this->volumeAnalyzer->defaultVolumeLookback);
              $currentVolume = end($this->volumeAnalyzer->volumes);
              if ($currentVolume !== false && $avgVolume > 0) {
                   if ($currentVolume > $avgVolume * 3) { $confidence += 20; }
                   if ($this->volumeAnalyzer->detectVolumeCluster()) { $confidence += 15; }
              }

              // Factor basado en ATR (ejemplo)
              $atr = $this->volumeAnalyzer->getATR($this->atrPeriod);
              $currentPrice = end($this->volumeAnalyzer->prices);
              if ($currentPrice !== false && $atr > 0) {
                   $atrPercentageOfPrice = ($atr / $currentPrice);
                   if ($atrPercentageOfPrice < 0.002) { $confidence -= 20; }
              } else if ($atr <= 0) {
                  $confidence -= 30;
              }

              // Puedes añadir aquí factores basados en tus otras dependencias (RSI, Probabilidad, MACD)
              // Ejemplo:
              // if ($this->probabilityAnalyzer !== null && $this->probabilityAnalyzer->getBreakoutProbability() > 0.8) { $confidence += 10; }
              // if ($this->impulseMacd !== null && $this->impulseMacd->isStrongTrendConfirmation()) { $confidence += 10; }

         } catch (\InvalidArgumentException $e) {
              // Log::error("Error en cálculo de confianza (InvalidArgument): " . $e->getMessage());
              $confidence -= 25; // Penalización por error en el cálculo
         } catch (\Exception $e) {
              // Log::error("Error inesperado en cálculo de confianza: " . $e->getMessage());
              $confidence -= 25; // Penalización por error en el cálculo
         }

        // Asegurarse de que la confianza esté entre 0 y 100
        return max(0, min(100, $confidence));
    }

    /**
     * Calcula puntuaciones ponderadas para la señal basadas en diferentes aspectos del análisis de volumen.
     * Estas puntuaciones son ADICIONALES a la puntuación de confianza general.
     *
     * @return array Un array asociativo con 'long_score' y 'short_score'.
     * @throws \InvalidArgumentException Si no hay suficientes datos en SmartMoneyVolume para los cálculos.
     */
    private function calculateSmartMoneyWeightedScores(): array
    {
        $longScore = 0;
        $shortScore = 0;

        try {
            // Obtener los resultados de los análisis de volumen necesarios
            $lastSpikeDirection = $this->volumeAnalyzer->getLastSpikeDirection();
            $isAccumulation = $this->volumeAnalyzer->isAccumulation(); // Asume que isAccumulation no lanza excepciones si los datos son suficientes para SmartMoneyVolume
            $isDistribution = $this->volumeAnalyzer->isDistribution(); // Asume que isDistribution no lanza excepciones
            $hasSpike = $this->volumeAnalyzer->hasSpike(); // Asume que hasSpike no lanza excepciones
            // Necesitarías más análisis de SmartMoneyVolume aquí si quieres más factores para ponderar
            // Ej: $isNearDetectedLiquidity = $this->isNearDetectedLiquidity(...); // Lógica para verificar si está cerca de niveles liquidez

            // Ponderación basada en la dirección del último spike significativo
            if ($lastSpikeDirection === 'up') {
                $longScore += $this->scoringWeights['spike_direction'];
            } elseif ($lastSpikeDirection === 'down') {
                $shortScore += $this->scoringWeights['spike_direction'];
            } else {
                // Si no hay spike direccional pero hay spike (actividad sin dirección clara)
                // Solo damos peso si el spike ocurrió, y el peso es menor y a AMBOS scores, indicando actividad ambigua.
                 // Podrías refinar esto para solo dar puntos si el spike ocurrió en un nivel clave, por ejemplo.
                 if ($hasSpike) {
                     $longScore += ($this->scoringWeights['undirected_spike_activity'] ?? 0);
                     $shortScore += ($this->scoringWeights['undirected_spike_activity'] ?? 0);
                 }
                // Si no hay spike en absoluto, no se suman puntos por spike activity.
            }

            // Ponderación basada en patrones de Acumulación/Distribución
            if ($isAccumulation) {
                $longScore += $this->scoringWeights['accumulation'];
            } elseif ($isDistribution) {
                $shortScore += $this->scoringWeights['distribution'];
            }

            // Puedes añadir aquí ponderaciones basadas en otros análisis de volumen si los tienes
            // Ejemplo: Si está cerca de un nivel de liquidez Y hubo un spike no direccional:
            // if ($isNearDetectedLiquidity && $hasSpike && $lastSpikeDirection === null) { $longScore += 5; $shortScore += 5; }


        } catch (\InvalidArgumentException $e) {
             // Log::error("Error en cálculo de scores ponderados (InvalidArgument): " . $e->getMessage());
             // Si hay un error en el cálculo de scores, devolver 0s o penalizar
             return ['long_score' => 0, 'short_score' => 0];
        } catch (\Exception $e) {
             // Log::error("Error inesperado en cálculo de scores ponderados: " . $e->getMessage());
             return ['long_score' => 0, 'short_score' => 0];
        }

        return ['long_score' => $longScore, 'short_score' => $shortScore];
    }


    /**
     * Ajusta el porcentaje de riesgo base según la puntuación de confianza.
     *
     * @param int $confidence La puntuación de confianza (0-100).
     * @return float El porcentaje de riesgo ajustado (e.g., 0.5 para 0.5%).
     */
    private function adjustRiskByConfidence(int $confidence): float
    {
        $clampedConfidence = max(0, min(100, $confidence));
        return $this->baseRiskPercent * ($clampedConfidence / 100.0);
    }

    // Eliminamos calculatePositionSize, el Algoritmo lo calculará
    // private function calculatePositionSize(...): float { ... }

    /**
     * Calcula los niveles sugeridos de Stop Loss (SL), Take Profit (TP) y Trailing Stop (TS)
     * basados en el precio de entrada y el ATR actual.
     *
     * @param float $entryPrice El precio al que se entraría en la operación.
     * @param float $atr El valor actual del Average True Range.
     * @param string $direction La dirección de la operación ('buy' o 'sell').
     * @return array Un array asociativo con 'stop_loss', 'take_profit', y 'trailing_stop' sugeridos.
     * @throws \InvalidArgumentException Si la dirección no es 'buy' ni 'sell'.
     */
    private function calculateTradeLevels(float $entryPrice, float $atr, string $direction): array
    {
        if (!in_array($direction, ['buy', 'sell'])) { throw new \InvalidArgumentException("Dirección de operación inválida: " . $direction); }

        $slDistance = $atr * $this->slMultiplier;
        $tpDistance = $atr * $this->tpMultiplier;
        $tsDistance = $atr * $this->tsMultiplier;

        if ($direction === 'buy') {
            $stopLoss = $entryPrice - $slDistance;
            $takeProfit = $entryPrice + $tpDistance;
            $trailingStop = $entryPrice - $tsDistance;
        } else { // sell
            $stopLoss = $entryPrice + $slDistance;
            $takeProfit = $entryPrice - $tpDistance;
            $trailingStop = $entryPrice + $tsDistance;
        }

        return [
            'stop_loss' => $stopLoss,
            'take_profit' => $takeProfit,
            'trailing_stop' => $trailingStop,
        ];
    }


    /**
     * Verifica si la hora actual está dentro de las horas de trading permitidas.
     *
     * @return bool True si es hora de operar, false en caso contrario.
     */
    private function isTradingHour(): bool
    {
        $hour = (int) date('G');
        return in_array($hour, $this->tradingHours);
    }

    //endregion

    //region Método para actualizar VolumeAnalyzer (útil si el objeto estrategia es de larga vida)
    /**
     * Actualiza la instancia del SmartMoneyVolume. Útil si el objeto estrategia es de larga vida
     * y el algoritmo no recrea la estrategia en cada ciclo.
     *
     * @param SmartMoneyVolume $volumeAnalyzer La nueva instancia con datos actualizados.
     */
    public function setVolumeAnalyzer(SmartMoneyVolume $volumeAnalyzer): void
    {
        $this->volumeAnalyzer = $volumeAnalyzer;
         // Si tus otras dependencias (RSI, MACD, etc.) también necesitan actualizarse con los nuevos datos,
         // deberías añadir métodos similares en esas clases y llamarlos aquí.
         // Ejemplo: $this->rsiAnalyzer->updateData($volumeAnalyzer->prices);
    }
    //endregion

     //region Setters para otras dependencias (opcional)
     // public function setRsiAnalyzer(RSI $rsiAnalyzer): void { $this->rsiAnalyzer = $rsiAnalyzer; }
     // public function setBreakoutProbability(BreakOutProbability $analyzer): void { $this->probabilityAnalyzer = $analyzer; }
     // public function setImpulseMacd(ImpulseMACD $macd): void { $this->impulseMacd = $macd; }
     // ... etc.
     //endregion
}