<?php

namespace App; // Asegúrate que el namespace sea correcto

use App\ImpulseMACDWS;         // La clase ImpulseMACDWS simple
use App\SmartMoneyVolumeSeries;    // La clase SmartMoneyVolumeSeries REFACTORIZADA para series
use App\Trader;            // Para EMA, ATR
use InvalidArgumentException;
use RuntimeException;

/**
 * Estrategia que combina ImpulseMACDWS (simple) con análisis de SmartMoneyVolumeSeries (refactorizado)
 * y un filtro de tendencia EMA(100). Calcula SL/TP basado en EMA +/- ATR.
 * Aplica filtros: Base MACD -> Fase A/D (SmartMoney) -> EMA(100).
 */
class SmartMacdStrategy
{
    //region Propiedades
    private ImpulseMACDWS $impulseMacd;
    private SmartMoneyVolumeSeries $smartMoney; // Objeto SmartMoneyVolumeSeries REFACTORIZADO
    private array $ohlcData;

    // Parámetros
    private int $emaPeriod = 100;
    private int $atrPeriod = 14;
    private float $atrMultiplierSL = 2.0;
    private float $rrRatio = 2.5;
    // Quitado $volumeTrendLookback, ya que se usa el getter de la serie precalculada

    // Indicadores internos
    private ?array $emaFilter = null;
    private ?array $atrValues = null;

    // Resultados
    private ?array $strategySignals = null;
    private bool $internalIndicatorsCalculated = false;
    //endregion

    //region Constructor
    public function __construct(
        ImpulseMACDWS $impulseMacd,
        SmartMoneyVolumeSeries $smartMoney,
        array $ohlcData,
        int $atrPeriod = 14,
        float $atrMultiplierSL = 2.0,
        float $rrRatio = 2.5
    ) {
        if ($impulseMacd->getMD() === null) { throw new InvalidArgumentException("ImpulseMACDWS debe estar calculado."); }
        if (!isset($ohlcData['high'], $ohlcData['low'], $ohlcData['close']) || empty($ohlcData['close']) || count($ohlcData['close']) !== count($ohlcData['high']) || count($ohlcData['close']) !== count($ohlcData['low'])) { throw new InvalidArgumentException("Datos OHLC incompletos/inconsistentes."); }
        if ($atrPeriod <= 0 || $atrMultiplierSL <= 0 || $rrRatio <= 0) { throw new InvalidArgumentException("Parámetros ATR/RR deben ser positivos."); }

        $this->impulseMacd = $impulseMacd;
        $this->smartMoney = $smartMoney; // Guardar el objeto SmartMoney
        $this->ohlcData = $ohlcData;
        $this->atrPeriod = $atrPeriod;
        $this->atrMultiplierSL = $atrMultiplierSL;
        $this->rrRatio = $rrRatio;

         // Importante: Asegurarse que SmartMoneyVolumeSeries calcule sus series
         // Es mejor llamarlo explícitamente antes de generar señales
         // o asumir que ya se llamó fuera. Por seguridad, lo llamamos aquí
         // si aún no se ha hecho (requiere añadir un getter de estado o hacerlo público).
         // Para este ejemplo, asumiremos que se llama fuera antes de generateSignals,
         // o podríamos añadir $this->smartMoney->calculateAllSeries(); aquí en el constructor,
         // aunque podría ser ineficiente si se instancia la estrategia muchas veces.
         // La mejor opción es llamarlo una vez fuera antes de usar generateSignals.
    }
    //endregion

    //region Cálculo Interno y Generación de Señales

    /** Calcula EMA y ATR para la estrategia. */
    private function calculateInternalIndicators(): bool
    {
        if ($this->internalIndicatorsCalculated) return true;
        try {
            $closePrices = array_filter($this->ohlcData['close'], 'is_numeric');
            $highPrices = array_filter($this->ohlcData['high'], 'is_numeric');
            $lowPrices = array_filter($this->ohlcData['low'], 'is_numeric');
            if (count($closePrices) < $this->emaPeriod || count($closePrices) < $this->atrPeriod) return false;
            $emaResult = Trader::EMA(array_values($closePrices), $this->emaPeriod);
            $this->emaFilter = $this->alignAndCleanWithReference($emaResult, $this->ohlcData['close']);
            $atrResult = Trader::ATR(array_values($highPrices), array_values($lowPrices), array_values($closePrices), $this->atrPeriod);
            $this->atrValues = $this->alignAndCleanWithReference($atrResult, $this->ohlcData['close']);
            if ($this->emaFilter === null || $this->atrValues === null) return false;
        } catch (\Exception $e) { $this->emaFilter = null; $this->atrValues = null; return false; }
        $this->internalIndicatorsCalculated = true;
        return true;
    }


    /**
     * Genera señales aplicando filtros (Base MACD -> Fase A/D -> EMA) y calculando SL/TP.
     * UTILIZA LAS SERIES PRECALCULADAS DE SmartMoneyVolumeSeries.
     *
     * @param int $confirmationPeriods Periodos de confirmación para la señal base de ImpulseMACDWS.
     * @return array|null Array [clave => ['signal'=>int, 'sl'=>float|null, 'tp'=>float|null]] o null si falla.
     */
    public function generateSignals(int $confirmationPeriods = 1): ?array
    {
        // 1. Calcular indicadores internos (EMA/ATR) si es necesario
        if (!$this->internalIndicatorsCalculated) {
            if (!$this->calculateInternalIndicators()) {
                throw new RuntimeException("No se pudieron calcular indicadores internos (EMA/ATR).");
            }
        }

        // 2. Asegurarse que SmartMoney ha calculado sus series (MEJOR HACERLO FUERA ANTES DE LLAMAR)
        // Por si acaso, lo llamamos aquí, pero es ineficiente si se llama múltiples veces.
        // Si calculateAllSeries ya se ejecutó, simplemente retornará.
         try {
              $this->smartMoney->calculateAllSeries();
         } catch (\Exception $e) {
               throw new RuntimeException("Error calculando series de SmartMoneyVolumeSeries: " . $e->getMessage());
         }


        // 3. Obtener señal base de ImpulseMACDWS
        try { $baseSignals = $this->impulseMacd->generateSignals($confirmationPeriods); }
        catch (\Exception $e) { throw new RuntimeException("Error generando señales base de ImpulseMACDWS: " . $e->getMessage()); }

        // 4. Obtener las SERIES necesarias ANTES del bucle
        $isAccumulationSeries = $this->smartMoney->getIsAccumulationSeries();
        $isDistributionSeries = $this->smartMoney->getIsDistributionSeries();
        // Obtener otras series de SmartMoney si se añaden más filtros (ej. volumen, spikes)
        // $volumeTrendSeries = $this->smartMoney->getVolumeTrendSeries();

        if ($isAccumulationSeries === null || $isDistributionSeries === null) {
             throw new RuntimeException("Las series de Fase A/D de SmartMoney no están disponibles.");
        }

        $closePrices = $this->ohlcData['close'];
        $this->strategySignals = [];
        $keys = array_keys($closePrices);

        // 5. Iterar y aplicar filtros usando las series
        foreach ($keys as $key) {
            $signalBase = $baseSignals[$key] ?? 0;
            $finalSignal = 0;
            $stopLoss = null; $takeProfit = null;

            $currentClose = $closePrices[$key] ?? null;
            $currentEma = $this->emaFilter[$key] ?? null;
            $currentAtr = $this->atrValues[$key] ?? null;

            // Datos de SmartMoney para esta barra
            $isAccumulationNow = $isAccumulationSeries[$key] ?? false; // Default a false si falta
            $isDistributionNow = $isDistributionSeries[$key] ?? false; // Default a false si falta

            // --- Aplicar Filtros Secuencialmente ---
            if ($signalBase !== 0 && $currentClose !== null && $currentEma !== null) {
                $passesPhaseFilter = false;
                $passesEmaFilter = false;

                // 1. Filtro de Fase (Usando las series obtenidas)
                if ($signalBase === 1 && $isAccumulationNow) { $passesPhaseFilter = true; }
                elseif ($signalBase === -1 && $isDistributionNow) { $passesPhaseFilter = true; }

                // 2. Filtro EMA (Solo si pasó Fase)
                if ($passesPhaseFilter) {
                    if ($signalBase === 1 && $currentClose > $currentEma) { $passesEmaFilter = true; }
                    elseif ($signalBase === -1 && $currentClose < $currentEma) { $passesEmaFilter = true; }
                }

                // 3. Determinar Señal Final
                if ($passesPhaseFilter && $passesEmaFilter) {
                    $finalSignal = $signalBase;
                }
            }

            // --- Calcular SL/TP (Lógica refinada basada en EMA +/- ATR) ---
            if (($finalSignal === 1 || $finalSignal === -1) && $currentAtr !== null && $currentAtr > 0 && $currentEma !== null && $currentClose !== null) {
                 $entryPrice = $currentClose;
                 $slDistance = $currentAtr * $this->atrMultiplierSL;
                 if ($finalSignal === 1) { // Long
                     $stopLoss = $currentEma - $slDistance; $riskAmount = $entryPrice - $stopLoss;
                     if ($riskAmount > 0) { $takeProfit = $entryPrice + ($riskAmount * $this->rrRatio); }
                     else { $stopLoss = null; $takeProfit = null; $finalSignal = 0; }
                 } else { // Short
                     $stopLoss = $currentEma + $slDistance; $riskAmount = $stopLoss - $entryPrice;
                      if ($riskAmount > 0) { $takeProfit = $entryPrice - ($riskAmount * $this->rrRatio); }
                      else { $stopLoss = null; $takeProfit = null; $finalSignal = 0; }
                 }
                 if($stopLoss !== null) $stopLoss = round($stopLoss, 5);
                 if($takeProfit !== null) $takeProfit = round($takeProfit, 5);
            } else { $stopLoss = null; $takeProfit = null; }
            // --- Fin SL/TP ---

            $this->strategySignals[$key] = [
                'signal' => $finalSignal,
                'sl' => $stopLoss,
                'tp' => $takeProfit
            ];
        }
        return $this->strategySignals;
    }

    //endregion

    //region --- Métodos de Utilidad ---
    private function cleanIndicatorOutput($indicatorOutput): array { if ($indicatorOutput === false || !is_array($indicatorOutput)) return []; $cleaned = []; foreach ($indicatorOutput as $key => $value) { if ($value === false || !is_numeric($value)) { $cleaned[$key] = null; } else { $cleaned[$key] = (float)$value; } } return $cleaned; }
    private function alignAndCleanWithReference(?array $resultArray, ?array $referenceArray): ?array { if ($resultArray === null || $referenceArray === null) return null; $cleanedResult = $this->cleanIndicatorOutput($resultArray); if (empty($referenceArray)) return []; if (empty($cleanedResult)) return array_fill_keys(array_keys($referenceArray), null); $alignedArray = []; $refKeys = array_keys($referenceArray); $refCount = count($refKeys); $resValues = array_values($cleanedResult); $resCount = count($resValues); $initialNulls = $refCount - $resCount; if ($initialNulls < 0) return null; foreach ($refKeys as $index => $key) { if ($index < $initialNulls) { $alignedArray[$key] = null; } else { $resIndex = $index - $initialNulls; $alignedArray[$key] = $resValues[$resIndex] ?? null; } } return $alignedArray; }
    //endregion

} // Fin Clase SmartMacdStrategy (Corregida para usar Series de SmartMoneyVolumeSeries)