<?php

namespace App;

use App\ResponseApi;
use App\SendTelegram;
use App\Alerts;
use App\gesSnifer;
use App\CancelPosition;
use Exception; // Import the Exception class

include 'OrdersModel.php';

/**
 * Class SetOrdersBuySel
 *
 * This class manages the process of setting buy/sell orders, including creating,
 * monitoring, and canceling orders, as well as handling alerts and notifications.
 */
class SetOrdersBuySel
{
    /**
     * @var ResponseApi
     */
    protected $apiResponse;

    /**
     * @var SendTelegram
     */
    protected $apiTelegram;

    /**
     * @var Alerts
     */
    protected $apiAlerts;

    /**
     * @var gesSnifer
     */
    protected $gesSnifer;

    /**
     * @var CancelPosition
     */
    protected $cancelPosition;

    /**
     * @var OneMaOrder
     */
    protected $oneMaOrder;

    /**
     * @var int|null
     */
    protected $maOrderId;

    /**
     * @var OneTpOrder
     */
    protected $oneTpOrder;

    /**
     * @var int|null
     */
    protected $tpOrderId;

    /**
     * @var OneSlOrder
     */
    protected $oneSlOrder;

    /**
     * @var int|null
     */
    protected $slOrderId;

    /**
     * @var float
     */
    protected $avaBalance = 0.0;

    /**
     * @var float|null
     */
    protected $maOrderPrice;

    /**
     * @var float
     */
    protected $currentPrice = 0.0;

    /**
     * @var float
     */
    protected $orderQty = 0.0;

    /**
     * @var int
     */
    protected $orderLev = 2;

    /**
     * @var string
     */
    protected $maSide = '';

    /**
     * @var string
     */
    protected $otSide = '';

    /**
     * @var string
     */
    protected $symbol = '';

    /**
     * @var string
     */
    protected $currency = '';

    /**
     * @var int
     */
    protected $status = 0;

    /**
     * @var float
     */
    protected $balance = 0.0;

    /**
     * @var int
     */
    protected $leverage = 0;

    /**
     * @var array
     */
    protected $porIni = [];

    /**
     * @var bool
     */
    protected $done = true;

    /**
     * @var int
     */
    protected $microt = 0;

    /**
     * @var float
     */
    protected $initialm = 0.0;

    /**
     * @var string|float
     */
    protected $breakevenprice = 0;

    /**
     * @var int
     */
    protected $inact = 0;

    /**
     * @var OneSlOrder|null
     */
    protected $prorder;

    /**
     * @var array|null
     */
    protected $tokentel;

    /**
     * @var int
     */
    protected $ISTP = 0;

    /**
     * @var bool
     */
    protected $snifer = false;

    /**
     * @var array
     */
    protected $retval = [
        "price" => 0,
        "avBal" => 0,
        "qty" => 0,
        "porLe" => 2,
        "porAv" => 0,
        "porTp" => 0,
        "porSm" => 0,
        "impTp" => 0,
        "impSm" => 0,
        "camSm" => false,
        "doit" => 0,
        "delor" => 0,
        "enBal" => 0,
        "tpSta" => '',
        "slSta" => '',
        "Tp" => false,
        "Sm" => false,
        'cierre' => ''
    ];

    /**
     * @var OneMaOrder|null
     */
    protected $maorder;

    /**
     * @var OneTpOrder|null
     */
    protected $tporder;

    /**
     * @var OneSlOrder|null
     */
    protected $slorder;


    /**
     * Constructor of the SetOrdersBuySel class.
     * Initializes dependencies and configurations.
     */
    public function __construct()
    {
        require_once __DIR__ . '/Settings.php';
        $params = new Settings();
        $config = $params->settings['config'];
        $this->apiResponse = new ResponseApi();
        $this->apiTelegram = new SendTelegram();
        $this->apiAlert = new Alerts();
        $this->gesSnifer = new gesSnifer("closeposrsimac");
        $this->cancelPosition = new CancelPosition();
        $this->ISTP = $config['istp'];
        $this->snifer = $config['snifer'];
    }

    /**
     * Retrieves the average price of an order by its ID.
     *
     * @param int|string $orderId The ID of the order.
     * @return float The average price of the order, or 0 if not found after several retries.
     */
    private function getOrderById($orderId): float
    {
        $times = 0;
        $price = 0.0;
        while (true) {
            try {
                $result = $this->apiResponse->getOrderById(['symbol' => $this->symbol, 'orderId' => $orderId], $times, 'fapi/v1/order');
                if (is_array($result) && array_key_exists('avgPrice', $result)) {
                    $price = floatval($result['avgPrice']);
                    break;
                } else {
                    $times++;
                    if ($times > 5) {
                        $price = 0.0;
                        break;
                    }
                }
            } catch (Exception $e) {
                $this->printLog("Error getting order by ID {$orderId}: " . $e->getMessage() . chr(10));
                $times++;
                if ($times > 5) {
                    $price = 0.0;
                    break;
                }
            }
        }
        return $price;
    }

    /**
     * Creates a main order.
     *
     * @return array An array containing the result of the order creation, including 'doit' and 'price'.
     */
    private function createMaOrder(): array
    {
        $times = 0;
        while (true) {
            try {
                $result = $this->apiResponse->createOrder($this->maorder, $times);
                if (is_array($result) && array_key_exists("orderId", $result)) {
                    $this->maOrderId = $result['orderId'];
                    sleep(5);
                    $price = $this->getOrderById($this->maOrderId);
                    if ($price > 0) {
                        $this->retval['doit'] = 1;
                        $this->retval['price'] = $price;
                    } else {
                        $this->retval['doit'] = 0;
                        $this->retval['price'] = 0;
                    }
                    break;
                } elseif (is_array($result) && array_key_exists("code", $result)) {
                    if ($result['code'] === -2019 || $result['code'] === -2022 || $result['code'] === -1102) {
                        if ($this->maorder['quantity'] <= 0.8) {
                            $this->sendAlert(6, 'FALSE', 'FALSE');
                            $this->retval['doit'] = 0;
                            $this->retval['price'] = 0;
                            break;
                        } else {
                            $this->maorder['quantity'] = round($this->maorder['quantity'] - 0.1, 1);
                            $this->orderQty = $this->maorder['quantity'];
                            $this->setQty();
                            $this->printLog('crear maorder - update qty:' . $this->maorder['quantity'] . chr(10));
                        }
                    } else {
                        $times++;
                        if ($times > 5) {
                            $this->printLog('error - crear maorder' . chr(10));
                            $this->sendAlert(3, 'FALSE', 'FALSE');
                            $this->retval['doit'] = 0;
                            $this->retval['price'] = 0;
                            break;
                        }
                    }
                } else {
                    $times++;
                    if ($times > 5) {
                        $this->printLog('error - crear maorder' . chr(10));
                        $this->sendAlert(3, 'FALSE', 'FALSE');
                        $this->retval['doit'] = 0;
                        $this->retval['price'] = 0;
                        break;
                    }
                }
            } catch (Exception $e) {
                $this->printLog("Error creating main order: " . $e->getMessage() . chr(10));
                $times++;
                if ($times > 5) {
                    $this->retval['doit'] = 0;
                    $this->retval['price'] = 0;
                    break;
                }
            }
        }
        return $this->retval;
    }

    /**
     * Creates a take profit order.
     *
     * @param int $CV The order type (e.g., BUY or SELL).
     * @return bool True if the order was created successfully, false otherwise.
     */
    private function createTpOrder(int $CV): bool
    {
        $times = 0;
        $resp = false;
        while (true) {
            try {
                $result = $this->apiResponse->createOrder($this->tporder, $times);
                if (is_array($result) && array_key_exists("orderId", $result)) {
                    $this->tpOrderId = $result['orderId'];
                    $resp = true;
                    break;
                } elseif (is_array($result) && array_key_exists("code", $result)) {
                    if ($result['code'] === -2021) {
                        $this->tporder['stopPrice'] = round($this->tporder['stopPrice'] + 0.01, 3);
                        $times++;
                        if ($times > 5) {
                            $resp = false;
                            break;
                        }
                    } else {
                        $times++;
                        if ($times > 5) {
                            $resp = false;
                            break;
                        }
                    }
                } else {
                    $times++;
                    if ($times > 5) {
                        $resp = false;
                        break;
                    }
                }
            } catch (Exception $e) {
                $this->printLog("Error creating take profit order: " . $e->getMessage() . chr(10));
                $times++;
                if ($times > 5) {
                    $resp = false;
                    break;
                }
            }
        }
        return $resp;
    }

    /**
     * Creates a stop loss order.
     *
     * @param int $CV The order type (e.g., BUY or SELL).
     * @return bool True if the order was created successfully, false otherwise.
     */
    private function createSlOrder(int $CV): bool
    {
        $times = 0;
        $resp = false;
        while (true) {
            try {
                $result = $this->apiResponse->createOrder($this->slorder, $times);
                if (is_array($result) && array_key_exists("orderId", $result)) {
                    $this->slOrderId = $result['orderId'];
                    $resp = true;
                    break;
                } elseif (is_array($result) && array_key_exists("code", $result)) {
                    if ($result['code'] === -2021) {
                        $this->slorder['stopPrice'] = round($this->slorder['stopPrice'] - 0.01, 3);
                        $times++;
                        if ($times > 3) {
                            $resp = false;
                            break;
                        }
                    } else {
                        $times++;
                        if ($times > 5) {
                            $resp = false;
                            break;
                        }
                    }
                } else {
                    $times++;
                    if ($times > 5) {
                        $resp = false;
                        break;
                    }
                }
            } catch (Exception $e) {
                $this->printLog("Error creating stop loss order: " . $e->getMessage() . chr(10));
                $times++;
                if ($times > 5) {
                    $resp = false;
                    break;
                }
            }
        }
        return $resp;
    }

    /**
     * Retrieves the status of the main position.
     *
     * @return int 1 if a position exists for the symbol, 0 otherwise.
     */
    private function getOrderStatusMa(): int
    {
        $times = 0;
        $status = 0;
        while (true) {
            try {
                $result = $this->apiResponse->getStatus(['symbol' => $this->symbol], $times);
                if (is_array($result) && array_key_exists("positions", $result)) {
                    foreach ($result['positions'] as $posit) {
                        if ($posit['symbol'] === $this->symbol) {
                            if (floatval($posit['initialMargin']) > 0) {
                                $status = 1;
                            }
                        }
                    }
                    break;
                } else {
                    $times++;
                    if ($times > 5) {
                        $status = 1;
                        break;
                    }
                }
            } catch (Exception $e) {
                $this->printLog("Error getting main order status: " . $e->getMessage() . chr(10));
                $times++;
                if ($times > 5) {
                    $status = 1;
                    break;
                }
            }
        }
        return $status;
    }

    /**
     * Retrieves the status of the take profit order.
     *
     * @return string The status of the take profit order, or "FALSE" if not found after several retries.
     */
    private function getOrderStatusTp(): string
    {
        $times = 0;
        $status = 'NEW';
        while (true) {
            try {
                $result = $this->apiResponse->getOrderById(['symbol' => $this->symbol, 'orderId' => $this->tpOrderId], $times);
                if (is_array($result) && array_key_exists("status", $result)) {
                    $status = $result['status'];
                    break;
                } else {
                    $times++;
                    if ($times > 5) {
                        $status = "FALSE";
                        break;
                    }
                }
            } catch (Exception $e) {
                $this->printLog("Error getting take profit order status: " . $e->getMessage() . chr(10));
                $times++;
                if ($times > 5) {
                    $status = "FALSE";
                    break;
                }
            }
        }
        return $status;
    }

    /**
     * Retrieves the status of the stop loss order.
     *
     * @return string The status of the stop loss order, or "FALSE" if not found after several retries.
     */
    private function getOrderStatusSl(): string
    {
        $times = 0;
        $status = 'NEW';
        while (true) {
            try {
                $result = $this->apiResponse->getOrderById(['symbol' => $this->symbol, 'orderId' => $this->slOrderId], $times);
                if (is_array($result) && array_key_exists("status", $result)) {
                    $status = $result['status'];
                    break;
                } else {
                    $times++;
                    if ($times > 5) {
                        $status = "FALSE";
                        break;
                    }
                }
            } catch (Exception $e) {
                $this->printLog("Error getting stop loss order status: " . $e->getMessage() . chr(10));
                $times++;
                if ($times > 5) {
                    $status = "FALSE";
                    break;
                }
            }
        }
        return $status;
    }

    /**
     * Manages the control of orders using WebSocket.
     *
     * @param int $CV The order type (e.g., BUY or SELL).
     * @return int 1 if the process was initiated, otherwise other values depending on the outcome.
     */
    private function cntrOrders(int $CV): int
    {
        $this->printLog("in cntlOrdersWs" . chr(10));
        $resp = false;
        $wsparams["pingInterval"] = 1800;
        $wsparams['maeprice'] = $this->maOrderPrice;
        $wsparams['maorder'] = $this->maorder;
        $wsparams['countup'] = 1800;
        $wsparams['slp'] = false;
        $wsparams['isprice'] = false;
        $wsparams['symbol'] = $this->symbol;
        $wsparams['CV'] = $CV;
        $wsparams['inact'] = $this->inact;
        $wsparams['breakevenprice'] = $this->breakevenprice;
        try {
            $resp = $this->apiResponse->getWebSocket(['connectId' => uniqid('', true)], $wsparams);
        } catch (Exception $e) {
            $this->printLog("Error with WebSocket: " . $e->getMessage() . chr(10));
            $this->retval['doit'] = 0;
            return $this->retval['doit']; // Consider a more robust error handling
        }

        if ($resp['istp']) {
            $this->retval['Tp'] = true;
            $this->retval['Sm'] = false;
        } else {
            $this->retval['Sm'] = true;
            $this->retval['Tp'] = false;
        }

        if ($resp['statusma'] !== 0) {
            $statusLi = $this->getOrderStatusMa();
        } else {
            $statusLi = $resp['statusma'];
        }

        $statusTp = $resp['resulttp'];
        $statusSm = $resp['resultsl'];
        $this->retval['tpSta'] = $statusTp;
        $this->retval['slSta'] = $statusSm;

        //$resp = $this->delMaOrder($this->maorder, true);
        if ($statusTp === 'FILLED' || $statusTp === "EXPIRED" || $statusSm === "FILLED" || $statusSm === "EXPIRED") {
            $this->printLog(" 4.- statusTp:" . $statusTp . " statusSm:" . $statusSm . " statusLi:" . $statusLi . chr(10));
            $statusLi = $this->getOrderStatusMa(); // ojo q pasa
            if ($statusLi !== 0) {
                $this->delMaOrder($this->maorder, false);
            }
        } elseif ($statusTp === "CANCELED" && $statusSm === "CANCELED") {
            $this->printLog(" 5.KO- statusTp:" . $statusTp . " statusSm:" . $statusSm . chr(10));
            if ($statusLi === 0) {
                $this->sendAlert(0, $statusTp, $statusSm);
            } else {
                $this->delMaOrder($this->maorder, false);
                $this->sendAlert(1, $statusTp, $statusSm);
            }
        } elseif ($statusTp === "FALSE" || $statusSm === "FALSE") {
            $this->printLog("SL 2.- statusTp:" . $statusTp . " statusSm:" . $statusSm . chr(10));
            if ($statusLi === 0) {
                $this->sendAlert(0, $statusTp, $statusSm);
            } else {
                $this->delMaOrder($this->maorder, false);
                $this->sendAlert(1, $statusTp, $statusSm);
            }
        }
        $this->retval['doit'] = 1;
        return $this->retval['doit'];
    }

    /**
     * Sends an alert message.
     *
     * @param int $type The type of alert.
     * @param string $statusTp The status of the take profit order.
     * @param string $statusSm The status of the stop loss order.
     */
    private function sendAlert(int $type, string $statusTp, string $statusSm): void
    {
        $tokentel = $this->apiAlert->getAlerts($type, $statusTp, $statusSm);
        $this->apiTelegram->sendAlertTelegram($tokentel);
    }

    /**
     * Retrieves stop orders for the symbol.
     *
     * @return bool True if stop orders exist, false otherwise.
     */
    private function getStopOrder(): bool
    {
        $times = 0;
        $allget = false;
        while (true) {
            try {
                $result = $this->apiResponse->getStopOrder(['symbol' => $this->symbol], $times);
                if (is_array($result) && array_key_exists("items", $result)) {
                    if (count($result["items"]) > 0) {
                        $allget = true;
                    }
                    break;
                } else {
                    $times++;
                    if ($times > 5) {
                        $allget = false;
                        break;
                    }
                }
            } catch (Exception $e) {
                $this->printLog("Error getting stop orders: " . $e->getMessage() . chr(10));
                $times++;
                if ($times > 5) {
                    $allget = false;
                    break;
                }
            }
        }
        return $allget;
    }

    /**
     * Deletes a main order.
     *
     * @param mixed $maorder The main order to delete.
     * @param bool $init Whether it's an initial deletion.
     */
    private function delMaOrder($maorder, bool $init = true): void
    {
        $resp = $this->cancelPosition->closePosition($maorder, true);
        $this->printLog('Response in delMaorder:' . $resp . chr(10));
    }

    /**
     * Retrieves the current price of the symbol.
     *
     * @return float The current price, or 0 if an error occurs.
     */
    private function getPrice(): float
    {
        $times = 0;
        while (true) {
            try {
                $result = $this->apiResponse->getPrice(['symbol' => $this->symbol], $times);
                if (is_array($result) && array_key_exists("price", $result)) {
                    $this->currentPrice = floatval($result["price"]);
                    break;
                } else {
                    $times++;
                    if ($times > 5) {
                        $this->currentPrice = 0;
                        break;
                    }
                }
            } catch (Exception $e) {
                $this->printLog("Error getting price: " . $e->getMessage() . chr(10));
                $times++;
                if ($times > 5) {
                    $this->currentPrice = 0;
                    break;
                }
            }
        }
        return $this->currentPrice;
    }

    /**
     * Sets the leverage for the symbol.
     *
     * @param int $leverage The leverage value to set.
     */
    private function setLastLeverage(int $leverage): void
    {
        $times = 0;
        $leverage = 1;  //hardcoded value
        while (true) {
            try {
                $result = $this->apiResponse->setLeverage(['symbol' => $this->symbol, 'leverage' =>  $leverage], $times);
                if (is_array($result) && array_key_exists("leverage", $result)) {
                    break;
                } else {
                    $times++;
                    if ($times > 5) {
                        $leverage = 1;
                        break;
                    }
                }
            } catch (Exception $e) {
                $this->printLog("Error setting leverage: " . $e->getMessage() . chr(10));
                $times++;
                if ($times > 5) {
                    $leverage = 1;
                    break;
                }
            }
        }
    }

    /**
     * Retrieves the last used leverage for a symbol.
     *
     * @param string $symbol The symbol to query.
     *@return int The last leverage used.
     */
    private function getLastLeverage(string $symbol): int
    {
        return $this->leverage;
    }

    /**
     * Prints a log message with a timestamp.
     *
     * @param string $from The message to log.
     */
    private function printLog(string $from): void
    {
        $date = (new \DateTime('now'))->format('Y-m-d H:i:s');
        print($date . ": " . $from);
    }

    /**
     * Sets the order quantity for all relevant orders.
     */
    private function setQty(): void
    {
        $this->maorder['quantity'] = $this->orderQty;
        $this->retval['qty'] = $this->orderQty;
    }

    /**
     * Sets the leverage for all relevant orders.
     */
    private function setLev(): void
    {
        $this->retval['porLe'] = $this->orderLev;
    }

    /**
     * Sets the order side (BUY or SELL) based on the given value.
     *
     * @param int $CV The order type (1 for BUY, 2 for SELL).
     */
    private function setSide(int $CV): void
    {
        if ($CV === 1) {
            $maside = 'BUY';
            $otside = 'SELL';
        } elseif ($CV === 2) {
            $maside = 'SELL';
            $otside = 'BUY';
        } else {
            throw new \InvalidArgumentException("Invalid CV value: $CV.  Must be 1 or 2.");
        }
        $this->maorder['side'] = $maside;
        $this->tporder['side'] = $otside;
        $this->slorder['side'] = $otside;
        $this->prorder['side'] = $otside;
        $this->maSide = $maside;
        $this->otSide = $otside;
        $this->retval['cierre'] = $otside === 'BUY' ? 'compra' : 'venta';
    }

    /**
     * Sets the stop price for take profit and stop loss orders.
     *
     * @param int $CV The order type (e.g., BUY or SELL).
     * @param float $price The reference price.
     */
    private function setStopPrice(int $CV, float $price): void
    {
        if ($CV === 1) {
            $this->tporder['stopPrice'] = round($price + (($price * $this->porIni['porTp']) / 100), 3);
            $this->slorder['stopPrice'] = round($price - (($price * $this->porIni['porSm']) / 100), 3);
        } else {
            $this->tporder['stopPrice'] = round($price - (($price * $this->porIni['porTp']) / 100), 3);
            $this->slorder['stopPrice'] = round($price + (($price * $this->porIni['porSm']) / 100), 3);
        }
        $this->prorder['stopPrice'] = $price;
        $this->retval['porAv'] = $this->porIni['porAv'];
        $this->retval['porTp'] = $this->porIni['porTp'];
        $this->retval['porSm'] = $this->porIni['porSm'];
        $this->retval['impTp'] = $this->tporder['stopPrice'];
        $this->retval['impSm'] = $this->slorder['stopPrice'];
    }

    /**
     * Retrieves the account status.
     *
     * @return int
     */
    public function getStatus(): int
    {
        $result = $this->getAccount($this->symbol, $this->currency);
        return $this->status;
    }

    /**
     * Retrieves account information, including status, balance, and leverage.
     *
     * @param string $symbol The symbol to query.
     * @param string $currency The currency.
     * @return array An array containing account information.
     */
    public function getAccount(string $symbol, string $currency): array
    {
        $this->status = 0;
        $this->balance = 0.0;
        $this->leverage = 0;
        $this->initialm = 0.0;
        $this->breakevenprice = 0;
        $times = 0;
        while (true) {
            try {
                $result = $this->apiResponse->getAccount(['symbol' => $this->symbol], $times);
                if ($result === null) {
                    return ['status' => 1, 'balance' => 0, 'leverage' => 0, 'initialm' => 0, 'breakevenprice' => 0];
                }
                if (is_array($result) && array_key_exists("totalInitialMargin", $result)) {
                    $positions = $result['positions'];
                    $assets = $result['assets'];
                    $this->balance = floatval($result['availableBalance']);
                    foreach ($positions as $posit) {
                        if ($posit['symbol'] === $symbol) {
                            if (floatval($posit['initialMargin']) > 0) {
                                $this->status = 1;
                            }
                            $this->leverage = intval($posit['leverage']);
                            $this->initialm = floatval($posit['initialMargin']);
                            $this->breakevenprice = $posit['breakEvenPrice'];
                        }
                    }
                    break;
                } else {
                    $times++;
                    if ($times > 5) {
                        $this->status = 0;
                        break;
                    }
                }
            } catch (Exception $e) {
                $this->printLog("Error getting account information: " . $e->getMessage() . chr(10));
                $times++;
                if ($times > 5) {
                    $this->status = 0;
                    break;
                }
            }
        }
        return ['status' => $this->status, 'balance' => $this->balance, 'leverage' => $this->leverage, 'initialm' => $this->initialm, 'breakevenprice' => $this->breakevenprice];
    }

    /**
     * Sets the order data and order type.
     *
     * @param array $data The order data.
     * @param int $CV The order type (1 for BUY, 2 for SELL).
     */
    public function setData(array $data, int $CV): void
    {
        $order = $data['order'];
        $order['newClientOrderId'] = uniqid();
        $this->symbol = $data['symbol'];
        $this->maorder = (array)new \OneMaOrder($order);
        $this->tporder = (array)new \OneTpOrder($order);
        $this->slorder = (array)new \OneSlOrder($order);
        $this->prorder = (array)new \OneSlOrder($order);
        $this->prorder['newClientOrderId'] = $this->prorder['newClientOrderId'] . 'C';
        $this->currency = $data['currency'];
        $this->porIni = $data['porini'];
        $this->tokentel = $data['tokentel'];
        $this->setSide($CV);
    }

    /**
     * Calculates and retrieves the available balance.
     * @return float
     */
    public function getBalanceAva(): float
    {
        if ($this->balance > 0) {
            $this->avaBalance = $this->balance;
            $this->retval['avBal'] = $this->avaBalance;
            $this->retval['enBal'] = $this->avaBalance;
            $this->avaBalance = round(($this->avaBalance * $this->porIni['porAv']), 3);
            print('avaBalance:' . $this->avaBalance . chr(10)); //quitar
            $locprice =  $this->getPrice();
            $this->orderLev = $this->leverage;
            if ($this->orderLev === 0) {
                $this->orderLev = $this->porIni['porLe'];
            }
            if ($locprice > 0) {
                $this->orderQty = round(($this->avaBalance / $locprice), 1);
                if ($this->orderQty === 0) {
                    $this->avaBalance = 0;
                };
                $this->orderQty = $this->orderQty * $this->orderLev;
            } else {
                $this->currentPrice = 0;
                $this->orderQty = 0;
                $this->avaBalance = 0;
            }
        } else {
            $this->currentPrice = 0;
            $this->orderQty = 0;
            $this->avaBalance = 0;
        }
        $this->setQty();
        $this->setLev();
        if ($this->avaBalance === 0) {
            $this->sendAlert(6, false, false);
        }
        $this->retval['avBal'] = $this->avaBalance;
        return $this->avaBalance;
    }

    /**
     * Optimizes and executes a trade based on the given order type.
     *
     * @param int $CV The order type (1 for BUY, 2 for SELL).
     * @return int
     */
    public function optTrade(int $CV): int
    {
        if ($this->getStopOrder()) {
            $this->delMaOrder(null, false);
        }
        $respMa = $this->createMaOrder();
        $doitli = $respMa['doit'];
        $this->retval['doit'] = 1;
        $this->microt = intval(round(microtime(true)));
        if ($doitli === 1) {
            $this->maOrderPrice = $respMa['price'];
            $this->setStopPrice($CV, $this->maOrderPrice);
            $respTp = $this->createTpOrder($CV);
            if ($respTp) {
                $respSl = $this->createSlOrder($CV);
                if ($respSl) {
                    $this->tokentel['init'] = 0;
                    $result = $this->getAccount($this->symbol, $this->currency);
                    $this->tokentel['resCV'] = $this->retval;
                    $this->apiTelegram->sendTrdTelegram($this->tokentel);
                    $this->inact = $this->tokentel['act'];
                    if ($this->snifer) {
                        $sArgs = [
                            "maorder" => $this->maorder,
                            "action" => "start",
                            "inrsi" => $this->tokentel['rsi'],
                            'inact' => $this->tokentel['act'],
                            'breakevenprice' => $this->breakevenprice
                        ];
                        $this->gesSnifer->gesArgument($sArgs);
                    }
                    $doit = $this->cntrOrders($CV);
                    if ($doit === 1) {
                        $this->tokentel['init'] = 1;
                        $this->tokentel['resCV'] = $this->retval;
                        $this->microt = intval(round(microtime(true)));
                        $this->ISTP = $this->retval['Tp'] === true ? 1 : 0;
                        $times = 0;
                        $result = $this->getAccount($this->symbol, $this->currency);
                        $this->tokentel['resCV']['enBal'] = round(($result['balance'] * $this->porIni['porAv']), 3);
                        $this->apiTelegram->sendTrdTelegram($this->tokentel);
                        if ($this->snifer) {
                            $sArgs = [
                                "maorder" => $this->maorder,
                                "action" => "kill",
                                "inrsi" => $this->tokentel['rsi'],
                                'inact' => $this->tokentel['act']
                            ];
                            $this->gesSnifer->gesArgument($sArgs);
                        }
                    }
                } else {
                    $this->apiTelegram->sendAlertTelegram("No se ha podido generar orden SL");
                    $this->delMaOrder($this->maorder, true);
                    $this->retval['doit'] = 0;
                }
            } else {
                $this->apiTelegram->sendAlertTelegram("No se ha podido generar orden TP");
                $this->delMaOrder($this->maorder, true);
                $this->retval['doit'] = 0;
            }
        } else {
            $this->sendAlert(3, false, false);
            $this->retval['doit'] = 0;
            $doitli = 0;
        }
        return $this->microt;
    }

     /**
     * Sends an alert message.
     *
     * @param int $type The type of alert.
     * @param string $statusTp The status of the take profit order.
     * @param string $statusSm The status of the stop loss order.
     */
    public function sendAlertOut(int $type, string $statusTp, string $statusSm): void
    {
        $tokentel = $this->apiAlert->getAlerts($type, $statusTp, $statusSm);
        $this->apiTelegram->sendAlertTelegram($tokentel);
    }

    /**
     * Retrieves the ISTP value.
     *
     * @return int
     */
    public function getISTP(): int
    {
        return $this->ISTP;
    }
}


