<?php
namespace App; // Asegúrate que el namespace coincida

use App\Trader; // O tu librería de indicadores
use InvalidArgumentException;

/**
 * Clase PineMarketDirectionIndicator (Traducción Pura)
 *
 * Implementación PHP diseñada para replicar los CÁLCULOS del script
 * PineScript basado en SMMA/ZLEMA/md/sb/sh.
 * Su objetivo es calcular y devolver las series de datos del indicador
 * para verificación contra la salida de PineScript.
 */
class PineImpulseMACD {
    //region Propiedades de Configuración (Inputs de PineScript)
    /** @var int */
    private $lengthMA; // = 34 en PineScript
    /** @var int */
    private $lengthSignal; // = 9 en PineScript
    //endregion

    //region Propiedades para Resultados Calculados
    /** @var array|null */ private $hlc3;
    /** @var array|null */ private $hi; // smma(high)
    /** @var array|null */ private $lo; // smma(low)
    /** @var array|null */ private $mi; // zlema(hlc3)
    /** @var array|null */ private $md; // Línea Principal (Market Direction)
    /** @var array|null */ private $sb; // Línea de Señal (SMA de md)
    /** @var array|null */ private $sh; // Histograma (md - sb)
    //endregion

    //region Constructor
    public function __construct(
        int $lengthMA = 34,
        int $lengthSignal = 9
    ) {
        if ($lengthMA <= 0 || $lengthSignal <= 0) {
             throw new InvalidArgumentException("Los períodos deben ser positivos.");
        }
        $this->lengthMA = $lengthMA;
        $this->lengthSignal = $lengthSignal;
        // Inicializar resultados
        $this->hlc3 = null; $this->hi = null; $this->lo = null; $this->mi = null;
        $this->md = null; $this->sb = null; $this->sh = null;
    }
    //endregion

    //region Método Principal de Cálculo
    /**
     * Calcula todos los componentes del indicador.
     *
     * @param array $data Array asociativo con 'high', 'low', 'close'. Claves deben ser consistentes.
     * @return bool True si el cálculo fue exitoso, false en caso contrario.
     */
    public function calculate(array $data): bool {
        // 1. Validaciones
        if (!isset($data['high'], $data['low'], $data['close'])) {
            // Opcional: Registrar un error en lugar de lanzar excepción si se prefiere
            // error_log("Datos de entrada incompletos.");
            return false;
        }
        $countData = count($data['close']);
        // Estimar puntos mínimos (muy conservador)
        $minPoints = max($this->lengthMA * 2, $this->lengthSignal) + 50;
        if ($countData < $minPoints) {
            // error_log("Datos insuficientes ({$countData} < {$minPoints}).");
            return false;
        }

        try {
            // 2. Calcular hlc3
            $this->hlc3 = [];
            foreach ($data['close'] as $key => $closeValue) {
                if (isset($data['high'][$key], $data['low'][$key]) && is_numeric($closeValue) && is_numeric($data['high'][$key]) && is_numeric($data['low'][$key])) {
                    $this->hlc3[$key] = ($data['high'][$key] + $data['low'][$key] + $closeValue) / 3.0;
                } else { $this->hlc3[$key] = null; }
            }
            $this->hlc3 = $this->cleanIndicatorOutput($this->hlc3);

            // 3. Calcular hi, lo, mi
            $highValues = array_values(array_filter($data['high'], 'is_numeric'));
            $lowValues = array_values(array_filter($data['low'], 'is_numeric'));
            $hlc3Values = array_values(array_filter($this->hlc3, 'is_numeric'));
            // Asegurar que tenemos suficientes valores filtrados
             if(count($highValues) < $minPoints || count($lowValues) < $minPoints || count($hlc3Values) < $minPoints) {
                  // error_log("Datos insuficientes después de filtrar nulos.");
                  return false;
             }


            $hiResult = $this->calculateSMMA($highValues, $this->lengthMA);
            $loResult = $this->calculateSMMA($lowValues, $this->lengthMA);
            $miResult = $this->calculateZLEMA($hlc3Values, $this->lengthMA);

            // Alinear resultados con las claves originales (usando hlc3 como referencia)
            $this->hi = $this->alignAndCleanWithReference($hiResult, $this->hlc3);
            $this->lo = $this->alignAndCleanWithReference($loResult, $this->hlc3);
            $this->mi = $this->alignAndCleanWithReference($miResult, $this->hlc3);

            // 4. Calcular md
            $this->md = [];
            foreach ($this->mi as $key => $miValue) {
                $hiValue = $this->hi[$key] ?? null;
                $loValue = $this->lo[$key] ?? null;
                $mdValue = null;
                if (is_numeric($miValue) && is_numeric($hiValue) && is_numeric($loValue)) {
                    if ($miValue > $hiValue) { $mdValue = $miValue - $hiValue; }
                    elseif ($miValue < $loValue) { $mdValue = $miValue - $loValue; }
                    else { $mdValue = 0.0; }
                }
                $this->md[$key] = $mdValue;
            }

            // 5. Calcular sb (SMA de md) y sh (md - sb)
            $mdValues = array_values(array_filter($this->md, 'is_numeric'));
             // Necesitamos suficientes datos para la SMA de señal
             if (count($mdValues) < $this->lengthSignal) {
                 // error_log("Datos 'md' insuficientes para calcular 'sb'.");
                 // Asignar null a sb y sh
                 $this->sb = array_fill_keys(array_keys($this->md), null);
                 $this->sh = array_fill_keys(array_keys($this->md), null);
                 return true; // El cálculo hasta md fue posible
             }

            $sbResult = Trader::SMA($mdValues, $this->lengthSignal);
            $this->sb = $this->alignAndCleanWithReference($sbResult, $this->md); // Alinear con MD

            $this->sh = [];
            foreach ($this->md as $key => $mdValue) {
                $sbValue = $this->sb[$key] ?? null;
                $shValue = null;
                if (is_numeric($mdValue) && is_numeric($sbValue)) {
                    $shValue = $mdValue - $sbValue;
                }
                $this->sh[$key] = $shValue;
            }

        } catch (\Exception $e) {
            // Capturar cualquier excepción durante el cálculo
            // error_log("Error en cálculo de PineMarketDirectionIndicator: " . $e->getMessage());
            // Asegurarse que las propiedades queden nulas o vacías
             $this->hlc3 = null; $this->hi = null; $this->lo = null; $this->mi = null;
             $this->md = null; $this->sb = null; $this->sh = null;
            return false;
        }

        return true; // Cálculo completado (aunque algunas series puedan contener nulls)
    }
    //endregion

    //region --- Implementaciones Indicadores Personalizados ---
    /** Calcula la Smoothed Moving Average (SMMA / RMA). */
    private function calculateSMMA(array $source, int $length): array {
        $output = []; $count = count($source);
        if ($count < $length) { return array_fill(0, $count, null); }
        $sum = 0;
        for ($i = 0; $i < $length; $i++) { if (!is_numeric($source[$i])) return array_fill(0, $count, null); $sum += $source[$i]; }
        $output[$length - 1] = $sum / $length;
        for ($i = $length; $i < $count; $i++) {
             if (!is_numeric($source[$i])) { $output[$i] = null; continue; }
             if (!isset($output[$i-1]) || $output[$i-1] === null) { $output[$i] = null; continue; } // Necesita valor previo
             $output[$i] = ($output[$i - 1] * ($length - 1) + $source[$i]) / $length;
        }
        // Rellenar nulos iniciales (esto asegura tamaño correcto)
         for ($i = 0; $i < $length - 1; $i++) {
             $output[$i] = null; // Usar índice correcto
         }
         ksort($output); // Ordenar por índice numérico

        return $output;
    }

    /** Calcula la Zero-Lag Exponential Moving Average (ZLEMA). */
    private function calculateZLEMA(array $source, int $length): array {
        $count = count($source);
        $ema1Result = Trader::EMA($source, $length);
        $ema1 = $this->cleanIndicatorOutput($ema1Result); // Array 0..N-1 con nulls
        if (count(array_filter($ema1, 'is_numeric')) === 0) return array_fill(0, $count, null);

        $ema1ValuesOnly = array_values(array_filter($ema1, 'is_numeric'));
        if (empty($ema1ValuesOnly)) return array_fill(0, $count, null);

        $ema2Result = Trader::EMA($ema1ValuesOnly, $length);
        // Alinear ema2 con la estructura de ema1 (que ya está alineado con source)
        $ema2 = $this->alignAndCleanWithReference($ema2Result, $ema1);

        $zlema = [];
        for($i=0; $i < $count; $i++) {
            $e1 = $ema1[$i] ?? null;
            $e2 = $ema2[$i] ?? null; // ema2 ya está alineado y limpiado por alignAndClean
            $zl = null;
            if (is_numeric($e1) && is_numeric($e2)) {
                $d = $e1 - $e2;
                $zl = $e1 + $d;
            }
            $zlema[$i] = $zl;
        }
        return $zlema; // Devuelve array indexado 0..N-1
    }
    //endregion

    //region Métodos Getters para acceder a los resultados
    // Getters para las series principales (md, sb, sh)
    public function getMD(): ?array { return $this->md; }
    public function getSB(): ?array { return $this->sb; }
    public function getSH(): ?array { return $this->sh; }
    // Getters para componentes intermedios (útil para debug)
    public function getMI(): ?array { return $this->mi; }
    public function getHI(): ?array { return $this->hi; }
    public function getLO(): ?array { return $this->lo; }
    public function getHLC3(): ?array { return $this->hlc3; }
    //endregion

    //region Métodos de Utilidad (Incluidos)
    /** Limpia la salida de funciones de indicadores. */
    private function cleanIndicatorOutput($indicatorOutput): array {
        if ($indicatorOutput === false || !is_array($indicatorOutput)) { return []; } $cleaned = [];
        foreach ($indicatorOutput as $key => $value) { if ($value === false) { $cleaned[$key] = null; } elseif (is_numeric($value)) { $cleaned[$key] = (float)$value; } else { $cleaned[$key] = null; } }
        return $cleaned;
    }
     /** Alinea un array resultado con uno de referencia. */
     private function alignAndCleanWithReference($resultArray, ?array $referenceArray): array {
        if ($resultArray === false || !is_array($resultArray) || $referenceArray === null || empty($referenceArray)) { return array_fill_keys(array_keys($referenceArray ?? []), null); }
        $cleanedResult = $this->cleanIndicatorOutput($resultArray); if (empty($cleanedResult)) { return array_fill_keys(array_keys($referenceArray), null); }
        $alignedArray = []; $refKeys = array_keys($referenceArray); $resValues = array_values($cleanedResult); $resCount = count($resValues); $refCount = count($refKeys); $offset = $refCount - $resCount;
        foreach ($refKeys as $index => $key) { $resIndex = $index - $offset; if ($resIndex >= 0 && isset($resValues[$resIndex])) { $alignedArray[$key] = $resValues[$resIndex]; } else { $alignedArray[$key] = null; } }
        return $alignedArray; // Devuelve array con las claves del array de referencia
     }
    //endregion

} // Fin Clase PineMarketDirectionIndicator (Traducción Pura)