<?php

namespace App; // Asegúrate que el namespace sea correcto

use App\ImpulseMACDWS; // La clase ImpulseMACDWS (versión simple O completa, esta estrategia usa su MD)
use App\Trader;       // Para EMA, ATR, SMA (usado para calcular umbrales estáticos aquí)
use InvalidArgumentException;
use RuntimeException;

/**
 * Estrategia que combina ImpulseMACDWS con filtros adicionales aplicados internamente:
 * 1. Calcula umbrales estáticos (promedio MD pos/neg) para filtro OB/OS.
 * 2. Calcula EMA(100) para filtro de tendencia.
 * 3. Calcula ATR para SL/TP.
 * 4. Aplica filtros secuencialmente: Señal Base MACD -> Filtro OB/OS -> Filtro EMA.
 * 5. Calcula SL basado en EMA +/- ATR y TP con R:R fijo para señales finales.
 */
class ImpulseMacdStrategy // Nombre consistente con el código que te dí
{
    //region Propiedades
    private ImpulseMACDWS $impulseMacd; // Objeto ImpulseMACDWS ya calculado
    private array $ohlcData;

    // Parámetros de la Estrategia
    private int $emaPeriod = 100;
    private int $atrPeriod = 14;
    private float $atrMultiplierSL = 2.0;
    private float $rrRatio = 2.5;

    // Indicadores y umbrales calculados INTERNAMENTE en esta clase
    private ?array $emaFilter = null;           // EMA(100) de Cierre
    private ?array $atrValues = null;           // ATR(14)
    private ?float $globalAvgPositiveMD = null; // Promedio MD Positivo (calculado aquí)
    private ?float $globalAvgNegativeMD = null; // Promedio MD Negativo (calculado aquí)

    // Resultados de la estrategia
    private ?array $strategySignals = null;
    private bool $internalCalculationsDone = false;
    //endregion

    //region Constructor
    public function __construct(
        ImpulseMACDWS $impulseMacd, // Recibe la instancia calculada
        array $ohlcData,
        int $atrPeriod = 14,
        float $atrMultiplierSL = 2.0,
        float $rrRatio = 2.5
    ) {
        if ($impulseMacd->getMD() === null) { throw new InvalidArgumentException("ImpulseMACDWS debe estar calculado."); }
        if (!isset($ohlcData['high'], $ohlcData['low'], $ohlcData['close']) || empty($ohlcData['close']) || count($ohlcData['close']) !== count($ohlcData['high']) || count($ohlcData['close']) !== count($ohlcData['low'])) { throw new InvalidArgumentException("Datos OHLC incompletos/inconsistentes."); }
        if ($atrPeriod <= 0 || $atrMultiplierSL <= 0 || $rrRatio <= 0) { throw new InvalidArgumentException("Parámetros ATR/RR deben ser positivos."); }

        $this->impulseMacd = $impulseMacd;
        $this->ohlcData = $ohlcData;
        $this->atrPeriod = $atrPeriod;
        $this->atrMultiplierSL = $atrMultiplierSL;
        $this->rrRatio = $rrRatio;
    }
    //endregion

    //region Cálculo Interno y Generación de Señales

    /** Calcula EMA, ATR y los promedios globales de MD para la estrategia. */
    private function calculateInternalIndicatorsAndThresholds(): bool
    {
        if ($this->internalCalculationsDone) return true; // Evitar recalcular
        try {
            // --- Calcular EMA y ATR ---
            $closePrices = array_filter($this->ohlcData['close'], 'is_numeric');
            $highPrices = array_filter($this->ohlcData['high'], 'is_numeric');
            $lowPrices = array_filter($this->ohlcData['low'], 'is_numeric');
            if (count($closePrices) < $this->emaPeriod || count($closePrices) < $this->atrPeriod) return false;

            $emaResult = Trader::EMA(array_values($closePrices), $this->emaPeriod);
            $this->emaFilter = $this->alignAndCleanWithReference($emaResult, $this->ohlcData['close']);

            $atrResult = Trader::ATR(array_values($highPrices), array_values($lowPrices), array_values($closePrices), $this->atrPeriod);
            $this->atrValues = $this->alignAndCleanWithReference($atrResult, $this->ohlcData['close']);

            if ($this->emaFilter === null || $this->atrValues === null) {
                // Opcional: Log error
                return false; // Fallo EMA o ATR
            }

            // --- Calcular Umbrales Estáticos (Promedios Globales MD) ---
            $mdSeries = $this->impulseMacd->getMD(); // Obtener MD de la clase base
            if (!is_array($mdSeries) || empty($mdSeries)) {
                 $this->globalAvgPositiveMD = null; $this->globalAvgNegativeMD = null;
                 // Continuar, pero el filtro OB/OS no se aplicará
            } else {
                 $this->calculateGlobalMDAveragesInternal($mdSeries); // Llamar al cálculo
            }

        } catch (\Exception $e) {
            // Resetear todo si falla
            $this->emaFilter = null; $this->atrValues = null;
            $this->globalAvgPositiveMD = null; $this->globalAvgNegativeMD = null;
            // Opcional: log $e->getMessage()
            return false;
        }
        $this->internalCalculationsDone = true;
        return true;
    }

    /** Lógica interna para calcular los promedios globales de la serie MD. */
    private function calculateGlobalMDAveragesInternal(array $mdSeries): void {
        $this->globalAvgPositiveMD = null; $this->globalAvgNegativeMD = null;
        $positiveMdValues = []; $negativeMdValues = [];
        // Usar array_filter para simplificar la obtención de valores numéricos
        $numericMdValues = array_filter($mdSeries, 'is_numeric');

        foreach ($numericMdValues as $value) {
           if ($value > 0) { $positiveMdValues[] = $value; }
           elseif ($value < 0) { $negativeMdValues[] = $value; }
        }

        $countPos = count($positiveMdValues);
        if ($countPos > 0) {
            // Usar array_values porque Trader::SMA espera índices 0..N-1
            $posSmaResult = Trader::SMA(array_values($positiveMdValues), $countPos);
            if ($posSmaResult !== false && !empty($posSmaResult)) {
                 $lastValue = end($posSmaResult);
                 if (is_numeric($lastValue)) { $this->globalAvgPositiveMD = (float)$lastValue; }
            }
        }
        $countNeg = count($negativeMdValues);
        if ($countNeg > 0) {
            $negSmaResult = Trader::SMA(array_values($negativeMdValues), $countNeg);
             if ($negSmaResult !== false && !empty($negSmaResult)) {
                 $lastValue = end($negSmaResult);
                 if (is_numeric($lastValue)) { $this->globalAvgNegativeMD = (float)$lastValue; }
             }
        }
    }

    /**
     * Genera señales aplicando filtros (Base MACD -> OB/OS -> EMA) y calculando SL/TP.
     * @param int $confirmationPeriods Periodos de confirmación para la señal base de ImpulseMACDWS.
     * @return array|null Array [clave => ['signal'=>int, 'sl'=>float|null, 'tp'=>float|null]] o null si falla.
     */
    public function generateSignals(int $confirmationPeriods = 1): ?array
    {
        // 1. Calcular indicadores y umbrales internos si es necesario
        if (!$this->internalCalculationsDone) {
            if (!$this->calculateInternalIndicatorsAndThresholds()) {
                throw new RuntimeException("No se pudieron calcular indicadores/umbrales internos de la estrategia.");
            }
        }

        // 2. Obtener señal base de ImpulseMACDWS
        try { $baseSignals = $this->impulseMacd->generateSignals($confirmationPeriods); }
        catch (\Exception $e) { throw new RuntimeException("Error generando señales base de ImpulseMACDWS: " . $e->getMessage()); }

        // 3. Obtener series necesarias
        $mdSeries = $this->impulseMacd->getMD(); // Necesario para filtro OB/OS
        $closePrices = $this->ohlcData['close'];
        $this->strategySignals = [];
        $keys = array_keys($closePrices);

        // 4. Iterar y aplicar filtros
        foreach ($keys as $key) {
            $signalBase = $baseSignals[$key] ?? 0;
            $finalSignal = 0;
            $stopLoss = null; $takeProfit = null;

            // Datos de la barra actual
            $currentClose = $closePrices[$key] ?? null;
            $currentEma = $this->emaFilter[$key] ?? null;
            $currentMd = $mdSeries[$key] ?? null; // MD de ImpulseMACDWS
            $currentAtr = $this->atrValues[$key] ?? null;

            // --- Aplicar Filtros Secuencialmente ---
            if ($signalBase !== 0 && $currentClose !== null && $currentEma !== null && $currentMd !== null) {
                $passesObOs = false;
                $passesEma = false;

                // 1. Filtro OB/OS (Usando umbrales calculados en esta clase)
                if ($signalBase === 1) { // Señal Larga Base
                    // Pasa si AvgNegMD es null (no se pudo calcular) O si MD < AvgNegMD (Reversión OS)
                    if ($this->globalAvgNegativeMD === null || $currentMd < $this->globalAvgNegativeMD) {
                         $passesObOs = true;
                    }
                } else { // Señal Corta Base (signalBase === -1)
                     // Pasa si AvgPosMD es null O si MD > AvgPosMD (Reversión OB)
                     if ($this->globalAvgPositiveMD === null || $currentMd > $this->globalAvgPositiveMD) {
                         $passesObOs = true;
                     }
                }

                // 2. Filtro EMA (Tendencia) - Solo si pasó OB/OS
                if ($passesObOs) {
                    if ($signalBase === 1 && $currentClose > $currentEma) { $passesEma = true; }
                    elseif ($signalBase === -1 && $currentClose < $currentEma) { $passesEma = true; }
                }

                // 3. Determinar Señal Final
                if ($passesObOs && $passesEma) {
                    $finalSignal = $signalBase; // Pasa todos los filtros
                }
            } // --- Fin Aplicación Filtros ---

            // --- Calcular SL/TP (Lógica refinada basada en EMA +/- ATR) ---
            if (($finalSignal === 1 || $finalSignal === -1) && $currentAtr !== null && $currentAtr > 0 && $currentEma !== null && $currentClose !== null) {
                 $entryPrice = $currentClose;
                 $slDistance = $currentAtr * $this->atrMultiplierSL;
                 if ($finalSignal === 1) { // Long
                     $stopLoss = $currentEma - $slDistance; $riskAmount = $entryPrice - $stopLoss;
                     if ($riskAmount > 1e-9) { // Evitar división por cero o riesgo negativo
                         $takeProfit = $entryPrice + ($riskAmount * $this->rrRatio);
                     } else { $stopLoss = null; $takeProfit = null; $finalSignal = 0; } // Invalidar si SL >= Entrada
                 } else { // Short
                     $stopLoss = $currentEma + $slDistance; $riskAmount = $stopLoss - $entryPrice;
                      if ($riskAmount > 1e-9) {
                          $takeProfit = $entryPrice - ($riskAmount * $this->rrRatio);
                      } else { $stopLoss = null; $takeProfit = null; $finalSignal = 0; } // Invalidar si SL <= Entrada
                 }
                 // Redondeo opcional
                 if($stopLoss !== null) $stopLoss = round($stopLoss, 5); // Ajustar precisión
                 if($takeProfit !== null) $takeProfit = round($takeProfit, 5);
            } else {
                 // Si no hay señal final o datos insuficientes para SL/TP
                 $stopLoss = null; $takeProfit = null;
                 // No invalidamos la señal aquí si solo falta ATR/EMA, pero SL/TP serán null
            } // --- Fin Cálculo SL/TP ---

            $this->strategySignals[$key] = [
                'signal' => $finalSignal,
                'sl' => $stopLoss,
                'tp' => $takeProfit
            ];
        }
        return $this->strategySignals;
    }
    //endregion

    //region --- Métodos de Utilidad ---
    /** Limpia la salida de funciones de indicadores. */
    private function cleanIndicatorOutput($indicatorOutput): array {
        if ($indicatorOutput === false || !is_array($indicatorOutput)) { return []; }
        $cleaned = [];
        foreach ($indicatorOutput as $key => $value) {
            if ($value === false || !is_numeric($value)) { $cleaned[$key] = null; }
            else { $cleaned[$key] = (float)$value; }
        } return $cleaned;
    }

    /** Alinea un array resultado con uno de referencia. */
     private function alignAndCleanWithReference(?array $resultArray, ?array $referenceArray): ?array {
         if ($resultArray === null || $referenceArray === null) return null;
         $cleanedResult = $this->cleanIndicatorOutput($resultArray);
         if (empty($referenceArray)) return [];
         if (empty($cleanedResult)) return array_fill_keys(array_keys($referenceArray), null);
         $alignedArray = []; $refKeys = array_keys($referenceArray); $refCount = count($refKeys);
         $resValues = array_values($cleanedResult); $resCount = count($resValues);
         $initialNulls = $refCount - $resCount;
         if ($initialNulls < 0) { /* Log error */ return null; }
         foreach ($refKeys as $index => $key) {
             if ($index < $initialNulls) { $alignedArray[$key] = null; }
             else { $resIndex = $index - $initialNulls; $alignedArray[$key] = $resValues[$resIndex] ?? null; }
         } return $alignedArray;
     }
    //endregion

} // Fin Clase ImpulseMacdStrategy (Versión con filtros OB/OS y EMA internos)