<?php

namespace App; // Asegúrate que el namespace sea correcto

use App\ImpulseMACDWS; // La clase ImpulseMACDWS (versión simple O completa, esta estrategia usa su MD)
use App\Trader;       // Para EMA, ATR, SMA (usado para calcular umbrales estáticos aquí)
use InvalidArgumentException;
use RuntimeException;

/**
 * Estrategia que combina ImpulseMACDWS con filtros adicionales aplicados internamente:
 * 1. Calcula EMA(100) para filtro de tendencia.
 * 2. Calcula ATR para SL/TP.
 * 3. Aplica filtros secuencialmente: Señal Base MACD -> Filtro EMA.
 * 4. Calcula SL basado en EMA +/- ATR y TP con R:R fijo para señales finales.
 */
class ImpulseMacdStrategy
{
    //region Propiedades
    private ImpulseMACDWS $impulseMacd; // Objeto ImpulseMACDWS ya calculado
    private array $ohlcData;

    // Parámetros de la Estrategia
    private int $emaPeriod = 100;
    private int $atrPeriod = 14;
    private float $atrMultiplierSL = 2.0;
    private float $rrRatio = 1.5; // probado 2.5 tp muy alto;

    // Indicadores y umbrales calculados INTERNAMENTE en esta clase
    private ?array $emaFilter = null;           // EMA(100) de Cierre
    private ?array $atrValues = null;           // ATR(14)
    private ?float $globalAvgPositiveMD = null; // Promedio MD Positivo (calculado aquí)
    private ?float $globalAvgNegativeMD = null; // Promedio MD Negativo (calculado aquí)

    // Resultados de la estrategia
    private ?array $strategySignals = null;
    private bool $internalCalculationsDone = false;
    //endregion

    //region Constructor
    public function __construct(
        ImpulseMACDWS $impulseMacd, // Recibe la instancia calculada
        array $ohlcData,
        int $atrPeriod = 14,
        float $atrMultiplierSL = 2.0,
        float $rrRatio = 1.5 // probado 2.5 tp muy alto
    ) {
        if ($impulseMacd->getMD() === null) { throw new InvalidArgumentException("ImpulseMACDWS debe estar calculado."); }
        if (!isset($ohlcData['high'], $ohlcData['low'], $ohlcData['close']) || empty($ohlcData['close']) || count($ohlcData['close']) !== count($ohlcData['high']) || count($ohlcData['close']) !== count($ohlcData['low'])) { throw new InvalidArgumentException("Datos OHLC incompletos/inconsistentes."); }
        if ($atrPeriod <= 0 || $atrMultiplierSL <= 0 || $rrRatio <= 0) { throw new InvalidArgumentException("Parámetros ATR/RR deben ser positivos."); }

        $this->impulseMacd = $impulseMacd;
        $this->ohlcData = $ohlcData;
        $this->atrPeriod = $atrPeriod;
        $this->atrMultiplierSL = $atrMultiplierSL;
        $this->rrRatio = $rrRatio;
    }
    //endregion

    //region Cálculo Interno y Generación de Señales

    /** Calcula EMA, ATR y los promedios globales de MD para la estrategia. */
    private function calculateInternalIndicatorsAndThresholds(): bool
    {
        if ($this->internalCalculationsDone) return true; // Evitar recalcular
        try {
            // --- Calcular EMA y ATR ---
            $closePrices = array_filter($this->ohlcData['close'], 'is_numeric');
            $highPrices = array_filter($this->ohlcData['high'], 'is_numeric');
            $lowPrices = array_filter($this->ohlcData['low'], 'is_numeric');
            if (count($closePrices) < $this->emaPeriod || count($closePrices) < $this->atrPeriod) return false;

            $emaResult = Trader::EMA(array_values($closePrices), $this->emaPeriod);
            $this->emaFilter = $this->alignAndCleanWithReference($emaResult, $this->ohlcData['close']);

            $atrResult = Trader::ATR(array_values($highPrices), array_values($lowPrices), array_values($closePrices), $this->atrPeriod);
            $this->atrValues = $this->alignAndCleanWithReference($atrResult, $this->ohlcData['close']);

            if ($this->emaFilter === null || $this->atrValues === null) {
                // Opcional: Log error
                return false; // Fallo EMA o ATR
            }

            // --- Calcular Umbrales Estáticos (Promedios Globales MD) ---
            // Se ha pasado a ImpulseMACDWS

        } catch (\Exception $e) {
            // Resetear todo si falla
            $this->emaFilter = null; $this->atrValues = null;
            $this->globalAvgPositiveMD = null; $this->globalAvgNegativeMD = null;
            // Opcional: log $e->getMessage()
            return false;
        }
        $this->internalCalculationsDone = true;
        return true;
    }

    /** Lógica interna para calcular los promedios globales de la serie MD. */
    // Se ha pasado a ImpulseMACDWS
    // Metodo: calculateGlobalMDAveragesInternal


    /**
     * Genera señales aplicando filtro (EMA) y calculando SL/TP.
     * @param int $confirmationPeriods Periodos de confirmación para la señal base de ImpulseMACDWS.
     * @return array|null Array [clave => ['signal'=>int, 'sl'=>float|null, 'tp'=>float|null]] o null si falla.
     */
    public function generateSignals(int $confirmationPeriods = 1): ?array
    {
        // 1. Calcular indicadores y umbrales internos si es necesario
        if (!$this->internalCalculationsDone) {
            if (!$this->calculateInternalIndicatorsAndThresholds()) {
                throw new RuntimeException("No se pudieron calcular indicadores/umbrales internos de la estrategia.");
            }
        }

        // 2. Obtener señal base de ImpulseMACDWS
        try { 
            $baseSignals = $this->impulseMacd->generateSignals($confirmationPeriods); 
        }catch (\Exception $e) { 
            throw new RuntimeException("Error generando señales base de ImpulseMACDWS: " . $e->getMessage());
        }

        // 3. Obtener series necesarias
        $mdSeries = $this->impulseMacd->getMD();
        $currentImpulse = $this->impulseMacd->getLastaDataOfSeries();
        $closePrices = $this->ohlcData['close'];
        $this->strategySignals = [];
        $keys = array_keys($closePrices);

        // 4. Iterar y aplicar filtros
        foreach ($keys as $key) {
            $signalBase = $baseSignals[$key] ?? 0;
            $finalSignal = 0;
            $stopLoss = null; $takeProfit = null;

            // Datos de la barra actual
            $currentClose = $closePrices[$key] ?? null;
            $currentEma = $this->emaFilter[$key] ?? null;
            $currentAtr = $this->atrValues[$key] ?? null;

            // --- Aplicar Filtro EMA ---
            if ($signalBase !== 0 && $currentClose !== null && $currentEma !== null) {
                $passesEma = false;

                // 1. Filtro EMA (Tendencia)
                if ($signalBase === 1 && $currentClose > $currentEma) { 
                    $passesEma = true; 
                }elseif ($signalBase === -1 && $currentClose < $currentEma) {
                     $passesEma = true; 
                }
                // 2. Determinar Señal Final
                if ($passesEma) {
                    $finalSignal = $signalBase; // Pasa el filtro
                }
            } // --- Fin Aplicación Filtro ---

            // --- Calcular SL/TP (Lógica refinada basada en EMA +/- ATR) ---
            if (($finalSignal === 1 || $finalSignal === -1) && $currentAtr !== null && $currentAtr > 0 && $currentEma !== null && $currentClose !== null) {
                 $entryPrice = $currentClose;
                 $slDistance = $currentAtr * $this->atrMultiplierSL;
                 if ($finalSignal === 1) { // Long
                    $stopLoss = $currentEma - $slDistance; 
                    $riskAmount = $entryPrice - $stopLoss;
                    if ($riskAmount > 1e-9) { // Evitar división por cero o riesgo negativo
                        $takeProfit = $entryPrice + ($riskAmount * $this->rrRatio);
                    } else { 
                        $stopLoss = null; 
                        $takeProfit = null; 
                        $finalSignal = 0; 
                    } // Invalidar si SL >= Entrada
                 } else { // Short
                    $stopLoss = $currentEma + $slDistance; 
                    $riskAmount = $stopLoss - $entryPrice;
                    if ($riskAmount > 1e-9) {
                        $takeProfit = $entryPrice - ($riskAmount * $this->rrRatio);
                    } else { 
                        $stopLoss = null; 
                        $takeProfit = null; 
                        $finalSignal = 0; 
                    } // Invalidar si SL <= Entrada
                 }
                 // Redondeo opcional
                 if($stopLoss !== null) $stopLoss = round($stopLoss, 5); // Ajustar precisión
                 if($takeProfit !== null) $takeProfit = round($takeProfit, 5);
            } else {
                 // Si no hay señal final o datos insuficientes para SL/TP
                 $stopLoss = null; 
                 $takeProfit = null;
                 // No invalidamos la señal aquí si solo falta ATR/EMA, pero SL/TP serán null
            } // --- Fin Cálculo SL/TP ---
            $this->strategySignals[$key] = [
                'signal' => $finalSignal,
                'stopLoss' => $stopLoss,
                'takeProfit' => $takeProfit,
                'currentEma' => $currentEma,
                'currentAtr' => $currentAtr,
                'currentMD' => $currentImpulse[0],
                'currentSB' => $currentImpulse[1],
                'currentSH' => $currentImpulse[2],
                'currentTH' => $currentImpulse[3],

            ];
        }
        return $this->strategySignals;
    }
    //endregion

    //region --- Métodos de Utilidad ---
    /** Limpia la salida de funciones de indicadores. */
    private function cleanIndicatorOutput($indicatorOutput): array {
        if ($indicatorOutput === false || !is_array($indicatorOutput)) { return []; }
        $cleaned = [];
        foreach ($indicatorOutput as $key => $value) {
            if ($value === false || !is_numeric($value)) { $cleaned[$key] = null; }
            else { $cleaned[$key] = (float)$value; }
        } return $cleaned;
    }

    /** Alinea un array resultado con uno de referencia. */
     private function alignAndCleanWithReference(?array $resultArray, ?array $referenceArray): ?array {
         if ($resultArray === null || $referenceArray === null) return null;
         $cleanedResult = $this->cleanIndicatorOutput($resultArray);
         if (empty($referenceArray)) return [];
         if (empty($cleanedResult)) return array_fill_keys(array_keys($referenceArray), null);
         $alignedArray = []; $refKeys = array_keys($referenceArray); $refCount = count($refKeys);
         $resValues = array_values($cleanedResult); $resCount = count($resValues);
         $initialNulls = $refCount - $resCount;
         if ($initialNulls < 0) { /* Log error */ return null; }
         foreach ($refKeys as $index => $key) {
             if ($index < $initialNulls) { $alignedArray[$key] = null; }
             else { $resIndex = $index - $initialNulls; $alignedArray[$key] = $resValues[$resIndex] ?? null; }
         } return $alignedArray;
     }
    //endregion

} // Fin Clase ImpulseMacdStrategy (Versión con filtros OB/OS y EMA internos)