<?php

namespace App; // Ajusta el namespace a tu estructura de proyecto

use App\Trader; // Asegúrate que la extensión PECL Trader esté instalada o que exista esta clase personalizada
use InvalidArgumentException;
use RuntimeException;

/**
 * Calcula el indicador Impulse MACD (Modificado).
 * Incluye un filtro de umbral dinámico opcional (SMA de abs(Histograma))
 * y filtro de barras de confirmación para reducir señales falsas (whipsaws).
 * Esta versión SI calcula umbrales estáticos globales (OB/OS).
 */
class ImpulseMACDWS
{
    //region Propiedades de Configuración
    private int $lengthMA;
    private int $lengthSignal;
    private int $thresholdPeriod; // Período para SMA(abs(SH)), 0 o <1 para desactivar
    private ?float $globalAvgPositiveMD = null; // Promedio MD Positivo (calculado aquí)
    private ?float $globalAvgNegativeMD = null; // Promedio MD Negativo (calculado aquí)

    //endregion

    //region Propiedades para Resultados Calculados
    private ?array $hlc3 = null;
    private ?array $hi = null;   // smma(high)
    private ?array $lo = null;   // smma(low)
    private ?array $mi = null;   // zlema(hlc3)
    private ?array $md = null;   // Línea Principal (Market Direction)
    private ?array $sb = null;   // Línea de Señal (SMA de md)
    private ?array $sh = null;   // Histograma (md - sb)
    private ?array $shThreshold = null; // Umbral dinámico SMA(abs(SH))
    //endregion

    //region Estado Interno
    private bool $calculationDone = false;
    private int $dataCount = 0;
    //endregion

    //region Constructor
    /**
     * Constructor.
     * @param int $lengthMA Período para SMMA y ZLEMA base. Default 34.
     * @param int $lengthSignal Período para la línea de señal (SMA de MD). Default 9.
     * @param int $thresholdPeriod Período para el umbral dinámico SMA(abs(SH)). Default 0 (desactivado).
     */
    public function __construct(
        int $lengthMA = 34,
        int $lengthSignal = 9,
        int $thresholdPeriod = 0
    ) {
        if ($lengthMA <= 0 || $lengthSignal <= 0) {
            throw new InvalidArgumentException("Los períodos (lengthMA, lengthSignal) deben ser positivos.");
        }
        if ($thresholdPeriod < 0) {
            throw new InvalidArgumentException("El período del umbral (thresholdPeriod) no puede ser negativo.");
        }
        $this->lengthMA = $lengthMA;
        $this->lengthSignal = $lengthSignal;
        $this->thresholdPeriod = $thresholdPeriod;
    }
    //endregion

    //region Métodos Públicos de Acceso a Resultados (Getters)
    /** Devuelve la serie de la línea principal (Market Direction). */
    public function getMD(): ?array
    {
        return $this->md;
    }

    /** Devuelve la serie de la línea de señal (SMA de MD). */
    public function getSignalLine(): ?array
    {
        return $this->sb;
    }

    /** Devuelve la serie del histograma (MD - SB). */
    public function getHistogram(): ?array
    {
        return $this->sh;
    }

    /** Devuelve la serie del umbral dinámico del histograma (SMA de abs(SH)), si se calculó. */
    public function getHistogramThreshold(): ?array
    {
        return $this->shThreshold;
    }

    /** Devuleve el elemnto mas reciente de las series MD, Signal, Histograma e HistogramThreshold */
    
    public function getLastaDataOfSeries(): ?array
    {
        $lastData = array(
            0 => round(end($this->md),3),
            1 => round(end($this->sb),3),
            2 => round(end($this->sh),3),
            3 => round(end($this->shThreshold),3),
        );
        return $lastData;
    }
    //endregion

    //region Método Principal de Cálculo
    /**
     * Calcula componentes del indicador y umbral dinámico si está activado.
     * @param array $data Array asociativo con 'high', 'low', 'close'. Claves deben ser consistentes.
     * @throws RuntimeException Si ocurren errores durante el cálculo o datos insuficientes.
     */
    public function calculate(array $data): void
    {
        $this->resetResults();

        // Validaciones de Datos
        if (!isset($data['high'], $data['low'], $data['close'])) {
            throw new InvalidArgumentException("Datos de entrada incompletos (requiere 'high', 'low', 'close').");
        }
        $keySample = array_key_first($data['close']);
        if ($keySample === null) {
            throw new InvalidArgumentException("Array de precios 'close' está vacío.");
        }
        if (count($data['close']) !== count($data['high']) || count($data['close']) !== count($data['low'])) {
            throw new InvalidArgumentException("Arrays HLC deben tener el mismo tamaño.");
        }
        $this->dataCount = count($data['close']);

        // Estimación de puntos mínimos requeridos
        $minPointsCalculation = max($this->lengthMA * 2, $this->lengthSignal) + 5;
        $minPointsThreshold = $this->thresholdPeriod > 0 ? $minPointsCalculation + $this->thresholdPeriod : $minPointsCalculation;
        if ($this->dataCount < $minPointsThreshold + 5) {
            throw new RuntimeException("Datos insuficientes ({$this->dataCount} puntos disponibles, se requieren aprox. " . ($minPointsThreshold + 5) . ").");
        }

        try {
            // Cálculos intermedios y principales
            $this->hlc3 = $this->calculateHLC3($data);
            $highValues = array_filter($data['high'], 'is_numeric');
            $lowValues = array_filter($data['low'], 'is_numeric');
            $hlc3Values = array_filter($this->hlc3, 'is_numeric');

            $hiResult = $this->calculateSMMA(array_values($highValues), $this->lengthMA);
            $loResult = $this->calculateSMMA(array_values($lowValues), $this->lengthMA);
            $miResult = $this->calculateZLEMA(array_values($hlc3Values), $this->lengthMA);

            $this->hi = $this->alignAndCleanWithReference($hiResult, $this->hlc3);
            $this->lo = $this->alignAndCleanWithReference($loResult, $this->hlc3);
            $this->mi = $this->alignAndCleanWithReference($miResult, $this->hlc3);
            if ($this->hi === null || $this->lo === null || $this->mi === null) {
                throw new RuntimeException("Fallo al calcular o alinear SMMA/ZLEMA intermedios.");
            }
            $this->md = $this->calculateMD($this->mi, $this->hi, $this->lo);
            $mdValuesNumeric = array_filter($this->md, 'is_numeric');
            if (count($mdValuesNumeric) < $this->lengthSignal) {
                $this->sb = array_fill_keys(array_keys($this->md), null);
            } else {
                $sbResult = Trader::SMA(array_values($mdValuesNumeric), $this->lengthSignal);
                $this->sb = $this->alignAndCleanWithReference($sbResult, $this->md);
                if ($this->sb === null) {
                    throw new RuntimeException("Fallo al calcular o alinear la línea de señal (SB).");
                }
            }
            $this->sh = $this->calculateSH($this->md, $this->sb);

            // Calcular umbral dinámico si está activado
            if ($this->thresholdPeriod > 0 && $this->sh !== null) {
                $absSh = array_map(fn($val) => is_numeric($val) ? abs($val) : null, $this->sh);
                $absShNumeric = array_filter($absSh, 'is_numeric');
                if (count($absShNumeric) >= $this->thresholdPeriod) {
                    $thresholdSmaResult = Trader::SMA(array_values($absShNumeric), $this->thresholdPeriod);
                    $this->shThreshold = $this->alignAndCleanWithReference($thresholdSmaResult, $this->sh);
                } else {
                    $this->shThreshold = array_fill_keys(array_keys($this->sh), null);
                }
            } else {
                $this->shThreshold = null;
            }

            
            // --- Calcular Umbrales Estáticos (Promedios Globales MD) ---
            $mdSeries = $this->md; // Obtener MD de la clase base
            if (!is_array($mdSeries) || empty($mdSeries)) {
                 $this->globalAvgPositiveMD = null; $this->globalAvgNegativeMD = null;
                 // Continuar, pero el filtro OB/OS no se aplicará
            } else {
                 $this->calculateGlobalMDAveragesInternal($mdSeries); // Llamar al cálculo
            }
            $this->calculationDone = true;
        } catch (\Exception $e) {
            $this->resetResults();
            throw new RuntimeException("Error calculando ImpulseMACDWS: " . $e->getMessage(), $e->getCode(), $e);
        }
    }
    //endregion

    //region Método de Generación de Señales
    /**
     * Genera señales base (cruce) filtradas por umbral dinámico y confirmación.
     * @param int $confirmationPeriods Número de barras consecutivas que la condición debe mantenerse. Default 1.
     * @return array Mapa [clave => señal], donde señal es: 1 (Compra), -1 (Venta), 0 (Neutral/Sin señal).
     * @throws RuntimeException Si el indicador no ha sido calculado.
     */
    public function generateSignals(int $confirmationPeriods = 1): array
    {
        if (!$this->calculationDone || $this->md === null || $this->sb === null || $this->sh === null) {
            throw new RuntimeException("Debe llamar a calculate() antes de generateSignals().");
        }
        if ($confirmationPeriods < 1) {
            $confirmationPeriods = 1;
        }

        $signals = array_fill_keys(array_keys($this->md), 0);
        $keys = array_keys($this->md);
        $numKeys = count($keys);
        $allBea = [];
        $allBul = [];
        $oSoB = [];
        // Encontrar el primer índice donde tenemos datos válidos para MD y SB
        $firstValidIndex = -1;
        for ($idx = 0; $idx < $numKeys; $idx++) {
            $key = $keys[$idx];
            if (isset($this->md[$key], $this->sb[$key]) && is_numeric($this->md[$key]) && is_numeric($this->sb[$key])) {
                $firstValidIndex = $idx;
                break;
            }
        }
        // Si no hay datos válidos o no alcanzan para la ventana de confirmación, devolver ceros
        if ($firstValidIndex === -1 || $firstValidIndex + $confirmationPeriods >= $numKeys) {

             return $signals;
        }

        // Iterar desde el primer punto donde podemos mirar atrás para confirmar
        for ($i = $firstValidIndex + $confirmationPeriods; $i < $numKeys; $i++) {
            $currentKey = $keys[$i];
            $prevKey = $keys[$i - 1];

            // Comprobaciones de seguridad y validez de datos
            if (
                !isset($this->md[$currentKey], $this->md[$prevKey], $this->sb[$currentKey], $this->sb[$prevKey], $this->sh[$currentKey]) ||
                !is_numeric($this->md[$currentKey]) || !is_numeric($this->md[$prevKey]) ||
                !is_numeric($this->sb[$currentKey]) || !is_numeric($this->sb[$prevKey]) ||
                !is_numeric($this->sh[$currentKey])
            ) {
                 continue;
            }

            // Asignación de variables locales
            $currentMD = $this->md[$currentKey];
            $prevMD = $this->md[$prevKey];
            $currentSB = $this->sb[$currentKey];
            $prevSB = $this->sb[$prevKey];
            $currentSH = $this->sh[$currentKey];
            $dynamicThreshold = ($this->shThreshold !== null && isset($this->shThreshold[$currentKey]) && is_numeric($this->shThreshold[$currentKey]))
                                ? $this->shThreshold[$currentKey]
                                : null;
            // Detección de Cruces
            $bullishCross = ($prevMD <= $prevSB && $currentMD > $currentSB);
            $bearishCross = ($prevMD >= $prevSB && $currentMD < $currentSB);
            $allBea[$i] = $bearishCross?1:0;
            $allBul[$i] = $bullishCross?1:0;
            // Aplicar Filtros
            // Filtro OB/OS (Usando umbrales calculados en esta clase)
            $passesObOs = false;
            if ($this->globalAvgPositiveMD === null || $currentMD > $this->globalAvgPositiveMD) {
                // Pasa si AvgPosMD es null O si MD > AvgPosMD 
                $passesObOs = true;
            }elseif($this->globalAvgNegativeMD === null || $currentMD < $this->globalAvgNegativeMD) {
                // Pasa si AvgNegMD es null (no se pudo calcular) O si MD < AvgNegMD    
                $passesObOs = true;
            }
            $oSoB[$i] =$passesObOs?1:0;
            if ($bullishCross) {
                // Filtro Umbral Dinámico
                // *** comentado para probar 22/04
                //if ($dynamicThreshold !== null && $currentSH <= $dynamicThreshold) {
                    //continue;
                //} 
                // Filtro Confirmación
                $confirmed = true;
                for ($j = 1; $j < $confirmationPeriods; $j++) {
                    $confirmIndex = $i - $j;
                    $confirmKey = $keys[$confirmIndex];
                    $confirmMdValue = $this->md[$confirmKey] ?? null;
                    $confirmSbValue = $this->sb[$confirmKey] ?? null;
                    if ($confirmMdValue === null || $confirmSbValue === null || $confirmMdValue <= $confirmSbValue) {
                        $confirmed = false;
                        break;
                    }
                } // Fin for confirmación

                //if ($confirmed && $passesObOs) {
                if ($passesObOs) {                    
                    $signals[$currentKey] = 1; // Señal Compra
                }
            } elseif ($bearishCross) {

                // Filtro Umbral Dinámico
                // *** comentado patra probar 22/04
                //if ($dynamicThreshold !== null && $currentSH >= -$dynamicThreshold) {
                    //continue;
                //}

                // Filtro Confirmación
                $confirmed = true;
                for ($j = 1; $j < $confirmationPeriods; $j++) {
                     $confirmIndex = $i - $j;
                     $confirmKey = $keys[$confirmIndex];
                     $confirmMdValue = $this->md[$confirmKey] ?? null;
                     $confirmSbValue = $this->sb[$confirmKey] ?? null;
                     if ($confirmMdValue === null || $confirmSbValue === null || $confirmMdValue >= $confirmSbValue) {
                         $confirmed = false;
                         break; 
                     }
                } // Fin for confirmación
                //if ($confirmed && $passesObOs) {
                if ($passesObOs) {
                    $signals[$currentKey] = -1; // Señal Venta
                }
            } // Fin if/elseif cross
        } // Fin for principal
// **** Insertar aqui el codigo para testing que esat al final comentado ******

        return $signals;
    }
    //endregion

    //region --- Métodos Internos y Utilitarios (Expandidos) ---

    /** Reinicia todas las propiedades de resultados calculados a null. */
    private function resetResults(): void
    {
        $this->hlc3 = null;
        $this->hi = null;
        $this->lo = null;
        $this->mi = null;
        $this->md = null;
        $this->sb = null;
        $this->sh = null;
        $this->shThreshold = null;
        $this->calculationDone = false;
        $this->dataCount = 0;
    }

    /** Calcula la serie HLC3 ((High + Low + Close) / 3). */
    private function calculateHLC3(array $data): array
    {
        $hlc3 = [];
        foreach ($data['close'] as $key => $closeValue) {
            if (
                isset($data['high'][$key], $data['low'][$key]) &&
                is_numeric($closeValue) &&
                is_numeric($data['high'][$key]) &&
                is_numeric($data['low'][$key])
            ) {
                $hlc3[$key] = ($data['high'][$key] + $data['low'][$key] + $closeValue) / 3.0;
            } else {
                $hlc3[$key] = null;
            }
        }
        return $hlc3;
    }

    /** Calcula la serie MD (Market Direction). */
    private function calculateMD(array $mi, array $hi, array $lo): array
    {
        $md = [];
        foreach ($mi as $key => $miValue) {
            $hiValue = $hi[$key] ?? null;
            $loValue = $lo[$key] ?? null;

            if (is_numeric($miValue) && is_numeric($hiValue) && is_numeric($loValue)) {
                if ($miValue > $hiValue) {
                    $md[$key] = $miValue - $hiValue;
                } elseif ($miValue < $loValue) {
                    $md[$key] = $miValue - $loValue;
                } else {
                    $md[$key] = 0.0;
                }
            } else {
                $md[$key] = null;
            }
        }
        return $md;
    }

    /** Calcula la serie SH (Histograma = MD - SB). */
    private function calculateSH(array $md, array $sb): array
    {
        $sh = [];
        foreach ($md as $key => $mdValue) {
            $sbValue = $sb[$key] ?? null;
            if (is_numeric($mdValue) && is_numeric($sbValue)) {
                $sh[$key] = $mdValue - $sbValue;
            } else {
                $sh[$key] = null;
            }
        }
        return $sh;
    }

    /** Calcula la Smoothed Moving Average (SMMA / RMA). */
    private function calculateSMMA(array $source, int $length): array
    {
        $output = [];
        $count = count($source);
        if ($count < $length) {
            return array_fill(0, $count, null);
        }

        $sum = 0.0;
        for ($i = 0; $i < $length; $i++) {
            if (!is_numeric($source[$i])) {
                return array_fill(0, $count, null);
            }
            $sum += $source[$i];
        }
        $output[$length - 1] = $sum / $length;

        for ($i = $length; $i < $count; $i++) {
            if (!is_numeric($source[$i]) || !isset($output[$i-1]) || $output[$i-1] === null) {
                $output[$i] = null;
                continue;
            }
            $output[$i] = ($output[$i - 1] * ($length - 1) + $source[$i]) / $length;
        }

        for ($i = 0; $i < $length - 1; $i++) {
            if (!isset($output[$i])) { // Evitar sobreescribir si ya existe por alguna razón
                $output[$i] = null;
            }
        }
        ksort($output); // Asegurar orden 0..N-1
        return $output;
    }

    /** Calcula la Zero-Lag Exponential Moving Average (ZLEMA). (Versión con corrección de claves) */
    private function calculateZLEMA(array $source, int $length): array
    {
        $count = count($source);
        if ($count < $length) {
            return array_fill(0, $count, null);
        }

        $ema1Result = Trader::EMA($source, $length);
        $ema1 = $this->cleanIndicatorOutput($ema1Result);
        if (empty($ema1)) {
            return array_fill(0, $count, null);
        }

        $ema1ValuesOnly = array_values(array_filter($ema1, 'is_numeric'));
        if (empty($ema1ValuesOnly) || count($ema1ValuesOnly) < $length) {
            return array_fill(0, $count, null);
        }

        $ema2Result = Trader::EMA($ema1ValuesOnly, $length);
        $ema2 = $this->alignAndCleanWithReference($ema2Result, $ema1);
        if ($ema2 === null) {
            return array_fill(0, $count, null);
        }

        $zlema = [];
        $referenceKeys = array_keys($source); // Usar claves 0..N-1 del input

        for ($i = 0; $i < $count; $i++) {
            $key = $referenceKeys[$i];

            if (!array_key_exists($key, $ema1) || !array_key_exists($key, $ema2)) {
                 $zlema[$key] = null;
                 continue;
            }

            $e1 = $ema1[$key];
            $e2 = $ema2[$key];
            $zl = null;
            if (is_numeric($e1) && is_numeric($e2)) {
                $lag = $e1 - $e2;
                $zl = $e1 + $lag;
            }
            $zlema[$key] = $zl;
        }
        return $zlema;
    }

    /** Lógica interna para calcular los promedios globales de la serie MD. */
    private function calculateGlobalMDAveragesInternal(array $mdSeries): void {
        $this->globalAvgPositiveMD = null; $this->globalAvgNegativeMD = null;
        $positiveMdValues = []; $negativeMdValues = [];
        // Usar array_filter para simplificar la obtención de valores numéricos
        $numericMdValues = array_filter($mdSeries, 'is_numeric');

        foreach ($numericMdValues as $value) {
           if ($value > 0) { $positiveMdValues[] = $value; }
           elseif ($value < 0) { $negativeMdValues[] = $value; }
        }

        $countPos = count($positiveMdValues);
        if ($countPos > 0) {
            // Usar array_values porque Trader::SMA espera índices 0..N-1
            $posSmaResult = Trader::SMA(array_values($positiveMdValues), $countPos);
            if ($posSmaResult !== false && !empty($posSmaResult)) {
                 $lastValue = end($posSmaResult);
                 if (is_numeric($lastValue)) { $this->globalAvgPositiveMD = (float)$lastValue; }
            }
        }
        $countNeg = count($negativeMdValues);
        if ($countNeg > 0) {
            $negSmaResult = Trader::SMA(array_values($negativeMdValues), $countNeg);
             if ($negSmaResult !== false && !empty($negSmaResult)) {
                 $lastValue = end($negSmaResult);
                 if (is_numeric($lastValue)) { $this->globalAvgNegativeMD = (float)$lastValue; }
             }
        }
    }    

    /** Limpia la salida de funciones de indicadores. */
    private function cleanIndicatorOutput($indicatorOutput): array
    {
        if ($indicatorOutput === false || !is_array($indicatorOutput)) {
            return [];
        }
        $cleaned = [];
        foreach ($indicatorOutput as $key => $value) {
            if ($value === false || !is_numeric($value)) {
                $cleaned[$key] = null;
            } else {
                $cleaned[$key] = (float)$value;
            }
        }
        return $cleaned;
    }

     /** Alinea un array resultado con uno de referencia. */
     private function alignAndCleanWithReference(?array $resultArray, ?array $referenceArray): ?array
     {
         if ($resultArray === null || $referenceArray === null) {
             return null;
         }
         $cleanedResult = $this->cleanIndicatorOutput($resultArray);
         if (empty($referenceArray)) {
             return [];
         }
         if (empty($cleanedResult)) {
             return array_fill_keys(array_keys($referenceArray), null);
         }

         $alignedArray = [];
         $refKeys = array_keys($referenceArray);
         $refCount = count($refKeys);
         $resValues = array_values($cleanedResult);
         $resCount = count($resValues);
         $initialNulls = $refCount - $resCount;

         if ($initialNulls < 0) {
             // Log error? El resultado no puede ser más largo que la referencia.
             return null;
         }

         foreach ($refKeys as $index => $key) {
             if ($index < $initialNulls) {
                 $alignedArray[$key] = null;
             } else {
                 $resIndex = $index - $initialNulls;
                 $alignedArray[$key] = $resValues[$resIndex] ?? null;
             }
         }
         return $alignedArray;
     }
    //endregion

} // Fin Clase ImpulseMACDWS

/*
codigo para testing
        $bu = array_map(function($v) { if($v > 0)return $v; }, array_slice($allBul, -10, 10, true));
        $be = array_map(function($v) { if($v > 0)return $v; }, array_slice($allBea, -10, 10, true));
        $se = array_map(function($v) { if($v > 0)return $v; }, array_slice($signals, -10, 10, true));
        $th = array_map(function($v) { if($v > 0)return $v; }, array_slice($this->shThreshold, -10, 10, true));
        $sb = array_map(function($v) { if($v > 0)return $v; }, array_slice($oSoB, -10, 10, true));

        $qpasa = [
            'bullishCross' => $bu,
            'bearishCross' => $be,
            'oSoB' => $sb,
            'globalAvgNegativeMD' => $this->globalAvgNegativeMD,
            'globalAvgPositiveMD' => $this->globalAvgPositiveMD,
            'shThreshold' => $th,
            'signal' => $se,

        ];
        return $qpasa;
*/