<?php
namespace App;
use App\Trader;
use InvalidArgumentException;

/**
 * Clase ImpulseMACD para calcular el indicador Impulse MACD y generar estrategias de trading.
 */
class ImpulseMACD {
    //region Propiedades de la clase
    /**
     * Ventana de tiempo corta para el cálculo del MACD.
     *
     * @var int
     */
    private $shortWindow;

    /**
     * Ventana de tiempo larga para el cálculo del MACD.
     *
     * @var int
     */
    private $longWindow;

    /**
     * Ventana de tiempo para el cálculo de la línea de señal del MACD.
     *
     * @var int
     */
    private $signalWindow;

    /**
     * Ventana de tiempo para el cálculo de la media móvil del rango high-low.
     *
     * @var int
     */
    private $maRangeWindow;

    /**
     * Multiplicador utilizado para calcular los umbrales.
     *
     * @var float
     */
    private $multiplier;

    /**
     * Array que contiene los valores de la línea MACD.
     *
     * @var array|null
     */
    private $macdLine;

    /**
     * Array que contiene los valores de la línea de señal del MACD.
     *
     * @var array|null
     */
    private $signalLine;

    /**
     * Array que contiene los valores del Impulse MACD.
     *
     * @var array|null
     */
    private $impulseMACD;

    /**
     * Array que contiene los valores de la EMA de la línea MACD.
     *
     * @var array|null
     */
    private $emaMacd;

    /**
     * Array que contiene los valores del histograma del MACD.
     *
     * @var array|null
     */
    private $histogram;

    /**
     * Array que contiene los valores de la EMA de 100 períodos del precio de cierre.
     *
     * @var array|null
     */
    private $ema100;

    /**
     * Período utilizado para el cálculo del Average True Range (ATR).
     *
     * @var int
     */
    private $atrPeriod;

    /**
     * Multiplicador utilizado para calcular el Stop Loss y Take Profit en función del ATR.
     *
     * @var float
     */
    private $atrMultiplier;

    /**
     * Umbral superior calculado.
     *
     * @var float
     */
    private $upperThreshold;

    /**
     * Umbral inferior calculado.
     *
     * @var float
     */
    private $lowerThreshold;

    //endregion

    //region Constructor
    /**
     * Constructor de la clase ImpulseMACD.
     *
     * @param int $shortWindow Ventana corta para el cálculo del MACD.
     * @param int $longWindow Ventana larga para el cálculo del MACD.
     * @param int $signalWindow Ventana para la línea de señal del MACD.
     * @param int $maRangeWindow Ventana para calcular la media móvil del rango high-low.
     * @param float $multiplier Multiplicador utilizado en los umbrales.
     * @param int $atrPeriod Período para el cálculo del ATR.
     * @param float $atrMultiplier Multiplicador para el cálculo del SL y TP con ATR.
     */
    public function __construct(
        int $shortWindow = 12,
        int $longWindow = 26,
        int $signalWindow = 9,
        int $maRangeWindow = 5,
        float $multiplier = 1.5,
        int $atrPeriod = 14,
        float $atrMultiplier = 2
    ) {
        $this->shortWindow = $shortWindow;
        $this->longWindow = $longWindow;
        $this->signalWindow = $signalWindow;
        $this->maRangeWindow = $maRangeWindow;
        $this->multiplier = $multiplier;
        $this->atrPeriod = $atrPeriod;
        $this->atrMultiplier = $atrMultiplier;
    }
    //endregion

    //region Métodos de Cálculo
    /**
     * Calcula el MACD tradicional e Impulse MACD utilizando la librería trader.
     *
     * @param array $data Array asociativo con claves 'high', 'low', y 'close'.
     * @return array Array de datos original.
     * @throws \InvalidArgumentException Si el array de datos no tiene las claves necesarias.
     */
    public function calculate(array $data): array {
        if (!is_array($data) || !isset($data['high'], $data['low'], $data['close'])) {
            throw new InvalidArgumentException("El array debe contener claves 'high', 'low', y 'close'.");
        }

        $close = $data['close'];

        // Calcula MACD utilizando trader_macd
        $macd = Trader::MACD($close, $this->shortWindow, $this->longWindow, $this->signalWindow);
        $this->macdLine = $macd[0];
        $this->signalLine = $macd[1];
        $this->histogram = $macd[2];
        $this->impulseMACD = $this->macdLine;

        // Calcula los umbrales superior e inferior
        $this->calculateThresholds();

        // Calcula la EMA de 100 períodos
        $this->ema100 = Trader::EMA($close, 100);

        // Calcula la EMA de la línea MACD para los umbrales
        $this->emaMacd = Trader::EMA($this->macdLine, 9);

        return $data;
    }

    /**
     * Calcula los umbrales superior e inferior para filtrar las señales del MACD y la línea de señal.
     *
     * @return void
     */
    private function calculateThresholds(): void {
        $positiveMacdValues = [];
        $negativeMacdValues = [];

        // Separar los valores positivos y negativos del MACD, excluyendo el cero
        foreach ($this->macdLine as $value) {
            if ($value > 0) {
                $positiveMacdValues[] = $value;
            } elseif ($value < 0) {
                $negativeMacdValues[] = $value;
            }
        }

        // Calcular las SMA de los valores positivos y negativos
        $positiveSma = count($positiveMacdValues) > 1 ? Trader::SMA($positiveMacdValues, count($positiveMacdValues))[count($positiveMacdValues)-1] : 0;
        $negativeSma = count($negativeMacdValues) > 1 ? Trader::SMA($negativeMacdValues, count($negativeMacdValues))[count($negativeMacdValues)-1] : 0;


        $this->upperThreshold = $positiveSma;
        $this->lowerThreshold = $negativeSma;
    }

    /**
     * Verifica si hay una cruz alcista (cruce bullish).
     *
     * @param int $index Índice actual para verificar el cruce.
     * @return bool True si hay una cruz alcista, false en caso contrario.
     */
    public function isBullishCross(int $index): bool {
        if ($index < 1) {
            return false;
        }
        return $this->macdLine[$index] > $this->signalLine[$index] && $this->macdLine[$index - 1] <= $this->signalLine[$index - 1];
    }

    /**
     * Verifica si hay una cruz bajista (cruce bearish).
     *
     * @param int $index Índice actual para verificar el cruce.
     * @return bool True si hay una cruz bajista, false en caso contrario.
     */
    public function isBearishCross(int $index): bool {
        if ($index < 1) {
            return false;
        }
        return $this->macdLine[$index] < $this->signalLine[$index] && $this->macdLine[$index - 1] >= $this->signalLine[$index - 1];
    }
    //endregion

    //region Métodos de Umbrales
    /**
     * Obtiene el umbral superior calculado.
     *
     * @return float Umbral superior.
     */
    public function getUpperThreshold(): float {
        return $this->upperThreshold;
    }

    /**
     * Obtiene el umbral inferior calculado.
     *
     * @return float Umbral inferior.
     */
    public function getLowerThreshold(): float {
        return $this->lowerThreshold;
    }
    //endregion

    //region Método de Estrategia
     /**
     * Genera una estrategia de trading basada en cruces, umbrales y la EMA de 100, con gestión de riesgos.
     *
     * @param array $data Array asociativo con claves 'high', 'low', y 'close'.
     * @return array Array de señales de trading.
     */
    public function generateStrategy(array $data): array {
        $this->calculate($data);
        $strategy = [];
        $lastIndex = $this->array_key_last($this->impulseMACD);

        $action = 'HOLD';
        $stopLoss = 0;
        $takeProfit = 0;
        $signalType = '';

        $closePrice = $data['close'][$lastIndex];

        if (isset($this->ema100[$lastIndex]) && $this->macdLine[$lastIndex] != 0 && $this->signalLine[$lastIndex] != 0) {
            // Calcular el cruce solo para el período actual.
            $isBullish = $this->isBullishCross($lastIndex);
            $isBearish = $this->isBearishCross($lastIndex);

            // Filtrar las señales del MACD y la línea de señal usando los umbrales calculados.
            if ($this->macdLine[$lastIndex] > $this->upperThreshold && $isBullish && $closePrice > $this->ema100[$lastIndex]) {
                $action = 'BUY';
                $signalType = 'BUY';
            } elseif ($this->macdLine[$lastIndex] < $this->lowerThreshold && $isBearish && $closePrice < $this->ema100[$lastIndex]) {
                $action = 'SELL';
                $signalType = 'SELL';
            }
        }

        // Calcular el ATR y establecer SL/TP solo si hay una señal.
        if ($action === 'BUY' || $action === 'SELL') {
            $atr = Trader::ATR($data['high'], $data['low'], $data['close'], $this->atrPeriod)[$lastIndex];
            if ($action === 'BUY') {
                $stopLoss = $this->ema100[$lastIndex] - ($atr * $this->atrMultiplier);
                $takeProfit = $closePrice + ($atr * $this->atrMultiplier);
            } else {
                $stopLoss = $this->ema100[$lastIndex] + ($atr * $this->atrMultiplier);
                $takeProfit = $closePrice - ($atr * $this->atrMultiplier);
            }
        }

        $strategy = [
            'action' => $action,
            'stopLoss' => $stopLoss,
            'takeProfit' => $takeProfit,
            'signalType' => $signalType,
            'upper' => $this->upperThreshold,
            'lower' => $this->lowerThreshold,
            'macd' => $this->macdLine[$lastIndex],
            'signal' => $this->signalLine[$lastIndex],
            'ema100' => $this->ema100[$lastIndex],
            'price' => $closePrice,
            'up' => $isBullish,
            'do' => $isBearish
        ];

        return $strategy;
    }
    //endregion

    //region Métodos de Utilidad
     /**
     * Obtiene el último índice de un array.
     *
     * @param array $arr El array del cual se quiere obtener el último índice.
     * @return int|string El último índice del array, o NULL si el array está vacío.
     */
    private function array_key_last(array $arr) {
        if (empty($arr)) {
            return null;
        }
        return array_key_last($arr);
    }

    /**
     * Obtiene el primer índice de un array.
     *
     * @param array $arr El array del cual se quiere obtener el primer índice.
     * @return int|string|null El primer índice del array, o NULL si el array está vacío.
     */
    private function array_key_first(array $arr) {
        if (empty($arr)) {
            return null;
        }
        return array_key_first($arr);
    }
    //endregion
}

