<?php
/*
**************** Uso *******
require __DIR__ . '/vendor/autoload.php';
use App\DailyRotatingLogger;

// Iniciar el logger (guardará hasta 3 días de logs)
$logger = new DailyRotatingLogger(__DIR__ . '/logs/app.log', 3);

// Escribir algunos mensajes
$logger->info('Usuario inició sesión');
$logger->warning('Memoria usada: 90%');
$logger->error('Error al conectar a la base de datos');
$logger->debug('Datos recibidos: ', ['usuario' => 'admin']);

echo "Logs escritos correctamente.";
*/

namespace App;

use Monolog\Logger;
use Monolog\Handler\RotatingFileHandler;
use Monolog\Formatter\LineFormatter;

class DailyRotatingLogger
{
    private Logger $logger;
    private string $logFilePath;
    private int $maxFiles;

    public function __construct(string $logFilePath, int $maxFiles = 7)
    {
        $this->logFilePath = $logFilePath;
        $this->maxFiles = $maxFiles;

        // Crear el directorio si no existe
        $logDir = dirname($logFilePath);
        if (!is_dir($logDir)) {
            mkdir($logDir, 0755, true);
        }

        // Crear el logger
        $this->logger = new Logger('daily_logger');

        // Configurar RotatingFileHandler
        $handler = new RotatingFileHandler($this->logFilePath, $this->maxFiles, Logger::DEBUG);

        // Formato personalizado
        $formatter = new LineFormatter(
            "[%datetime%] [%level_name%]: %message% %context%\n",
            'Y-m-d H:i:s'
        );
        $handler->setFormatter($formatter);

        // Agregar handler al logger
        $this->logger->pushHandler($handler);
    }

    /**
     * Registrar mensaje con nivel arbitrario
     */
    public function log(string $level, string $message, array $context = []): void
    {
        if (method_exists($this->logger, $level)) {
            $this->logger->{$level}($message, $context);
        } else {
            $this->logger->info("[$level] $message", $context);
        }
    }

    // Métodos rápidos para cada nivel de log
    public function debug(string $message, array $context = []): void
    {
        $this->logger->debug($message, $context);
    }

    public function info(string $message, array $context = []): void
    {
        $this->logger->info($message, $context);
    }

    public function notice(string $message, array $context = []): void
    {
        $this->logger->notice($message, $context);
    }

    public function warning(string $message, array $context = []): void
    {
        $this->logger->warning($message, $context);
    }

    public function error(string $message, array $context = []): void
    {
        $this->logger->error($message, $context);
    }

    public function critical(string $message, array $context = []): void
    {
        $this->logger->critical($message, $context);
    }

    public function alert(string $message, array $context = []): void
    {
        $this->logger->alert($message, $context);
    }

    public function emergency(string $message, array $context = []): void
    {
        $this->logger->emergency($message, $context);
    }

    /**
     * Método para verificar si el logger está funcionando
     */
    public function test(): void
    {
        $this->info('Logger test - funcionando correctamente');
        echo "Test escrito en: " . $this->logFilePath . "\n";
        echo "Directorio existe: " . (is_dir(dirname($this->logFilePath)) ? 'Sí' : 'No') . "\n";
        echo "Archivo escribible: " . (is_writable(dirname($this->logFilePath)) ? 'Sí' : 'No') . "\n";
    }
}