<?php
namespace App;

use App\Trader;
use App\PivotCalculator;

class BreakoutProbabilitySR {
    private $high;
    private $low;
    private $close;
    private $volume;
    private $highPivots = [];
    private $lowPivots = [];
    private $lastAtr;
    
    private $config = [
        'atr_period' => 14,
        'stop_loss_threshold' => 25,
        'tp_sl_ratio' => 2.5,
        'trend_threshold' => 40,
        'volume_multiplier' => 1.5,
        'max_pivot_distance' => 2.0,
        'min_pivot_distance' => 0.5
    ];

    public function __construct(
        array $high, 
        array $low, 
        array $close, 
        array $volume,
        array $config = []
    ) {
        $this->high = $high;
        $this->low = $low;
        $this->close = $close;
        $this->volume = $volume;
        $this->config = array_merge($this->config, $config);
        $this->calculatePivots();
        $this->lastAtr = $this->calculateAtr();
        $this->ensureValidLevels();

    }
    private function ensureValidLevels(): void {
        $currentPrice = end($this->close);
        $atrThreshold = max(0.031, min(0.067, $this->lastAtr)); // Usamos el ATR actual como referencia

        $this->lowPivots = $this->removeCloseLevels($this->lowPivots, $atrThreshold, 'support');
        $this->highPivots = $this->removeCloseLevels($this->highPivots, $atrThreshold, 'resistance');
        
        // Filtrado final basado en ATR para asegurar que los pivotes no estén demasiado cerca
        $minDistance = max(0.02 * $currentPrice, 2 * $this->lastAtr);
        $this->lowPivots = array_filter($this->lowPivots, fn($l) => $l < $currentPrice - $minDistance);
        $this->highPivots = array_filter($this->highPivots, fn($h) => $h > $currentPrice + $minDistance);
    }

    private function removeCloseLevels(array $levels, float $threshold, string $type = 'support'): array {
        if (empty($levels)) return [];
        
        sort($levels); // Asegura que los niveles estén ordenados
        $filtered = [$levels[0]]; // Incluir el primer nivel siempre
        
        foreach ($levels as $level) {
            // Asegúrate de que no eliminamos niveles demasiado cerca
            if (abs($level - end($filtered)) >= $threshold) {
                $filtered[] = $level; // Solo agregar si está lo suficientemente lejos
            }
        }
        
        return $filtered;
    }    
    
    private function getFallbackLevels(float $price, string $type): array {
        $atrBuffer = $this->lastAtr * 3;
        return $type === 'support'
            ? [round($price - $atrBuffer, 3), round($price * 0.97, 3)]
            : [round($price + $atrBuffer, 3), round($price * 1.03, 3)];
    }

    private function calculatePivots(): void {
        $pivotCalculator = new PivotCalculator($this->high, $this->low, $this->close, $this->lastAtr);
        $pivotsData = $pivotCalculator->calculateClassicPivots();
        $this->highPivots = $pivotsData['resistances'];
        $this->lowPivots = $pivotsData['supports'];
    }

    private function filterLevels(array $levels, float $currentPrice, string $type): array {
        return array_filter($levels, function($level) use ($currentPrice, $type) {
            $distance = abs($level - $currentPrice);
            $minValid = $this->lastAtr * $this->config['min_pivot_distance'];
            $maxValid = $this->lastAtr * $this->config['max_pivot_distance'];
            
            return $type === 'support' 
                ? ($level < $currentPrice && $distance >= $minValid && $distance <= $maxValid)
                : ($level > $currentPrice && $distance >= $minValid && $distance <= $maxValid);
        });
    }


    private function addFallbackLevels(): void {
        $current = end($this->close);
        $atrBuffer = $this->lastAtr * 4; // 4x ATR
        
        if(empty($this->highPivots)) {
            $this->highPivots = [round($current + $atrBuffer, 3)];
        }
        
        if(empty($this->lowPivots)) {
            $this->lowPivots = [round($current - $atrBuffer, 3)];
        }
        
        // Forzar eliminación de niveles cercanos
        $this->highPivots = array_filter($this->highPivots, fn($h) => abs($h - $current) > ($this->lastAtr * 1));
        $this->lowPivots = array_filter($this->lowPivots, fn($l) => abs($l - $current) > ($this->lastAtr * 1));
    }

    private function calculateAtr(): float {
        $atrValues = Trader::ATR(
            $this->high,
            $this->low,
            $this->close,
            $this->config['atr_period']
        );
        return end($atrValues) ?: 0.01;
    }

    private function findNearestLevel(float $price, array $levels, string $levelType): ?float {
        $validLevels = array_filter($levels, function($l) use ($price, $levelType) {
            $distance = abs($l - $price);
            return $levelType === 'support'
                ? ($l < $price && $distance >= ($this->lastAtr * $this->config['min_pivot_distance']))
                : ($l > $price && $distance >= ($this->lastAtr * $this->config['min_pivot_distance']));
        });
        
        if (empty($validLevels)) return null;
        
        $closest = null;
        $minDistance = PHP_FLOAT_MAX;
        foreach ($validLevels as $level) {
            $currentDistance = abs($price - $level);
            if ($currentDistance < $minDistance) {
                $closest = $level;
                $minDistance = $currentDistance;
            }
        }
        
        return $closest;
    }

    private function calculateProbability(float $distance, float $volumeFactor, string $direction): float {
        $maxDistance = max($this->lastAtr * $this->config['max_pivot_distance'], 0.01);
        $effectiveDistance = max($distance, $this->lastAtr * 0.2);
        $probability = 100 * (1 - ($effectiveDistance / $maxDistance));
        $probability *= $volumeFactor * ($direction === 'bullish' ? 1.05 : 0.95);
        return min(max(round($probability, 2), 10), 90);
    }

    public function getBullishProbability(): float {
        $currentPrice = end($this->close);
        $resistance = $this->findNearestLevel($currentPrice, $this->highPivots, 'resistance');
        return $resistance ? $this->calculateProbability(
            $resistance - $currentPrice, 
            $this->getVolumeFactor(),
            'bullish'
        ) : 0;
    }

    public function getBearishProbability(): float {
        $currentPrice = end($this->close);
        $support = $this->findNearestLevel($currentPrice, $this->lowPivots, 'support');
        return $support ? $this->calculateProbability(
            $currentPrice - $support, 
            $this->getVolumeFactor(),
            'bearish'
        ) : 0;
    }

    private function getVolumeFactor(): float {
        $lastVolume = end($this->volume);
        $averageVolume = array_sum($this->volume) / max(count($this->volume), 1);
        
        return $lastVolume > $averageVolume ? $this->config['volume_multiplier'] : 1.0;
    }

    private function determineTrend(): array {
        $bullish = $this->getBullishProbability();
        $bearish = $this->getBearishProbability();
        
        $threshold = $this->config['trend_threshold'];
        $differential = 15; // Diferencia mínima requerida
        
        if ($bullish > $threshold && ($bullish - $bearish) > $differential) {
            return ['type' => 'bullish', 'probability' => $bullish];
        }
        
        if ($bearish > $threshold && ($bearish - $bullish) > $differential) {
            return ['type' => 'bearish', 'probability' => $bearish];
        }
        
        return ['type' => 'neutral', 'probability' => 0];
    }

    private function getTrendStrength(float $probability): string {
        if($probability > 80) return 'strong';
        return $probability > 55 ? 'moderate' : 'weak';
    }

    public function getFullAnalysis(): array {
        $currentPrice = end($this->close);
        return [
            'price' => round($currentPrice, 3),
            'volatility' => [
                'atr' => round($this->lastAtr, 3),
                'daily_range' => round(max($this->high) - min($this->low), 3)
            ],
            'volume' => [
                'current' => end($this->volume),
                'average' => round(array_sum($this->volume) / max(count($this->volume), 1), 2),
                'factor' => round($this->getVolumeFactor(), 2)
            ],
            'trend' => $this->determineTrend(),
            'key_levels' => [
                'supports' => array_slice($this->lowPivots, 0, 3),
                'resistances' => array_slice($this->highPivots, 0, 3)
            ],
            'probability_matrix' => [
                'most_probable' => $this->getMostProbableLevel(),
                'breakout_probabilities' => [
                    'bullish' => $this->getBullishProbability(),
                    'bearish' => $this->getBearishProbability()
                ]
            ],
            'trade_signals' => $this->generateTradeSignals()
        ];
    }

    private function getMostProbableLevel(): array {
        $bullishProb = $this->getBullishProbability();
        $bearishProb = $this->getBearishProbability();
        
        if($bullishProb > $bearishProb && $bullishProb > 0) {
            return [
                'type' => 'resistance',
                'level' => $this->findNearestLevel(end($this->close), $this->highPivots, 'resistance'),
                'probability' => $bullishProb
            ];
        }
        
        if($bearishProb > $bullishProb && $bearishProb > 0) {
            return [
                'type' => 'support',
                'level' => $this->findNearestLevel(end($this->close), $this->lowPivots, 'support'),
                'probability' => $bearishProb
            ];
        }
        
        return ['type' => 'none', 'level' => null, 'probability' => 0];
    }

    private function generateTradeSignals(): array {
        $signals = [];
        $currentPrice = end($this->close);
        $trend = $this->determineTrend();
        
        if($trend['type'] !== 'neutral') {
            $stopLoss = $this->calculateStopLoss($trend['type']);
            $takeProfit = $this->calculateTakeProfit($currentPrice, $stopLoss);
            
            $signals = [
                'entry' => round($currentPrice, 3),
                'stop_loss' => $stopLoss,
                'take_profit' => $takeProfit,
                'risk_reward' => round(
                    ($takeProfit['level'] - $currentPrice) / 
                    ($currentPrice - $stopLoss['level']), 
                    2
                )
            ];
        }
        
        return $signals;
    }

    private function calculateStopLoss(string $trendType): array {
        $currentPrice = end($this->close);
        
        if($trendType === 'bullish') {
            $level = $this->findNearestLevel($currentPrice, $this->lowPivots, 'support')
                   ?? $currentPrice - ($this->lastAtr * 1.5);
            return [
                'type' => 'below',
                'level' => round($level, 3),
                'source' => $level ? 'pivot_level' : 'atr_extension'
            ];
        }
        
        $level = $this->findNearestLevel($currentPrice, $this->highPivots, 'resistance')
               ?? $currentPrice + ($this->lastAtr * 1.5);
        return [
            'type' => 'above',
            'level' => round($level, 3),
            'source' => $level ? 'pivot_level' : 'atr_extension'
        ];
    }

    private function calculateTakeProfit(float $entry, array $stopLoss): array {
        $distance = abs($entry - $stopLoss['level']);
        $multiplier = $this->config['tp_sl_ratio'];
        
        return [
            'type' => $stopLoss['type'] === 'below' ? 'above' : 'below',
            'level' => round(
                $stopLoss['type'] === 'below' 
                    ? $entry + ($distance * $multiplier)
                    : $entry - ($distance * $multiplier), 
                3
            ),
            'ratio' => $multiplier
        ];
    }
}