<?php
namespace App;
use App\Trader;

class BreakoutProbabilityKLinesCal
{
    // Propiedades privadas que almacenan los datos y configuraciones de la clase
    private array $data; // Array de datos históricos de velas (klines)
    private float $percentageStep = 1.0; // Paso porcentual para calcular niveles
    private int $numberOfLevels = 4; // Número de niveles alcistas y bajistas a calcular
    private bool $showZeroProbabilities = false; // Indica si se muestran niveles con probabilidad cero
    private int $atrPeriod = 14; // Período utilizado para calcular el ATR
    private bool $usarATR = false; // Indica si se utiliza el ATR para calcular niveles
    private bool $ponderacionTemporal = false; // Indica si se aplica ponderación temporal
    private bool $filtrarVolumen = false; // Indica si se filtran las velas según el volumen

    /**
     * Constructor de la clase.
     * Inicializa los datos y filtros opcionales.
     *
     * @param array $data Datos históricos ['open', 'high', 'low', 'close', 'volume']. Ordenados cronológicamente.
     * @param array $filters Filtros opcionales ['usarATR' => bool, 'ponderacionTemporal' => bool, 'filtrarVolumen' => bool]
     */
    public function __construct(array $data, array $filters = [])
    {
        if (count($data) < $this->atrPeriod + 2) { // +2 para tener vela actual y previa
            throw new \InvalidArgumentException("Se requieren al menos " . ($this->atrPeriod + 2) . " velas para los cálculos.");
        }
        $this->data = $data;
        $this->usarATR = $filters['usarATR'] ?? false;
        $this->ponderacionTemporal = $filters['ponderacionTemporal'] ?? false;
        $this->filtrarVolumen = $filters['filtrarVolumen'] ?? false;
    }

    // --- Setters para ajustar parámetros ---
    public function setPercentageStep(float $step): void {
        $this->percentageStep = $step;
    }

    public function setNumberOfLevels(int $levels): void {
        $this->numberOfLevels = $levels;
    }

    public function setShowZeroProbabilities(bool $show): void {
        $this->showZeroProbabilities = $show;
    }

    public function setAtrPeriod(int $period): void {
        $this->atrPeriod = $period;
    }

    /**
     * Método principal que realiza el análisis completo y devuelve los resultados en formato JSON.
     * @return string Resultados del análisis en formato JSON.
     */
    public function calculate(): string
    {
        $currentCandle = end($this->data);
        // Mover el puntero interno para obtener la vela previa REAL
        reset($this->data); // Reset pointer
        $tempData = $this->data; // Copia temporal si end()/prev() modifican el original
        end($tempData); // Mover al final
        $previosCandle = prev($tempData); // Obtener el penúltimo elemento

        // Validar que se obtuvo la vela previa
        if ($previosCandle === false || $currentCandle === false) {
             return json_encode(['error' => 'No se pudieron obtener las velas necesarias.'], JSON_PRETTY_PRINT);
        }

        $currentClose = (float) $currentCandle['close'];
        $isGreen = (float) $previosCandle['close'] >= (float) $previosCandle['open'];
        $highs = array_column($this->data, 'high');
        $lows = array_column($this->data, 'low');
        $closes = array_column($this->data, 'close');
        $atr = Trader::atr($highs, $lows, $closes, $this->atrPeriod);
        $currentATR = end($atr);
        
        // Asegurar que currentATR es numérico y positivo
        $currentATR = (is_numeric($currentATR) && $currentATR > 0) ? (float) $currentATR : 0.0;

        // Determinar el paso para los niveles
        $step = 0.0;
        if ($this->usarATR && $currentATR > 0) {
             $step = $currentATR;
        } elseif (!$this->usarATR && $currentClose > 0 && $this->percentageStep > 0) {
             $step = round($currentClose * ($this->percentageStep / 100.0),3);
        } else {
            $step = round($currentClose * 0.01,3); // Fallback a 1% si ATR es 0 o inválido y se pidió ATR
        }

        // Calcular niveles
        $upperLevels = $this->calculateLevels($step, true);
        $lowerLevels = $this->calculateLevels($step, false);

        // Calcula probabilidades de ruptura alcistas y bajistas
        $triggerColor = $isGreen ? 'green' : 'red';
        $upperProbabilities = $this->calculateProbabilities($upperLevels, 'high', $triggerColor);
        $lowerProbabilities = $this->calculateProbabilities($lowerLevels, 'low', $triggerColor);

        // Aquí podrías retornar los resultados en formato deseado
        return json_encode([
            'upperProbabilities' => $upperProbabilities,
            'lowerProbabilities' => $lowerProbabilities
        ], JSON_PRETTY_PRINT);
    }

    /**
     * Genera niveles alcistas o bajistas basados en un paso.
     * @param float $step El paso (ATR o % del precio).
     * @param bool $isUpper True para niveles superiores, false para inferiores.
     * @return array Lista de niveles relativos.
     */
    private function calculateLevels(float $step, bool $isUpper): array
    {
        $levels = [0.0]; // El nivel 0 siempre se incluye
        for ($i = 1; $i <= $this->numberOfLevels; $i++) {
            $absLevel = abs($step * $i);
            $levels[] = round($absLevel * ($isUpper ? 1 : -1), 5);
        }
        return $levels;
    }

    /**
     * Calcula las probabilidades de que se alcancen los niveles generados.
     * @param array $levels Niveles relativos.
     * @param string $priceType 'high' o 'low'.
     * @param string $triggerColor 'green' o 'red' (color de la vela i-1).
     * @return array Lista de ['level' => float, 'probability' => float].
     */
    private function calculateProbabilities(array $levels, string $priceType, string $triggerColor): array
    {
        $probabilities = [];
        $numData = count($this->data);
        if ($numData < 2) return []; // No hay pares de velas para comparar

        $volumenPromedio = 0;
        if ($this->filtrarVolumen) {
             $volumes = array_column($this->data, 'volume');
             $validVolumes = array_filter($volumes, 'is_numeric');
             $volumenPromedio = !empty($validVolumes) ? array_sum($validVolumes) / count($validVolumes) : 0;
        }

        $initialWeight = 1.0;
        $decayFactor = 0.995; // Factor de decaimiento para ponderación temporal

        foreach ($levels as $level) {
            $weightedCount = 0.0;
            $totalWeight = 0.0; // Suma de pesos de velas que coinciden con triggerColor
            $currentWeight = $initialWeight;

            // Iterar desde la penúltima vela hacia atrás
            for ($i = $numData - 2; $i >= 0; $i--) {
                $prevCandle = $this->data[$i]; // Vela disparadora (i)
                $candle = $this->data[$i + 1];  // Vela donde se mide el resultado (i+1)
                $prevIsGreen = (float) $prevCandle['close'] >= (float) $prevCandle['open'];
                $prevColor = $prevIsGreen ? 'green' : 'red';

                // Comprobar si la vela anterior coincide con el color buscado
                if ($prevColor === $triggerColor) {
                    $pesoActual = $this->ponderacionTemporal ? $currentWeight : 1.0;
                    $totalWeight += $pesoActual; // Contar esta vela para la base del %

                    // Comprobar filtro de volumen (si aplica)
                    $cumpleVolumen = !$this->filtrarVolumen || ($volumenPromedio > 0 && (float) $candle['volume'] >= $volumenPromedio);

                    // Comprobar si se alcanzó el nivel en la vela SIGUIENTE (candle)
                    $targetPrice = 0.0;
                    $conditionMet = false;
                    
                    if ($priceType === 'high') {
                        $targetPrice = (float)$prevCandle['high'] + $level;
                        $conditionMet = (float)$candle['high'] >= $targetPrice;
                    } else { // 'low'
                        $targetPrice = (float)$prevCandle['low'] + $level; // level es negativo aquí
                        $conditionMet = (float)$candle['low'] <= $targetPrice;
                    }

                    if ($conditionMet && $cumpleVolumen) {
                        $weightedCount += $pesoActual;
                    }
                }

                // Aplicar decaimiento para la siguiente iteración si aplica
                if ($this->ponderacionTemporal) {
                    $currentWeight *= $decayFactor;
                }
            } // Fin del bucle for

            $probability = ($totalWeight > 0) ? ($weightedCount / $totalWeight) * 100.0 : 0.0;
            
            if ($this->showZeroProbabilities || $probability > 0 || $level == 0) {
                $probabilities[] = [
                    'level' => $level, // Nivel relativo
                    'probability' => round($probability, 2),
                    'weightedCount' => $weightedCount
                ];
            }
        } // Fin del bucle foreach levels

        // Asegurarse de que los niveles están ordenados (ascendente para upper, descendente para lower)
        usort($probabilities, function ($a, $b) {
            return $a['level'] <=> $b['level'];
        });

        return $probabilities;
    }
}