<?php

namespace App;

use App\Trader; // Asegúrate de que esta clase Trader existe y tiene el método atr()

class BreakoutProbabilityKLines
{
    // Propiedades privadas que almacenan los datos y configuraciones de la clase
    private array $data; // Array de datos históricos de velas (klines)
    private float $percentageStep = 1.0; // Paso porcentual para calcular niveles
    private int $numberOfLevels = 4; // Número de niveles alcistas y bajistas a calcular
    private bool $showZeroProbabilities = false; // Indica si se muestran niveles con probabilidad cero
    private bool $noLevelZero = false; // Indica si se excluye el nivel 0 para calcular probabilidad
    private bool $printStats = false; // Indica si se muestran estadísticas adicionales
    private float $stopLossThreshold = 10.0; // Umbral de probabilidad para sugerir stop loss
    private int $atrPeriod = 14; // Período utilizado para calcular el ATR
    private float $tpSlRatio = 2.5; // Ratio de riesgo/recompensa para calcular take profit
    private float $trendThreshold = 50.0; // Umbral de probabilidad para determinar la tendencia
    private array $trend; // Almacena la tendencia actual del mercado
    private bool $usarATR = false; // Indica si se utiliza el ATR para calcular niveles
    private bool $ponderacionTemporal = false; // Indica si se aplica ponderación temporal
    private bool $filtrarVolumen = false; // Indica si se filtran las velas según el volumen
    private float $minConfidence = 25.0; // Confianza mínima requerida para generar señales
    private float $minSingleLevel = 15.0; // Probabilidad mínima para considerar un nivel significativo
    private string $marketCondition = 'normal'; // Condición actual del mercado

    // Propiedades para almacenar probabilidades calculadas para el panel de stats (Opción 2)
    private ?array $upperProbabilities = null;
    private ?array $lowerProbabilities = null;

    /**
     * Constructor de la clase.
     * Inicializa los datos y filtros opcionales.
     *
     * @param array $data Datos históricos ['open', 'high', 'low', 'close', 'volume']. Ordenados cronológicamente.
     * @param array $filters Filtros opcionales ['usarATR' => bool, 'ponderacionTemporal' => bool, 'filtrarVolumen' => bool]
     */
    public function __construct(array $data, array $filters = [])
    {
        if (count($data) < $this->atrPeriod + 2) { // +2 para tener vela actual y previa
            throw new \InvalidArgumentException("Se requieren al menos " . ($this->atrPeriod + 2) . " velas para los cálculos.");
        }
        $this->data = $data;
        $this->usarATR = $filters['usarATR'] ?? false;
        $this->ponderacionTemporal = $filters['ponderacionTemporal'] ?? false;
        $this->filtrarVolumen = $filters['filtrarVolumen'] ?? false;

        $this->analyzeMarketCondition(); // Evalúa la condición del mercado
    }

    // --- Setters para ajustar parámetros ---
    public function setPercentageStep(float $step): void {$this->percentageStep = $step;}
    public function setNumberOfLevels(int $levels): void {$this->numberOfLevels = $levels;}
    public function setShowZeroProbabilities(bool $show): void {$this->showZeroProbabilities = $show;}
    public function setNoLevelZero(bool $nozero): void {$this->noLevelZero = $nozero;}
    public function setShowStats(bool $show): void {$this->printStats = $show;}
    public function setStopLossThreshold(float $threshold): void {$this->stopLossThreshold = $threshold;}
    public function setAtrPeriod(int $period): void {$this->atrPeriod = $period;}
    public function setTpSlRatio(float $ratio): void {$this->tpSlRatio = $ratio;}
    public function setTrendThreshold(float $threshold): void {$this->trendThreshold = $threshold;}
    public function setMinConfidence(float $confidence): void {$this->minConfidence = $confidence;}
    public function setMinSingleLevel(float $level): void {$this->minSingleLevel = $level;}

    /**
     * Analiza la condición del mercado en función de la volatilidad y la fuerza de la tendencia.
     */
    private function analyzeMarketCondition(): void
    {
        $volatility = $this->calculateVolatility(); // Usa la media del ATR
        $trendStrength = $this->calculateTrendStrength(); // Usa cambio porcentual simple

        // Ajusta umbrales basados en ATR relativo al precio actual
        $currentPrice = end($this->data)['close'];
        $relativeVolatility = ($currentPrice > 0) ? ($volatility / $currentPrice) * 100 : 0; // Volatilidad como % del precio

        // Umbrales ajustables (ejemplos, requieren optimización)
        $sidewaysThreshold = 0.5; // Ejemplo: ATR medio < 0.5% del precio
        $volatileThreshold = 2.0; // Ejemplo: ATR medio > 2.0% del precio
        if ($relativeVolatility < $sidewaysThreshold && $relativeVolatility > 0) {
            $this->marketCondition = 'sideways';
            // Reducir confianza mínima en lateral puede ser contraproducente, considerar aumentarla
            // $this->minConfidence = 20; // Ejemplo original, revisar lógica
        } elseif ($relativeVolatility > $volatileThreshold) {
            $this->marketCondition = 'volatile';
            $this->minConfidence = 30; // Aumentar requerimiento en volatilidad alta
        } else {
            $this->marketCondition = 'normal';
            $this->minConfidence = 25; // Confianza base para normal
        }

        // Podría usarse $trendStrength aquí también para refinar la condición
    }

    /**
     * Calcula la volatilidad del mercado utilizando la media del ATR.
     * @return float Media del ATR o 0 si no se puede calcular.
     */
    private function calculateVolatility(): float
    {
        $highs = array_column($this->data, 'high');
        $lows = array_column($this->data, 'low');
        $closes = array_column($this->data, 'close');

        // Asegurarse de que Trader::atr maneja arrays vacíos o con pocos datos
        $atrValues = Trader::atr($highs, $lows, $closes, $this->atrPeriod);

        $validAtrValues = array_filter($atrValues, 'is_numeric'); // Filtrar NaNs o nulls si los hubiera
        if (empty($validAtrValues)) {
            return 0.0;
        }
        return array_sum($validAtrValues) / count($validAtrValues);
    }

    /**
     * Evalúa la fuerza de la tendencia comparando precios de cierre (método simple).
     * @return float Fuerza de tendencia (cambio porcentual absoluto) o 0.
     */
    private function calculateTrendStrength(): float
    {
        $closes = array_column($this->data, 'close');
        $numCandles = count($closes);

        if ($numCandles < 50) return 0.0; // Necesita suficientes datos

        $period = intval($numCandles / 3); // O usar un periodo fijo, ej. 50
        if ($period < 1) return 0.0;

        $lastSlice = array_slice($closes, -$period);
        $firstSlice = array_slice($closes, 0, $period);

        $avgLast = array_sum($lastSlice) / count($lastSlice);
        $avgFirst = array_sum($firstSlice) / count($firstSlice);

        if ($avgFirst == 0) return 0.0; // Evitar división por cero

        $trendChange = (($avgLast - $avgFirst) / $avgFirst) * 100;
        return abs($trendChange);
    }

    /**
     * Método principal que realiza el análisis completo y devuelve los resultados en formato JSON.
     * @return string Resultados del análisis en formato JSON.
     */
    public function calculate(): string
    {
        $currentCandle = end($this->data);
        // Mover el puntero interno para obtener la vela previa REAL
        reset($this->data); // Reset pointer
        $tempData = $this->data; // Copia temporal si end()/prev() modifican el original
        end($tempData); // Mover al final
        $previosCandle = prev($tempData); // Obtener el penúltimo elemento

        // Validar que se obtuvo la vela previa
        if ($previosCandle === false || $currentCandle === false) {
             return json_encode(['error' => 'No se pudieron obtener las velas necesarias.'], JSON_PRETTY_PRINT);
        }

        $currentClose = (float) $currentCandle['close'];
        $isGreen = (float) $previosCandle['close'] >= (float) $previosCandle['open'];

        $highs = array_column($this->data, 'high');
        $lows = array_column($this->data, 'low');
        $closes = array_column($this->data, 'close');

        $atr = Trader::atr($highs, $lows, $closes, $this->atrPeriod);
        $currentATR = end($atr);
        // Asegurar que currentATR es numérico y positivo
        $currentATR = (is_numeric($currentATR) && $currentATR > 0) ? (float) $currentATR : 0.0;

        // Determinar el paso para los niveles
        $step = 0.0;
        if ($this->usarATR && $currentATR > 0) {
             $step = $currentATR;
        } elseif (!$this->usarATR && $currentClose > 0 && $this->percentageStep > 0) {
             $step = round($currentClose * ($this->percentageStep / 100.0),3);
        } else {
            // Fallback o error si no se puede determinar un step válido
            // return json_encode(['error' => 'No se pudo determinar un step válido para los niveles.'], JSON_PRETTY_PRINT);
             $step = round($currentClose * 0.01,3); // Fallback a 1% si ATR es 0 o inválido y se pidió ATR
        }

        // Calcular niveles
        $upperLevels = $this->calculateLevels($step, true);
        $lowerLevels = $this->calculateLevels($step, false);

        // Calcula probabilidades de ruptura alcistas y bajistas y las guarda
        $triggerColor = $isGreen ? 'green' : 'red';
        $this->upperProbabilities = $this->calculateProbabilities($upperLevels, 'high', $triggerColor);
        $this->lowerProbabilities = $this->calculateProbabilities($lowerLevels, 'low', $triggerColor);
        // Determina la tendencia, stop loss, take profit y señal de entrada
        $this->trend = $this->determineTrend($this->upperProbabilities, $this->lowerProbabilities);
        $stopLoss = $this->suggestStopLoss($this->upperProbabilities, $this->lowerProbabilities, $isGreen, $currentATR);
        $takeProfit = $this->suggestTakeProfit($stopLoss, $currentClose);
        $entrySignal = $this->determineEntrySignal($isGreen, $this->upperProbabilities, $this->lowerProbabilities, $currentATR, $currentClose);

        // Añadir precios absolutos a SL y TP después de que se hayan calculado
        $stopLoss = $this->addAbsolutePricesToSL($stopLoss, $currentClose);
        $takeProfit = $this->addAbsolutePricesToTP($takeProfit, $currentClose);

        $mostProbableLevel = $this->getMostProbableLevel($this->upperProbabilities, $this->lowerProbabilities);

        // Construye el resultado final
        $result = [
            'current_price' => $currentClose,
            'current_color' => $triggerColor, // Color de la vela PREVIA
            'entry_signal' => $entrySignal,
            'upper_levels' => $this->addAbsolutePrices($this->upperProbabilities, $currentClose, true),
            'lower_levels' => $this->addAbsolutePrices($this->lowerProbabilities, $currentClose, false),
            'most_probable_level' => $mostProbableLevel ? $this->enhanceMostProbableLevel($mostProbableLevel, $currentClose) : null,
            'stop_loss' => $stopLoss,
            'take_profit' => $takeProfit,
            'atr' => round($currentATR, 5), // Mayor precisión para ATR
            'trend' => $this->trend,
            // Calcular R:R sólo si hay entrada y SL/TP válidos
            'risk_reward_ratio' => ($entrySignal['should_enter'] && $stopLoss['price'] !== null && $takeProfit['price'] !== null)
                                    ? $this->calculateRiskRewardRatio($stopLoss, $takeProfit, $currentClose)
                                    : null,
            'market_condition' => $this->marketCondition
        ];

        // Agrega estadísticas adicionales si están habilitadas (Opción 2)
        if ($this->printStats) {
            $result['stats'] = $this->printStatsPanel();
        }

        return json_encode($result, JSON_PRETTY_PRINT | JSON_INVALID_UTF8_IGNORE);
    }

    /**
     * Genera niveles alcistas o bajistas basados en un paso.
     * @param float $step El paso (ATR o % del precio).
     * @param bool $isUpper True para niveles superiores, false para inferiores.
     * @return array Lista de niveles relativos.
     */
    private function calculateLevels(float $step, bool $isUpper): array
    {
        $levels = [];
        // El nivel 0 siempre se incluye
        $levels[] = 0.0;
        for ($i = 1; $i <= $this->numberOfLevels; $i++) {
            $absLevel = abs($step * $i);
            // Usar precisión adecuada, ej. 5 decimales
            $levels[] = round($absLevel * ($isUpper ? 1 : -1), 5);
        }
        return $levels;
    }

    /**
     * Calcula las probabilidades de que se alcancen los niveles generados.
     * @param array $levels Niveles relativos.
     * @param string $priceType 'high' o 'low'.
     * @param string $triggerColor 'green' o 'red' (color de la vela i-1).
     * @return array Lista de ['level' => float, 'probability' => float].
     */
    private function calculateProbabilities(array $levels, string $priceType, string $triggerColor): array
    {
        $probabilities = [];
        $numData = count($this->data);
        if ($numData < 2) return []; // No hay pares de velas para comparar

        $volumenPromedio = 0;
        if ($this->filtrarVolumen) {
             $volumes = array_column($this->data, 'volume');
             $validVolumes = array_filter($volumes, 'is_numeric');
             $volumenPromedio = !empty($validVolumes) ? array_sum($validVolumes) / count($validVolumes) : 0;
        }

        $initialWeight = 1.0;
        $decayFactor = 0.995; // Factor de decaimiento para ponderación temporal

        foreach ($levels as $level) {
            $weightedCount = 0.0;
            $totalWeight = 0.0; // Suma de pesos de velas que coinciden con triggerColor

            $currentWeight = $initialWeight;

            // Iterar desde la penúltima vela hacia atrás
            for ($i = $numData - 2; $i >= 0; $i--) {
                $prevCandle = $this->data[$i]; // Vela disparadora (i)
                $candle = $this->data[$i + 1];  // Vela donde se mide el resultado (i+1)

                $prevIsGreen = (float) $prevCandle['close'] >= (float) $prevCandle['open'];
                $prevColor = $prevIsGreen ? 'green' : 'red';

                // Comprobar si la vela anterior coincide con el color buscado
                if ($prevColor === $triggerColor) {
                    $pesoActual = $this->ponderacionTemporal ? $currentWeight : 1.0;
                    $totalWeight += $pesoActual; // Contar esta vela para la base del %

                    // Comprobar filtro de volumen (si aplica)
                    $cumpleVolumen = !$this->filtrarVolumen || ($volumenPromedio > 0 && (float) $candle['volume'] >= $volumenPromedio);

                    // Comprobar si se alcanzó el nivel en la vela SIGUIENTE (candle)
                    // relativo al high/low de la vela ANTERIOR (prevCandle)
                    $targetPrice = 0.0;
                    $conditionMet = false;
                    if ($priceType === 'high') {
                        $targetPrice = (float)$prevCandle['high'] + $level;
                        $conditionMet = (float)$candle['high'] >= $targetPrice;
                    } else { // 'low'
                        $targetPrice = (float)$prevCandle['low'] + $level; // level es negativo aquí
                        $conditionMet = (float)$candle['low'] <= $targetPrice;
                    }

                    if ($conditionMet && $cumpleVolumen) {
                        $weightedCount += $pesoActual;
                    }
                }

                // Aplicar decaimiento para la siguiente iteración si aplica
                if ($this->ponderacionTemporal) {
                    $currentWeight *= $decayFactor;
                }
            } // Fin del bucle for
            $probability = ($totalWeight > 0) ? ($weightedCount / $totalWeight) * 100.0 : 0.0;

            if ($this->showZeroProbabilities || $probability > 0 || $level == 0) {
                $probabilities[] = [
                    'level' => $level, // Nivel relativo
                    'probability' => round($probability, 2),
                    'weightedCount' => $weightedCount
                ];
            }
        } // Fin del bucle foreach levels

        // Asegurarse de que los niveles están ordenados (ascendente para upper, descendente para lower)
         usort($probabilities, function ($a, $b) {
            return $a['level'] <=> $b['level'];
         });

        return $probabilities;
    }

    /**
     * Encuentra el nivel más probable entre los alcistas y bajistas (excluyendo nivel 0).
     * @param array|null $upperProbabilities
     * @param array|null $lowerProbabilities
     * @return array|null El nivel más probable o null.
     */
    private function getMostProbableLevel(?array $upperProbabilities, ?array $lowerProbabilities): ?array
    {
         $allLevels = [];
         $exclud = $this->noLevelZero?0:-1;
         if ($upperProbabilities) {
             foreach ($upperProbabilities as $level) {
                 if ($level['level'] != $exclud) { // Excluir nivel 0
                      $allLevels[] = ['type' => 'upper', ...$level];
                 }
             }
         }
          if ($lowerProbabilities) {
             foreach ($lowerProbabilities as $level) {
                  if ($level['level'] != $exclud) { // Excluir nivel 0
                      $allLevels[] = ['type' => 'lower', ...$level];
                 }
             }
         }

        if (!$this->showZeroProbabilities) {
            $allLevels = array_filter($allLevels, function ($level) {
                return $level['probability'] > 0;
            });
        }

        if (empty($allLevels)) {
            return null;
        }

        usort($allLevels, function ($a, $b) {
            // Ordenar por probabilidad descendente
            return $b['probability'] <=> $a['probability'];
        });
        return reset($allLevels); // Devuelve el primero (el más probable)
    }

    /**
     * Mejora el nivel más probable añadiendo el precio absoluto.
     * @param array $level El nivel probable.
     * @param float $currentPrice Precio actual.
     * @return array Nivel mejorado.
     */
    private function enhanceMostProbableLevel(array $level, float $currentPrice): array
    {
        $level['absolute_price'] = $this->calculateAbsolutePrice(
            $currentPrice,
            $level['level'],
            $level['type'] === 'upper' // Pasar bool
        );
        return $level;
    }

    /**
     * Determina la señal de entrada (long, short o neutral).
     * @param bool $isPrevGreen Color de la vela PREVIA.
     * @param array|null $upperProbabilities
     * @param array|null $lowerProbabilities
     * @param float $atr ATR actual.
     * @param float $currentPrice Precio actual.
     * @return array Descripción de la señal.
     */
    private function determineEntrySignal(bool $isPrevGreen, ?array $upperProbabilities, ?array $lowerProbabilities, float $atr, float $currentPrice): array
    {
        $signal = 'neutral';
        $confidence = 0.0;
        $reason = '';
        $exclud = $this->noLevelZero?0:-1;
        // Filtrar niveles relevantes (no-cero y con alguna probabilidad si showZero=false)
        $relevantUpper = $upperProbabilities ? array_values(array_filter($upperProbabilities, fn($l) => $l['level'] != $exclud && ($this->showZeroProbabilities || $l['probability'] > 0))) : [];
        $relevantLower = $lowerProbabilities ? array_values(array_filter($lowerProbabilities, fn($l) => $l['level'] != $exclud && ($this->showZeroProbabilities || $l['probability'] > 0))) : [];

        $bestUpper = $this->getBestLevel($relevantUpper); // Mejor probabilidad entre relevantes
        $bestLower = $this->getBestLevel($relevantLower); // Mejor probabilidad entre relevantes

        // Filtro de volatilidad mínima (ej: ATR < 0.1% del precio es muy bajo)
        $minAtrThreshold = $currentPrice * 0.001; // 0.1% - Ajustar según sea necesario
        if ($atr < $minAtrThreshold && $atr > 0) {
            return [
                'action' => 'neutral',
                'confidence' => 0,
                'description' => 'Market without sufficient volatility (very low ATR)',
                'should_enter' => false,
                'reason' => sprintf('ATR %.5f < threshold %.5f (%.2f%%)', $atr, $minAtrThreshold, ($minAtrThreshold/$currentPrice)*100),
                'market_condition' => $this->marketCondition
            ];
        }

        // Lógica de señal basada en el color de la VELA PREVIA
        if ($isPrevGreen) { // Si la vela previa fue verde, buscamos LONG
            if ($bestUpper && $bestUpper['probability'] >= $this->minSingleLevel) {
                $signal = 'long';
                $confidence = $bestUpper['probability']; // Confianza base
                $reason = sprintf("Bullish level %.5f (%.2f%%)", $bestUpper['level'], $bestUpper['probability']);

                // Añadir confianza por niveles adicionales con > X% prob
                $additionalLevels = count(array_filter($relevantUpper, fn($l) => $l['probability'] > 10 && $l['level'] != $bestUpper['level']));
                if ($additionalLevels > 0) {
                    $confidence += $additionalLevels * 5; // Sumar 5 puntos por cada nivel adicional
                    $reason .= sprintf(" + %d additional levels (>10%%)", $additionalLevels);
                }
                 // Añadir confianza si la tendencia acompaña
                 if ($this->trend['type'] === 'bullish') {
                      // Ponderar por la probabilidad de la tendencia
                      $confidence += $this->trend['probability'] * 0.3; // Ej: 30% de la prob. de trend
                      $reason .= sprintf(" + trend bullish (%.2f%%)", $this->trend['probability']);
                 }
            }
        } else { // Si la vela previa fue roja, buscamos SHORT
            if ($bestLower && $bestLower['probability'] >= $this->minSingleLevel) {
                $signal = 'short';
                $confidence = $bestLower['probability'];
                $reason = sprintf("Bearish level %.5f (%.2f%%)", $bestLower['level'], $bestLower['probability']);

                $additionalLevels = count(array_filter($relevantLower, fn($l) => $l['probability'] > 10 && $l['level'] != $bestLower['level']));
                 if ($additionalLevels > 0) {
                    $confidence += $additionalLevels * 5;
                    $reason .= sprintf(" + %d additional levels (>10%%)", $additionalLevels);
                }
                 if ($this->trend['type'] === 'bearish') {
                      $confidence += $this->trend['probability'] * 0.3;
                      $reason .= sprintf(" + trend bearish (%.2f%%)", $this->trend['probability']);
                 }
            }
        }

        // Normalizar confianza entre 0 y 100
        $confidence = min(max($confidence, 0), 100);

        // Determinar si se debe entrar basado en la confianza mínima ajustada por condición de mercado
        $shouldEnter = $confidence >= $this->minConfidence;

        // Razón si no hay señal
        if ($signal === 'neutral') {
            $reason = $this->getNoSignalReason($bestUpper, $bestLower);
        }

        return [
            'action' => $signal,
            'confidence' => round($confidence, 2),
            'description' => $this->getSignalDescription($signal, $confidence),
            'should_enter' => $shouldEnter,
            'reason' => $reason,
            'market_condition' => $this->marketCondition // Incluir condición que afectó minConfidence
        ];
    }

    /**
     * Encuentra el mejor nivel (mayor probabilidad) entre una lista de niveles relevantes.
     * @param array $levels Lista de niveles ['level'=>..., 'probability'=>...].
     * @return array|null El mejor nivel o null si la lista está vacía.
     */
    private function getBestLevel(array $levels): ?array
    {
        if (empty($levels)) return null;
        // Ordenar por probabilidad descendente
        usort($levels, fn($a, $b) => $b['probability'] <=> $a['probability']);
        return $levels[0]; // Devuelve el de mayor probabilidad
    }

    /**
     * Genera una razón textual cuando no hay señales claras.
     * @param array|null $bestUpper
     * @param array|null $bestLower
     * @return string Razón.
     */
    private function getNoSignalReason(?array $bestUpper, ?array $bestLower): string
    {
        $reasons = [];
        $minProb = $this->minSingleLevel; // Umbral mínimo para considerarlo "significativo" en la razón

        if ($bestUpper && $bestUpper['probability'] >= $minProb) {
            $reasons[] = sprintf("Best bullish: %.5f (%.2f%%)", $bestUpper['level'], $bestUpper['probability']);
        } elseif ($bestUpper) {
             $reasons[] = sprintf("Best bullish (low threshold): %.5f (%.2f%% < %.1f%%)", $bestUpper['level'], $bestUpper['probability'], $minProb);
        } else {
            $reasons[] = "No significant bullish levels";
        }

        if ($bestLower && $bestLower['probability'] >= $minProb) {
            $reasons[] = sprintf("Best bearish: %.5f (%.2f%%)", $bestLower['level'], $bestLower['probability']);
        } elseif ($bestLower) {
             $reasons[] = sprintf("Best bearish (low threshold): %.5f (%.2f%% < %.1f%%)", $bestLower['level'], $bestLower['probability'], $minProb);
        } else {
            $reasons[] = "No significant bearish levels";
        }
        return implode(" - ", $reasons);
    }

    /**
     * Genera una descripción textual de la señal de entrada.
     * @param string $signal 'long', 'short', 'neutral'.
     * @param float $confidence Confianza calculada (0-100).
     * @return string Descripción.
     */
    private function getSignalDescription(string $signal, float $confidence): string
    {
        // Umbrales de confianza para descripción (ajustables)
        $highConfidenceThreshold = 70.0;
        $mediumConfidenceThreshold = 40.0;

        $descriptions = [
            'long' => [
                'high' => 'Strong buy signal (high confidence)',
                'medium' => 'Buy signal (moderate confidence)',
                'low' => 'Weak buy signal'
            ],
            'short' => [
                'high' => 'Strong sell signal (high confidence)',
                'medium' => 'Sell signal (moderate confidence)',
                'low' => 'Weak sell signal'
            ],
            'neutral' => 'Wait for a clearer signal'
        ];

        if ($signal === 'neutral') return $descriptions['neutral'];

        $level = 'low'; // Nivel por defecto
        if ($confidence >= $highConfidenceThreshold) {
            $level = 'high';
        } elseif ($confidence >= $mediumConfidenceThreshold) {
            $level = 'medium';
        }

        return $descriptions[$signal][$level];
    }

    /**
     * Determina la tendencia del mercado en función de las probabilidades acumuladas.
     * @param array|null $upperProbabilities
     * @param array|null $lowerProbabilities
     * @return array ['type' => string, 'probability' => float]
     */
    private function determineTrend(?array $upperProbabilities, ?array $lowerProbabilities): array
    {
        // Calcular probabilidad media de niveles relevantes (no-cero)
        $relevantUpper = $upperProbabilities ? array_filter($upperProbabilities, fn($l) => $l['level'] != 0) : [];
        $relevantLower = $lowerProbabilities ? array_filter($lowerProbabilities, fn($l) => $l['level'] != 0) : [];

        $avgUpperProb = 0.0;
        if (!empty($relevantUpper)) {
             $avgUpperProb = array_sum(array_column($relevantUpper, 'probability')) / count($relevantUpper);
        }

        $avgLowerProb = 0.0;
        if (!empty($relevantLower)) {
             $avgLowerProb = array_sum(array_column($relevantLower, 'probability')) / count($relevantLower);
        }

        // Determinar tendencia basada en la probabilidad media y el umbral
        if ($avgUpperProb > $this->trendThreshold && $avgUpperProb > $avgLowerProb) {
            return ['type' => 'bullish', 'probability' => round($avgUpperProb, 2)];
        } elseif ($avgLowerProb > $this->trendThreshold && $avgLowerProb > $avgUpperProb) {
            return ['type' => 'bearish', 'probability' => round($avgLowerProb, 2)];
        } else {
            // Podríamos añadir una condición de lateral si ambas son bajas
            return ['type' => 'neutral', 'probability' => 0.0];
        }
    }

    /**
     * Sugiere un nivel de stop loss basado en probabilidades o el ATR.
     * @param array|null $upperProbabilities
     * @param array|null $lowerProbabilities
     * @param bool $isPrevGreen Color de la vela PREVIA.
     * @param float $currentATR ATR actual.
     * @return array Nivel de SL sugerido ['type', 'level', 'probability', 'source', 'price'=>null].
     */
    private function suggestStopLoss(?array $upperProbabilities, ?array $lowerProbabilities, bool $isPrevGreen, float $currentATR): array
    {
        $stopLoss = ['type' => 'none', 'level' => null, 'probability' => null, 'source' => null, 'price' => null];
        $foundLevelSL = false;

        if ($isPrevGreen) { // Buscamos SL por debajo (niveles lower)
             if ($lowerProbabilities) {
                 // Buscar el primer nivel inferior con probabilidad <= umbral
                 foreach ($lowerProbabilities as $level) {
                     if ($level['level'] < 0 && $level['probability'] <= $this->stopLossThreshold) {
                         $stopLoss = [
                             'type' => 'lower',
                             'level' => $level['level'],
                             'probability' => $level['probability'],
                             'source' => 'nivel_prob', // Fuente: Nivel de baja probabilidad
                             'price' => null
                         ];
                         $foundLevelSL = true;
                         break; // Usar el primero que cumpla
                     }
                 }
             }
        } else { // Buscamos SL por encima (niveles upper)
             if ($upperProbabilities) {
                 // Buscar el primer nivel superior con probabilidad <= umbral
                 foreach ($upperProbabilities as $level) {
                     if ($level['level'] > 0 && $level['probability'] <= $this->stopLossThreshold) {
                         $stopLoss = [
                             'type' => 'upper',
                             'level' => $level['level'],
                             'probability' => $level['probability'],
                             'source' => 'level_prob',
                             'price' => null
                         ];
                         $foundLevelSL = true;
                         break; // Usar el primero que cumpla
                     }
                 }
            }
        }

        // Fallback a ATR si no se encontró nivel por probabilidad o si ATR es más conservador
        // O si el ATR es válido
        if (!$foundLevelSL && $currentATR > 0) {
            // Colocar SL a 1x ATR (o un múltiplo configurable)
            $atrMultiplier = 1.0;
            $slLevelBasedOnATR = round($currentATR * $atrMultiplier, 5);

            $stopLoss = [
                'type' => $isPrevGreen ? 'lower' : 'upper',
                'level' => $isPrevGreen ? -$slLevelBasedOnATR : $slLevelBasedOnATR,
                'probability' => null, // No basado en probabilidad directa
                'source' => 'level_ATR',
                'price' => null
            ];
        } elseif ($foundLevelSL && $currentATR > 0) {
            // Opcional: Comparar SL por probabilidad vs SL por ATR y elegir el más lejano/seguro?
            // $slDistProb = abs($stopLoss['level']);
            // $slDistATR = $currentATR * $atrMultiplier;
            // if (($isPrevGreen && -$slDistATR < $stopLoss['level']) || (!$isPrevGreen && $slDistATR > $stopLoss['level'])) {
            //     // Si el ATR da un SL más lejano, usar ATR? Depende de la estrategia.
            // }
        }


        return $stopLoss;
    }

    /**
     * Sugiere un nivel de take profit basado en el SL y el ratio R:R.
     * @param array $stopLoss El SL calculado.
     * @param float $currentPrice Precio actual.
     * @return array|null Nivel de TP sugerido ['type', 'level', 'price'=>null] o null.
     */
    private function suggestTakeProfit(array $stopLoss, float $currentPrice): ?array
    {
        if ($stopLoss['type'] === 'none' || $stopLoss['level'] === null) {
             return null;
        }

        $distanceToStopLoss = abs($stopLoss['level']);
        if ($distanceToStopLoss == 0) return null; // Evitar TP infinito si SL está en 0

        $takeProfitDistance = $distanceToStopLoss * $this->tpSlRatio;

        // El TP está en la dirección opuesta al SL
        if ($stopLoss['type'] === 'lower') { // SL abajo, TP arriba
            return [
                'type' => 'upper',
                'level' => round($takeProfitDistance, 5),
                'price' => null
            ];
        } else { // SL arriba, TP abajo
            return [
                'type' => 'lower',
                'level' => round(-$takeProfitDistance, 5),
                'price' => null
            ];
        }
    }

    /**
     * Calcula el ratio de riesgo/recompensa.
     * @param array $stopLoss SL con precio absoluto.
     * @param array $takeProfit TP con precio absoluto.
     * @param float $currentPrice Precio actual (punto de entrada teórico).
     * @return float|null Ratio R:R o null si no se puede calcular.
     */
    private function calculateRiskRewardRatio(array $stopLoss, array $takeProfit, float $currentPrice): ?float
    {
        if ($stopLoss['price'] === null || $takeProfit['price'] === null) {
            return null;
        }

        $risk = abs($currentPrice - $stopLoss['price']);
        $reward = abs($takeProfit['price'] - $currentPrice);

        if ($risk == 0) {
            return null; // Evitar división por cero
        }

        return round($reward / $risk, 2);
    }

    /**
     * Añade el precio absoluto al array de stop loss.
     * @param array $stopLoss Array de SL.
     * @param float $currentPrice Precio actual.
     * @return array SL con 'price' añadido.
     */
    private function addAbsolutePricesToSL(array $stopLoss, float $currentPrice): array
    {
        if ($stopLoss['type'] !== 'none' && $stopLoss['level'] !== null) {
            $stopLoss['price'] = $this->calculateAbsolutePrice(
                $currentPrice,
                $stopLoss['level'],
                $stopLoss['type'] === 'upper' // Pasar bool
            );
        }
        return $stopLoss;
    }

    /**
     * Añade el precio absoluto al array de take profit.
     * @param array|null $takeProfit Array de TP o null.
     * @param float $currentPrice Precio actual.
     * @return array|null TP con 'price' añadido o null.
     */
    private function addAbsolutePricesToTP(?array $takeProfit, float $currentPrice): ?array
    {
        if ($takeProfit !== null && $takeProfit['level'] !== null) {
             $takeProfit['price'] = $this->calculateAbsolutePrice(
                $currentPrice,
                $takeProfit['level'],
                $takeProfit['type'] === 'upper' // Pasar bool
            );
        }
        return $takeProfit;
    }

    /**
     * Calcula el precio absoluto a partir de un nivel relativo.
     * @param float $currentPrice Precio de referencia.
     * @param float $relativeLevel Nivel relativo (+ para upper, - para lower).
     * @param bool $isUpper True si el nivel es superior.
     * @return float Precio absoluto.
     */
    private function calculateAbsolutePrice(float $currentPrice, float $relativeLevel, bool $isUpper): float
    {
        // Nivel relativo ya tiene el signo correcto
         $absolutePrice = $currentPrice + $relativeLevel;
        // Redondear a una precisión razonable para precios
        return round($absolutePrice, 5);
    }

    /**
     * Añade precios absolutos a una lista de niveles relativos.
     * @param array|null $levels Lista de niveles ['level'=>..., 'probability'=>...].
     * @param float $currentPrice Precio actual.
     * @param bool $isUpper True si son niveles superiores.
     * @return array Lista de niveles con 'price' añadido.
     */
    private function addAbsolutePrices(?array $levels, float $currentPrice, bool $isUpper): array
    {
        if ($levels === null) return [];

        return array_map(function ($level) use ($currentPrice, $isUpper) {
            $level['price'] = $this->calculateAbsolutePrice($currentPrice, $level['level'], $isUpper);
            return $level;
        }, $levels);
    }

    /**
     * Genera un panel de estadísticas adicionales (Opción 2).
     * @return array Panel de estadísticas.
     */
    private function printStatsPanel(): array
    {
        // Usar las probabilidades almacenadas en las propiedades de la clase
        return [
            'total_candles_analyzed' => count($this->data), // O count($this->data) - 1 si la última no se usa en probabilidad
            'prob_first_upper_hit_now' => $this->getFirstLevelProbability($this->upperProbabilities, 1) . '%', // Probabilidad del nivel índice 1
            'prob_first_lower_hit_now' => $this->getFirstLevelProbability($this->lowerProbabilities, 1) . '%', // Probabilidad del nivel índice 1
        ];
    }

     /**
     * Helper para obtener la probabilidad del nivel N desde los resultados ya calculados.
     * Busca el nivel en el índice especificado (asumiendo que el índice 0 es el nivel 0.0).
     *
     * @param array|null $probabilities Array de ['level'=>..., 'probability'=>...] o null.
     * @param int $index Índice del nivel a buscar (ej: 1 para el primer nivel no-cero).
     * @return float La probabilidad (0-100) o 0.0 si no se encuentra o no aplica.
     */
    private function getFirstLevelProbability(?array $probabilities, int $index): float {
        // Validar que $probabilities es un array y el índice existe
        if (!is_array($probabilities) || !isset($probabilities[$index])) {
            return 0.0;
        }
        // Podríamos añadir una comprobación extra para asegurarnos que el nivel en ese índice no es 0,
        // pero generalmente el índice 1 corresponderá al primer nivel calculado distinto de cero.
        // if (isset($probabilities[$index]['level']) && $probabilities[$index]['level'] == 0 && $index > 0) {
        //     // Esto no debería pasar si calculateLevels() funciona bien
        //     return 0.0; // O buscar el siguiente índice
        // }

        // Devolver la probabilidad de ese índice
        return $probabilities[$index]['probability'] ?? 0.0;
    }

} // Fin de la clase BreakoutProbabilityKLines