<?php

namespace App;

use App\Trader; // Asegúrate de que esta clase Trader existe y tiene el método atr()

class BreakoutProbabilityKLinesStrategy
{
    //private ImpulseMACDWS $impulseMacd; // Objeto ImpulseMACDWS ya calculado
    // Propiedades privadas que almacenan los datos y configuraciones de la clase
    private array $data; // Array de datos históricos de velas (klines)
    private bool $showZeroProbabilities = false; // Indica si se muestran niveles con probabilidad cero
    private bool $noLevelZero = false; // Indica si se excluye el nivel 0 para calcular probabilidad
    private bool $printStats = false; // Indica si se muestran estadísticas adicionales
    private float $stopLossThreshold = 10.0; // Umbral de probabilidad para sugerir stop loss
    private int $atrPeriod = 14; // Período utilizado para calcular el ATR
    private float $tpSlRatio = 2.5; // Ratio de riesgo/recompensa para calcular take profit
    private float $trendThreshold = 50.0; // Umbral de probabilidad para determinar la tendencia
    private array $trend; // Almacena la tendencia actual del mercado
    private bool $usarATR = false; // Indica si se utiliza el ATR para calcular niveles (Este flag es el único usado de los filtros originales)
    private float $minConfidence = 25.0; // Confianza mínima requerida para generar señales
    private float $minSingleLevel = 15.0; // Probabilidad mínima para considerar un nivel significativo
    private string $marketCondition = 'normal'; // Condición actual del mercado

    // Propiedades para almacenar probabilidades calculadas para el panel de stats
    private ?array $upperProbabilities = null;
    private ?array $lowerProbabilities = null;

    /**
     * Constructor de la clase.
     * Inicializa los datos y filtros opcionales.
     *
     * @param ImpulseMACDWS $impulseMacd señales de compra o venta.
     * @param array ohlcvData Datos históricos ['open', 'high', 'low', 'close', 'volume']. Ordenados cronológicamente.
     * @param array $probability probabilidades calculadas, upperProbabilities y lowerProbabilities
     * @param array $filters Filtros opcionales ['usarATR' => bool].
     */
    public function __construct(
        //ImpulseMACDWS $impulseMacd, // Recibe la instancia calculada
        array $ohlcvData, 
        array $probability, 
        array $filters = []
    )
    {
        if (count($ohlcvData) < $this->atrPeriod + 2) { // +2 para tener vela actual y previa
            throw new \InvalidArgumentException("Se requieren al menos " . ($this->atrPeriod + 2) . " velas para los cálculos.");
        }
        $this->data = $ohlcvData;
        // Se asume que $probability contiene 'upperProbabilities' y 'lowerProbabilities'
        if (!isset($probability['upperProbabilities']) || !isset($probability['lowerProbabilities'])) {
             throw new \InvalidArgumentException("El array de probabilidades debe contener 'upperProbabilities' y 'lowerProbabilities'.");
        }
        //$this->impulseMacd = $impulseMacd;
        $this->upperProbabilities = $probability['upperProbabilities'];
        $this->lowerProbabilities = $probability['lowerProbabilities'];
        $this->usarATR = $filters['usarATR'] ?? false;

        $this->analyzeMarketCondition(); // Evalúa la condición del mercado
    }

    // --- Setters para ajustar parámetros ---
    // Se eliminó setPercentageStep y setNumberOfLevels ya que las propiedades asociadas no se usan.
    public function setShowZeroProbabilities(bool $show): void {$this->showZeroProbabilities = $show;}
    public function setNoLevelZero(bool $nozero): void {$this->noLevelZero = $nozero;}
    public function setShowStats(bool $show): void {$this->printStats = $show;}
    public function setStopLossThreshold(float $threshold): void {$this->stopLossThreshold = $threshold;}
    public function setAtrPeriod(int $period): void {$this->atrPeriod = $period;}
    public function setTpSlRatio(float $ratio): void {$this->tpSlRatio = $ratio;}
    public function setTrendThreshold(float $threshold): void {$this->trendThreshold = $threshold;}
    public function setMinConfidence(float $confidence): void {$this->minConfidence = $confidence;}
    public function setMinSingleLevel(float $level): void {$this->minSingleLevel = $level;}

    /**
     * Analiza la condición del mercado en función de la volatilidad y la fuerza de la tendencia.
     */
    private function analyzeMarketCondition(): void
    {
        $volatility = $this->calculateVolatility(); // Usa la media del ATR
        // $trendStrength = $this->calculateTrendStrength(); // Se eliminó ya que no se usa para marketCondition

        // Ajusta umbrales basados en ATR relativo al precio actual
        $currentPrice = end($this->data)['close'];
        $relativeVolatility = ($currentPrice > 0) ? ($volatility / $currentPrice) * 100 : 0; // Volatilidad como % del precio

        // Umbrales ajustables (ejemplos, requieren optimización)
        $sidewaysThreshold = 0.5; // Ejemplo: ATR medio < 0.5% del precio
        $volatileThreshold = 2.0; // Ejemplo: ATR medio > 2.0% del precio

        // Nota: La lógica de ajuste de minConfidence aquí puede requerir revisión o calibración
        if ($relativeVolatility < $sidewaysThreshold && $relativeVolatility > 0) {
            $this->marketCondition = 'sideways';
            // $this->minConfidence = 20; // Ejemplo original, revisar lógica si se activa
        } elseif ($relativeVolatility > $volatileThreshold) {
            $this->marketCondition = 'volatile';
            $this->minConfidence = 30; // Aumentar requerimiento en volatilidad alta
        } else {
            $this->marketCondition = 'normal';
            $this->minConfidence = 25; // Confianza base para normal
        }

        // Podría usarse $trendStrength aquí también para refinar la condición si se reincorpora
    }

    /**
     * Calcula la volatilidad del mercado utilizando la media del ATR.
     * @return float Media del ATR o 0 si no se puede calcular.
     */
    private function calculateVolatility(): float
    {
        $highs = array_column($this->data, 'high');
        $lows = array_column($this->data, 'low');
        $closes = array_column($this->data, 'close');

        // Asegurarse de que Trader::atr maneja arrays vacíos o con pocos datos
        // Tomar solo los datos necesarios para el cálculo del ATR (últimas $atrPeriod velas + 1 para cálculo)
         $dataForATR = array_slice($this->data, max(0, count($this->data) - ($this->atrPeriod + 1)));
        $highsForATR = array_column($dataForATR, 'high');
        $lowsForATR = array_column($dataForATR, 'low');
        $closesForATR = array_column($dataForATR, 'close');


        // Trader::atr requiere al menos $atrPeriod+1 velas para el cálculo del último ATR.
        // Si hay menos datos, Trader::atr puede devolver warnings o NaNs.
        // Ya validamos el número de velas en el constructor, pero esta validación extra es segura.
        if (count($closesForATR) <= $this->atrPeriod) {
             return 0.0; // No hay suficientes datos para calcular un ATR válido
        }

        $atrValues = Trader::atr($highsForATR, $lowsForATR, $closesForATR, $this->atrPeriod);

        // Tomar solo el último valor de ATR que es el más relevante
        $currentATR = end($atrValues);

        // Filtrar NaNs o nulls si los hubiera y asegurar que es numérico y positivo
        if (!is_numeric($currentATR) || $currentATR <= 0) {
             return 0.0;
        }

        // Aquí se calculaba la media, pero para volatilidad *actual* el último ATR es más representativo.
        // Si se quisiera la volatilidad media histórica, habría que promediar $atrValues.
        // Mantengamos el último ATR como medida de volatilidad actual para analyzeMarketCondition.
        return (float) $currentATR;
    }

    /**
     * Evalúa la fuerza de la tendencia comparando precios de cierre (método simple).
     * Este método no se usa en la versión optimizada para determinar marketCondition,
     * pero se mantiene por si se reincorpora o usa para otros fines (ej. en determineTrend).
     * @return float Fuerza de tendencia (cambio porcentual absoluto) o 0.
     */
    private function calculateTrendStrength(): float
    {
        $closes = array_column($this->data, 'close');
        $numCandles = count($closes);

        if ($numCandles < 50) return 0.0; // Necesita suficientes datos

        $period = intval($numCandles / 3); // O usar un periodo fijo, ej. 50
        if ($period < 1) return 0.0;

        $lastSlice = array_slice($closes, -$period);
        $firstSlice = array_slice($closes, 0, $period);

        $avgLast = array_sum($lastSlice) / count($lastSlice);
        $avgFirst = array_sum($firstSlice) / count($firstSlice);

        if ($avgFirst == 0) return 0.0; // Evitar división por cero

        $trendChange = (($avgLast - $avgFirst) / $avgFirst) * 100;
        return abs($trendChange);
    }

    /**
     * Método principal que realiza el análisis completo y devuelve los resultados en formato JSON.
     * @return string Resultados del análisis en formato JSON.
     */
    public function calculate(): string
    {
        $currentCandle = end($this->data);
        // Mover el puntero interno para obtener la vela previa REAL
        reset($this->data); // Reset pointer
        $tempData = $this->data; // Copia temporal si end()/prev() modifican el original
        end($tempData); // Mover al final
        $previosCandle = prev($tempData); // Obtener el penúltimo elemento

        // Validar que se obtuvo la vela previa
        if ($previosCandle === false || $currentCandle === false) {
             return json_encode(['error' => 'No se pudieron obtener las velas necesarias.'], JSON_PRETTY_PRINT);
        }

        $currentClose = (float) $currentCandle['close'];
        $isPrevGreen = (float) $previosCandle['close'] >= (float) $previosCandle['open'];
        $triggerColor = $isPrevGreen ? 'green' : 'red'; // Determinar el color de la vela PREVIA


        // Calcular el ATR actual usando los datos completos para Trader::atr
        $highs = array_column($this->data, 'high');
        $lows = array_column($this->data, 'low');
        $closes = array_column($this->data, 'close');
        $atrValues = Trader::atr($highs, $lows, $closes, $this->atrPeriod);
        $currentATR = end($atrValues);
        // Asegurar que currentATR es numérico y positivo
        $currentATR = (is_numeric($currentATR) && $currentATR > 0) ? (float) $currentATR : 0.0;

        // Se eliminó la lógica de cálculo de $step ya que los niveles vienen pre-calculados.

        // Determina la tendencia, stop loss, take profit y señal de entrada
        $this->trend = $this->determineTrend($this->upperProbabilities, $this->lowerProbabilities);
        $stopLoss = $this->suggestStopLoss($this->upperProbabilities, $this->lowerProbabilities, $isPrevGreen, $currentATR);
        $takeProfit = $this->suggestTakeProfit($stopLoss, $currentClose);
        $entrySignal = $this->determineEntrySignal($isPrevGreen, $this->upperProbabilities, $this->lowerProbabilities, $currentATR, $currentClose);

        // Añadir precios absolutos a SL y TP después de que se hayan calculado
        $stopLoss = $this->addAbsolutePricesToSL($stopLoss, $currentClose);
        $takeProfit = $this->addAbsolutePricesToTP($takeProfit, $currentClose);

        $mostProbableLevel = $this->getMostProbableLevel($this->upperProbabilities, $this->lowerProbabilities);

        // Construye el resultado final
        $entrySignal['stop_loss'] = $stopLoss['price'];
        $entrySignal['take_profit'] = $takeProfit['price'];
        $result = [
            'current_price' => $currentClose,
            'trigger_candle_color' => $triggerColor, // Color de la vela PREVIA
            'entry_signal' => $entrySignal,
            'upper_levels' => $this->addAbsolutePrices($this->upperProbabilities, $currentClose, true),
            'lower_levels' => $this->addAbsolutePrices($this->lowerProbabilities, $currentClose, false),
            'most_probable_level' => $mostProbableLevel ? $this->enhanceMostProbableLevel($mostProbableLevel, $currentClose) : null,
            'stop_loss' => $stopLoss,
            'take_profit' => $takeProfit,
            'atr' => round($currentATR, 5), // Mayor precisión para ATR
            'trend' => $this->trend,
            // Calcular R:R sólo si hay entrada y SL/TP válidos
            'risk_reward_ratio' => ($entrySignal['should_enter'] && $stopLoss['price'] !== null && $takeProfit['price'] !== null)
                                    ? $this->calculateRiskRewardRatio($stopLoss, $takeProfit, $currentClose)
                                    : null,
            'market_condition' => $this->marketCondition
        ];

        // Agrega estadísticas adicionales si están habilitadas
        if ($this->printStats) {
            $result['stats'] = $this->printStatsPanel();
        }

        return json_encode($result, JSON_PRETTY_PRINT | JSON_INVALID_UTF8_IGNORE);
    }

    /**
     * Encuentra el nivel más probable entre los alcistas y bajistas (excluyendo nivel 0 si noLevelZero es true).
     * @param array|null $upperProbabilities
     * @param array|null $lowerProbabilities
     * @return array|null El nivel más probable o null.
     */
    private function getMostProbableLevel(?array $upperProbabilities, ?array $lowerProbabilities): ?array
    {
         $allLevels = [];
         // Determinar qué nivel excluir si noLevelZero es true
         $levelToExclude = $this->noLevelZero ? 0 : null; // Usar null si no se excluye nada

         if ($upperProbabilities) {
             foreach ($upperProbabilities as $level) {
                 // Excluir el nivel 0 si está configurado
                 if ($levelToExclude === null || $level['level'] != $levelToExclude) {
                      $allLevels[] = ['type' => 'upper', ...$level];
                 }
             }
         }
          if ($lowerProbabilities) {
             foreach ($lowerProbabilities as $level) {
                 // Excluir el nivel 0 si está configurado
                  if ($levelToExclude === null || $level['level'] != $levelToExclude) {
                      $allLevels[] = ['type' => 'lower', ...$level];
                 }
             }
         }

        // Filtrar niveles con probabilidad cero si showZeroProbabilities es false
        if (!$this->showZeroProbabilities) {
            $allLevels = array_filter($allLevels, function ($level) {
                return $level['probability'] > 0;
            });
        }

        if (empty($allLevels)) {
            return null;
        }

        // Ordenar por probabilidad descendente
        usort($allLevels, function ($a, $b) {
            return $b['probability'] <=> $a['probability'];
        });

        return reset($allLevels); // Devuelve el primero (el más probable)
    }

    /**
     * Mejora el nivel más probable añadiendo el precio absoluto.
     * @param array $level El nivel probable.
     * @param float $currentPrice Precio actual.
     * @return array Nivel mejorado.
     */
    private function enhanceMostProbableLevel(array $level, float $currentPrice): array
    {
        $level['absolute_price'] = $this->calculateAbsolutePrice(
            $currentPrice,
            $level['level'],
            $level['type'] === 'upper' // Pasar bool
        );
        return $level;
    }

    /**
     * Determina la señal de entrada (long, short o neutral).
     * @param bool $isPrevGreen Color de la vela PREVIA.
     * @param array|null $upperProbabilities
     * @param array|null $lowerProbabilities
     * @param float $atr ATR actual.
     * @param float $currentPrice Precio actual.
     * @return array Descripción de la señal.
     */
    private function determineEntrySignal(bool $isPrevGreen, ?array $upperProbabilities, ?array $lowerProbabilities, float $atr, float $currentPrice): array
    {
        $signal = 'neutral';
        $confidence = 0.0;
        $reason = '';
        $levelToExclude = $this->noLevelZero ? 0 : null; // Usar null si no se excluye nada

        // Filtrar niveles relevantes (no-cero si noLevelZero, y con alguna probabilidad si showZero=false)
        $relevantUpper = $upperProbabilities ? array_values(array_filter($upperProbabilities, fn($l) => ($levelToExclude === null || $l['level'] != $levelToExclude) && ($this->showZeroProbabilities || $l['probability'] > 0))) : [];
        $relevantLower = $lowerProbabilities ? array_values(array_filter($lowerProbabilities, fn($l) => ($levelToExclude === null || $l['level'] != $levelToExclude) && ($this->showZeroProbabilities || $l['probability'] > 0))) : [];

        $bestUpper = $this->getBestLevel($relevantUpper); // Mejor probabilidad entre relevantes
        $bestLower = $this->getBestLevel($relevantLower); // Mejor probabilidad entre relevantes

        // Filtro de volatilidad mínima (ej: ATR < 0.1% del precio es muy bajo)
        $minAtrThreshold = $currentPrice * 0.001; // 0.1% - Ajustar según sea necesario
        if ($atr > 0 && $atr < $minAtrThreshold) {
            return [
                'action' => 'neutral',
                'confidence' => 0,
                'description' => 'Market without sufficient volatility (very low ATR)',
                'should_enter' => false,
                'reason' => sprintf('ATR %.5f < threshold %.5f (%.2f%%)', $atr, $minAtrThreshold, ($minAtrThreshold/$currentPrice)*100),
                'market_condition' => $this->marketCondition
            ];
        }

        // Lógica de señal basada en el color de la VELA PREVIA
        if ($isPrevGreen) { // Si la vela previa fue verde, buscamos LONG
            if ($bestUpper && $bestUpper['probability'] >= $this->minSingleLevel) {
                $signal =  'long';
                $confidence = $bestUpper['probability']; // Confianza base
                $reason = sprintf("Bullish level %.5f (%.2f%%)", $bestUpper['level'], $bestUpper['probability']);

                // Añadir confianza por niveles adicionales con > X% prob (excluyendo el mejor nivel si es el mismo)
                $additionalLevels = count(array_filter($relevantUpper, fn($l) => $l['probability'] > 10 && ($bestUpper === null || $l['level'] != $bestUpper['level'])));
                if ($additionalLevels > 0) {
                    $confidence += $additionalLevels * 5; // Sumar 5 puntos por cada nivel adicional
                    $reason .= sprintf(" + %d additional levels (>10%%)", $additionalLevels);
                }
                 // Añadir confianza si la tendencia acompaña
                 if ($this->trend['type'] === 'bullish') {
                      // Ponderar por la probabilidad de la tendencia
                      $confidence += $this->trend['probability'] * 0.3; // Ej: 30% de la prob. de trend
                      $reason .= sprintf(" + trend bullish (%.2f%%)", $this->trend['probability']);
                 }
            }
        } else { // Si la vela previa fue roja, buscamos SHORT
            if ($bestLower && $bestLower['probability'] >= $this->minSingleLevel) {
                $signal = 'short';
                $confidence = $bestLower['probability'];
                $reason = sprintf("Bearish level %.5f (%.2f%%)", $bestLower['level'], $bestLower['probability']);

                // Añadir confianza por niveles adicionales con > X% prob (excluyendo el mejor nivel si es el mismo)
                $additionalLevels = count(array_filter($relevantLower, fn($l) => $l['probability'] > 10 && ($bestLower === null || $l['level'] != $bestLower['level'])));
                 if ($additionalLevels > 0) {
                    $confidence += $additionalLevels * 5;
                    $reason .= sprintf(" + %d additional levels (>10%%)", $additionalLevels);
                }
                 if ($this->trend['type'] === 'bearish') {
                      $confidence += $this->trend['probability'] * 0.3;
                      $reason .= sprintf(" + trend bearish (%.2f%%)", $this->trend['probability']);
                 }
            }
        }

        // Normalizar confianza entre 0 y 100
        $confidence = min(max($confidence, 0), 100);

        // Determinar si se debe entrar basado en la confianza mínima ajustada por condición de mercado
        $shouldEnter = $confidence >= $this->minConfidence;

        // Razón si no hay señal
        if ($signal === 'neutral' ) {
            $reason = $this->getNoSignalReason($bestUpper, $bestLower);
        }
        $signalnum = 0;
        if($signal === 'long'){
            $signalnum = 1;
        }elseif($signal === 'short'){
            $signalnum = -1;
        }
        return [
            'action' => $signalnum, // $signal,
            'confidence' => round($confidence, 2),
            'description' => $this->getSignalDescription($signal, $confidence),
            'should_enter' => $shouldEnter,
            'reason' => $reason,
            'market_condition' => $this->marketCondition // Incluir condición que afectó minConfidence
        ];
    }

    /**
     * Encuentra el mejor nivel (mayor probabilidad) entre una lista de niveles relevantes.
     * @param array $levels Lista de niveles ['level'=>..., 'probability'=>...].
     * @return array|null El mejor nivel o null si la lista está vacía.
     */
    private function getBestLevel(array $levels): ?array
    {
        if (empty($levels)) return null;
        // Ordenar por probabilidad descendente
        usort($levels, fn($a, $b) => $b['probability'] <=> $a['probability']);
        return $levels[0]; // Devuelve el de mayor probabilidad
    }

    /**
     * Genera una razón textual cuando no hay señales claras.
     * @param array|null $bestUpper
     * @param array|null $bestLower
     * @return string Razón.
     */
    private function getNoSignalReason(?array $bestUpper, ?array $bestLower): string
    {
        $reasons = [];
        $minProb = $this->minSingleLevel; // Umbral mínimo para considerarlo "significativo" en la razón

        if ($bestUpper && $bestUpper['probability'] >= $minProb) {
            $reasons[] = sprintf("Mejor alcista: %.5f (%.2f%%)", $bestUpper['level'], $bestUpper['probability']);
        } elseif ($bestUpper) {
             $reasons[] = sprintf("Mejor alcista (bajo umbral): %.5f (%.2f%% < %.1f%%)", $bestUpper['level'], $bestUpper['probability'], $minProb);
        } else {
            $reasons[] = "No hay niveles alcistas significativos";
        }

        if ($bestLower && $bestLower['probability'] >= $minProb) {
            $reasons[] = sprintf("Mejor bajista: %.5f (%.2f%%)", $bestLower['level'], $bestLower['probability']);
        } elseif ($bestLower) {
             $reasons[] = sprintf("Mejor bajista (bajo umbral): %.5f (%.2f%% < %.1f%%)", $bestLower['level'], $bestLower['probability'], $minProb);
        } else {
            $reasons[] = "No hay niveles bajistas significativos";
        }
        return implode(" - ", $reasons);
    }

    /**
     * Genera una descripción textual de la señal de entrada.
     * @param string $signal 'long', 'short', 'neutral'.
     * @param float $confidence Confianza calculada (0-100).
     * @return string Descripción.
     */
    private function getSignalDescription(string $signal, float $confidence): string
    {
        // Umbrales de confianza para descripción (ajustables)
        $highConfidenceThreshold = 70.0;
        $mediumConfidenceThreshold = 40.0;

        $descriptions = [
            'long' => [
                'high' => 'Strong buy signal (high confidence)',
                'medium' => 'Buy signal (moderate confidence)',
                'low' => 'Weak buy signal'
            ],
            'short' => [
                'high' => 'Strong sell signal (high confidence)',
                'medium' => 'Sell signal (moderate confidence)',
                'low' => 'Weak sell signal'
            ],
            'neutral' => 'Wait for a clearer signal'
        ];

        if ($signal === 'neutral') return $descriptions['neutral'];

        $level = 'low'; // Nivel por defecto
        if ($confidence >= $highConfidenceThreshold) {
            $level = 'high';
        } elseif ($confidence >= $mediumConfidenceThreshold) {
            $level = 'medium';
        }

        return $descriptions[$signal][$level];
    }

    /**
     * Determina la tendencia del mercado en función de las probabilidades acumuladas.
     * @param array|null $upperProbabilities
     * @param array|null $lowerProbabilities
     * @return array ['type' => string, 'probability' => float]
     */
    private function determineTrend(?array $upperProbabilities, ?array $lowerProbabilities): array
    {
        // Calcular probabilidad media de niveles relevantes (no-cero)
        $relevantUpper = $upperProbabilities ? array_filter($upperProbabilities, fn($l) => $l['level'] != 0) : [];
        $relevantLower = $lowerProbabilities ? array_filter($lowerProbabilities, fn($l) => $l['level'] != 0) : [];

        $avgUpperProb = 0.0;
        if (!empty($relevantUpper)) {
             $avgUpperProb = array_sum(array_column($relevantUpper, 'probability')) / count($relevantUpper);
        }

        $avgLowerProb = 0.0;
        if (!empty($relevantLower)) {
             $avgLowerProb = array_sum(array_column($relevantLower, 'probability')) / count($relevantLower);
        }

        // Determinar tendencia basada en la probabilidad media y el umbral
        if ($avgUpperProb > $this->trendThreshold && $avgUpperProb > $avgLowerProb) {
            return ['type' => 'bullish', 'probability' => round($avgUpperProb, 2)];
        } elseif ($avgLowerProb > $this->trendThreshold && $avgLowerProb > $avgUpperProb) {
            return ['type' => 'bearish', 'probability' => round($avgLowerProb, 2)];
        } else {
            // Podríamos añadir una condición de lateral si ambas son bajas
            return ['type' => 'neutral', 'probability' => 0.0];
        }
    }

    /**
     * Sugiere un nivel de stop loss basado en probabilidades o el ATR.
     * @param array|null $upperProbabilities
     * @param array|null $lowerProbabilities
     * @param bool $isPrevGreen Color de la vela PREVIA.
     * @param float $currentATR ATR actual.
     * @return array Nivel de SL sugerido ['type', 'level', 'probability', 'source', 'price'=>null].
     */
    private function suggestStopLoss(?array $upperProbabilities, ?array $lowerProbabilities, bool $isPrevGreen, float $currentATR): array
    {
        $stopLoss = ['type' => 'none', 'level' => null, 'probability' => null, 'source' => null, 'price' => null];
        $foundLevelSL = false;

        if ($isPrevGreen) { // Buscamos SL por debajo (niveles lower)
             if ($lowerProbabilities) {
                 // Buscar el primer nivel inferior con probabilidad <= umbral
                 // Ordenar por nivel descendente para encontrar el más cercano primero (menos negativo)
                 usort($lowerProbabilities, fn($a, $b) => $b['level'] <=> $a['level']);

                 foreach ($lowerProbabilities as $level) {
                     if ($level['level'] < 0 && $level['probability'] <= $this->stopLossThreshold) {
                         $stopLoss = [
                             'type' => 'lower',
                             'level' => $level['level'],
                             'probability' => $level['probability'],
                             'source' => 'level_prob', // Fuente: Nivel de baja probabilidad
                             'price' => null
                         ];
                         $foundLevelSL = true;
                         break; // Usar el primer nivel (más cercano a 0) que cumpla la condición
                     }
                 }
             }
        } else { // Buscamos SL por encima (niveles upper)
             if ($upperProbabilities) {
                 // Buscar el primer nivel superior con probabilidad <= umbral
                 // Ordenar por nivel ascendente para encontrar el más cercano primero (menos positivo)
                 usort($upperProbabilities, fn($a, $b) => $a['level'] <=> $b['level']);

                 foreach ($upperProbabilities as $level) {
                     if ($level['level'] > 0 && $level['probability'] <= $this->stopLossThreshold) {
                         $stopLoss = [
                             'type' => 'upper',
                             'level' => $level['level'],
                             'probability' => $level['probability'],
                             'source' => 'level_prob',
                             'price' => null
                         ];
                         $foundLevelSL = true;
                         break; // Usar el primer nivel (más cercano a 0) que cumpla la condición
                     }
                 }
            }
        }

        // Fallback a ATR si no se encontró nivel por probabilidad o si ATR es válido
        if (!$foundLevelSL && $currentATR > 0) {
            // Colocar SL a 1x ATR (o un múltiplo configurable)
            $atrMultiplier = 1.0; // Multiplicador del ATR para el SL
            $slLevelBasedOnATR = round($currentATR * $atrMultiplier, 5);

            $stopLoss = [
                'type' => $isPrevGreen ? 'lower' : 'upper',
                'level' => $isPrevGreen ? -$slLevelBasedOnATR : $slLevelBasedOnATR,
                'probability' => null, // No basado en probabilidad directa
                'source' => 'level_ATR',
                'price' => null
            ];
        }
        // No se implementa la lógica para comparar SL por probabilidad vs ATR y elegir el más lejano/seguro
        // ya que no estaba completamente definida y no es estrictamente necesaria para la optimización.

        return $stopLoss;
    }

    /**
     * Sugiere un nivel de take profit basado en el SL y el ratio R:R.
     * @param array $stopLoss El SL calculado.
     * @param float $currentPrice Precio actual.
     * @return array|null Nivel de TP sugerido ['type', 'level', 'price'=>null] o null.
     */
    private function suggestTakeProfit(array $stopLoss, float $currentPrice): ?array
    {
        if ($stopLoss['type'] === 'none' || $stopLoss['level'] === null) {
             return null;
        }

        $distanceToStopLoss = abs($stopLoss['level']);
        if ($distanceToStopLoss == 0) return null; // Evitar TP infinito si SL está en 0

        $takeProfitDistance = $distanceToStopLoss * $this->tpSlRatio;

        // El TP está en la dirección opuesta al SL
        if ($stopLoss['type'] === 'lower') { // SL abajo, TP arriba
            return [
                'type' => 'upper',
                'level' => round($takeProfitDistance, 5),
                'price' => null
            ];
        } else { // SL arriba, TP abajo
            return [
                'type' => 'lower',
                'level' => round(-$takeProfitDistance, 5),
                'price' => null
            ];
        }
    }

    /**
     * Calcula el ratio de riesgo/recompensa.
     * @param array $stopLoss SL con precio absoluto.
     * @param array $takeProfit TP con precio absoluto.
     * @param float $currentPrice Precio actual (punto de entrada teórico).
     * @return float|null Ratio R:R o null si no se puede calcular.
     */
    private function calculateRiskRewardRatio(array $stopLoss, array $takeProfit, float $currentPrice): ?float
    {
        if ($stopLoss['price'] === null || $takeProfit['price'] === null) {
            return null;
        }

        $risk = abs($currentPrice - $stopLoss['price']);
        $reward = abs($takeProfit['price'] - $currentPrice);

        if ($risk == 0) {
            return null; // Evitar división por cero
        }

        return round($reward / $risk, 2);
    }

    /**
     * Añade el precio absoluto al array de stop loss.
     * @param array $stopLoss Array de SL.
     * @param float $currentPrice Precio actual.
     * @return array SL con 'price' añadido.
     */
    private function addAbsolutePricesToSL(array $stopLoss, float $currentPrice): array
    {
        if ($stopLoss['type'] !== 'none' && $stopLoss['level'] !== null) {
            $stopLoss['price'] = $this->calculateAbsolutePrice(
                $currentPrice,
                $stopLoss['level'],
                $stopLoss['type'] === 'upper' // Pasar bool
            );
        }
        return $stopLoss;
    }

    /**
     * Añade el precio absoluto al array de take profit.
     * @param array|null $takeProfit Array de TP o null.
     * @param float $currentPrice Precio actual.
     * @return array|null TP con 'price' añadido o null.
     */
    private function addAbsolutePricesToTP(?array $takeProfit, float $currentPrice): ?array
    {
        if ($takeProfit !== null && $takeProfit['level'] !== null) {
             $takeProfit['price'] = $this->calculateAbsolutePrice(
                $currentPrice,
                $takeProfit['level'],
                $takeProfit['type'] === 'upper' // Pasar bool
            );
        }
        return $takeProfit;
    }

    /**
     * Calcula el precio absoluto a partir de un nivel relativo.
     * @param float $currentPrice Precio de referencia.
     * @param float $relativeLevel Nivel relativo (+ para upper, - para lower).
     * @param bool $isUpper True si el nivel es superior.
     * @return float Precio absoluto.
     */
    private function calculateAbsolutePrice(float $currentPrice, float $relativeLevel, bool $isUpper): float
    {
        // Nivel relativo ya tiene el signo correcto
         $absolutePrice = $currentPrice + $relativeLevel;
        // Redondear a una precisión razonable para precios
        return round($absolutePrice, 5);
    }

    /**
     * Añade precios absolutos a una lista de niveles relativos.
     * @param array|null $levels Lista de niveles ['level'=>..., 'probability'=>...].
     * @param float $currentPrice Precio actual.
     * @param bool $isUpper True si son niveles superiores.
     * @return array Lista de niveles con 'price' añadido.
     */
    private function addAbsolutePrices(?array $levels, float $currentPrice, bool $isUpper): array
    {
        if ($levels === null) {
            return [];
        }
        return array_map(function ($level) use ($currentPrice, $isUpper) {
             $level['price'] = $this->calculateAbsolutePrice($currentPrice, $level['level'], $isUpper);
             return $level;
        }, $levels);
    }

    /**
     * Genera un panel de estadísticas con las probabilidades de los niveles.
     * @return array Estadísticas de probabilidad.
     */
    private function printStatsPanel(): array
    {
        // Usar las probabilidades almacenadas en las propiedades de la clase
        return [
            'total_candles_analyzed' => count($this->data), // O count($this->data) - 1 si la última no se usa en probabilidad
            'prob_first_upper_hit_now' => $this->getFirstLevelProbability($this->upperProbabilities, 1) . '%', // Probabilidad del nivel índice 1
            'prob_first_lower_hit_now' => $this->getFirstLevelProbability($this->lowerProbabilities, 1) . '%', // Probabilidad del nivel índice 1
        ];
    }
     /**
     * Helper para obtener la probabilidad del nivel N desde los resultados ya calculados.
     * Busca el nivel en el índice especificado (asumiendo que el índice 0 es el nivel 0.0).
     *
     * @param array|null $probabilities Array de ['level'=>..., 'probability'=>...] o null.
     * @param int $index Índice del nivel a buscar (ej: 1 para el primer nivel no-cero).
     * @return float La probabilidad (0-100) o 0.0 si no se encuentra o no aplica.
     */
    private function getFirstLevelProbability(?array $probabilities, int $index): float {
        // Validar que $probabilities es un array y el índice existe
        if (!is_array($probabilities) || !isset($probabilities[$index])) {
            return 0.0;
        }
        // Podríamos añadir una comprobación extra para asegurarnos que el nivel en ese índice no es 0,
        // pero generalmente el índice 1 corresponderá al primer nivel calculado distinto de cero.
        // if (isset($probabilities[$index]['level']) && $probabilities[$index]['level'] == 0 && $index > 0) {
        //     // Esto no debería pasar si calculateLevels() funciona bien
        //     return 0.0; // O buscar el siguiente índice
        // }

        // Devolver la probabilidad de ese índice
        return $probabilities[$index]['probability'] ?? 0.0;
    }
}