<?php

use Psr\Log\LoggerInterface;
use Psr\Log\LoggerTrait;
use Psr\Log\LogLevel;

/**
 * Clase BreakoutProbability
 *
 * Esta clase calcula los niveles de ruptura y las probabilidades basadas en el historial de precios,
 * y genera alertas cuando se alcanzan los niveles de ruptura.
 */
class BreakoutProbability
{
    private float $percentageStep;
    private int $numberOfLines;
    private string $upTrendColor;
    private string $downTrendColor;
    private bool $fillArea;
    private string $fillColor;
    private bool $disableZeroProbability;
    private bool $showStatisticPanel;
    private array $alertOptions;
    private array $priceHistory = [];
    private array $probabilities = [];
    private ?LoggerInterface $logger;
    private array $total = []; // Simula la matriz 'total' de Pine Script
    private array $vals = [];  // Simula la matriz 'vals' de Pine Script
    private int $barIndex = 0; // Rastrea el índice de la barra
    private ?NotificacionesAction $notificacionesAction = null;

    /**
     * Constructor de la clase BreakoutProbability.
     *
     * @param float $percentageStep Espaciado entre niveles como porcentaje del precio actual.
     * @param int $numberOfLines Número de líneas horizontales a dibujar.
     * @param string $upTrendColor Color de las líneas para movimientos alcistas.
     * @param string $downTrendColor Color de las líneas para movimientos bajistas.
     * @param bool $fillArea Indica si se debe rellenar el área entre las líneas.
     * @param string $fillColor Color del relleno del área entre las líneas.
     * @param bool $disableZeroProbability Indica si se deben ocultar los niveles con 0% de probabilidad.
     * @param bool $showStatisticPanel Indica si se debe mostrar el panel de estadísticas.
     * @param array $alertOptions Opciones de configuración de las alertas.
     * @param LoggerInterface|null $logger (Opcional) Logger para registrar eventos y errores.
     */
    public function __construct(
        float $percentageStep,
        int $numberOfLines,
        string $upTrendColor,
        string $downTrendColor,
        bool $fillArea,
        string $fillColor,
        bool $disableZeroProbability,
        bool $showStatisticPanel,
        array $alertOptions,
        ?LoggerInterface $logger = null
    ) {
        if ($numberOfLines < 1 || $numberOfLines > 5) {
            throw new InvalidArgumentException("NumberOfLines debe estar entre 1 y 5.");
        }

        $this->percentageStep = $percentageStep;
        $this->numberOfLines = $numberOfLines;
        $this->upTrendColor = $upTrendColor;
        $this->downTrendColor = $downTrendColor;
        $this->fillArea = $fillArea;
        $this->fillColor = $fillColor;
        $this->disableZeroProbability = $disableZeroProbability;
        $this->showStatisticPanel = $showStatisticPanel;
        $this->alertOptions = $alertOptions;
        $this->logger = $logger;

        $this->total = array_fill(0, 7, array_fill(0, 4, 0));
        $this->vals = array_fill(0, 5, array_fill(0, 4, 0.0));
    }

    /**
     * Establece la instancia de NotificacionesAction para enviar alertas.
     *
     * @param NotificacionesAction $notificacionesAction Instancia de NotificacionesAction.
     */
    public function setNotificacionesAction(NotificacionesAction $notificacionesAction): void
    {
        $this->notificacionesAction = $notificacionesAction;
    }

    /**
     * Agrega un nuevo registro de precio.
     *
     * @param float $open Precio de apertura de la vela.
     * @param float $high Precio máximo de la vela.
     * @param float $low Precio mínimo de la vela.
     * @param float $close Precio de cierre de la vela.
     */
    public function addPrice(float $open, float $high, float $low, float $close): void
    {
        $this->priceHistory[] = [
            'barIndex' => $this->barIndex,
            'open' => $open,
            'high' => $high,
            'low' => $low,
            'close' => $close,
            'isBullish' => $close > $open,
        ];
        $this->barIndex++;

        if (count($this->priceHistory) > 1500) {
            array_shift($this->priceHistory);
        }
    }

    /**
     * Calcula los niveles de ruptura y las probabilidades.
     *
     * @return array Array con los niveles de ruptura y las probabilidades.
     */
    public function calculateBreakoutLevels(): array
    {
        if (empty($this->priceHistory)) {
            return [];
        }
        $currentCandle = end($this->priceHistory); // Última vela
        $previosCandle = prev($this->priceHistory); // Anterior

        $closePrice = $currentCandle['close'];
        $highPrice = $currentCandle['high'];
        $lowPrice = $currentCandle['low'];
        $green = $previosCandle['close'] >= $previosCandle['open'];
        $red = $previosCandle['close'] <= $previosCandle['open'];
        $latestPrice = end($this->priceHistory)['close'];
        $levels = [];
        $this->probabilities = [];
        for ($i = 1; $i <= $this->numberOfLines; $i++) {
            $upLevel = $latestPrice * (1 + $this->percentageStep * $i / 100);
            $downLevel = $latestPrice * (1 - $this->percentageStep * $i / 100);

            $levels['up'][$i] = $upLevel;
            $levels['down'][$i] = $downLevel;

            $this->probabilities['up'][$i] = $this->calculateProbability($upLevel, true);
            $this->probabilities['down'][$i] = $this->calculateProbability($downLevel, false);
        }
        return $levels;
    }

    /**
     * Calcula la probabilidad de que el precio alcance un nivel dado.
     *
     * @param float $level Nivel de precio.
     * @param bool $isUp Indica si el nivel está por encima (true) o por debajo (false).
     * @return float Probabilidad de alcanzar el nivel (0-100).
     */
    private function calculateProbability(float $level, bool $isUp): float
    {
        $ghh = 0;
        $gll = 0;
        $gtot = 0;
        $rhh = 0;
        $rll = 0;
        $rtot = 0;
        foreach ($this->priceHistory as $index => $record) {
            $previousRecord = $this->priceHistory[$index - 1] ?? null;
            if ($previousRecord) {
                if ($isUp && $previousRecord['close'] > $previousRecord['open']) {
                    $gtot++;
                    if ($record['high'] >= $level) {
                        $ghh++;
                    } elseif ($record['low'] <= $level) {
                        $gll++;
                    }
                } elseif (!$isUp && $previousRecord['close'] < $previousRecord['open']) {
                    $rtot++;
                    if ($record['high'] >= $level) {
                        $rhh++;
                    } elseif ($record['low'] <= $level) {
                        $rll++;
                    }
                }
            }
        }
        $lev = $isUp?'up':'dw';
        $cal = ($this / $attempts) * 100;
print("level:{$level} level:{$lev} hits:{$hits} attemps:{$attempts} %:{$cal}" .chr(10));
        return ($attempts > 0) ? ($hits / $attempts) * 100 : 0;
    }

    /**
     * Genera alertas basadas en los niveles de ruptura alcanzados.
     *
     * @param array $levels Niveles de ruptura calculados.
     */
    public function generateAlerts(array $levels): void
    {
        if (empty($levels) || empty($this->alertOptions)) {
            return;
        }

        $latestPrice = end($this->priceHistory)['close'];
        $latestHigh = end($this->priceHistory)['high'];
        $latestLow = end($this->priceHistory)['low'];
        $symbol = $this->alertOptions['symbol'] ?? 'N/A';

        foreach ($levels['up'] as $i => $level) {
            if ($latestHigh >= $level) {
                $alertMessage = [
                    'symbol' => $symbol,
                    'highPrice' => $level,
                    'lowPrice' => null,
                    'bias' => 'alcista',
                    'probability' => $this->probabilities['up'][$i] ?? 0,
                ];
                $this->logAlert($alertMessage);
            }
        }

        foreach ($levels['down'] as $i => $level) {
            if ($latestLow <= $level) {
                $alertMessage = [
                    'symbol' => $symbol,
                    'highPrice' => null,
                    'lowPrice' => $level,
                    'bias' => 'bajista',
                    'probability' => $this->probabilities['down'][$i] ?? 0,
                ];
                $this->logAlert($alertMessage);
            }
        }
    }

    /**
     * Registra una alerta utilizando el logger o envía una notificación.
     *
     * @param array $alertMessage Mensaje de la alerta.
     */
    private function logAlert(array $alertMessage): void
    {
        if ($this->notificacionesAction) {
            $token = $this->notificacionesAction->getUsuariosResource()->getTokenBySymbol($alertMessage['symbol']);
            $device = $this->notificacionesAction->getUsuariosResource()->getDeviceBySymbol($alertMessage['symbol']);
            $body = "Ruptura " . $alertMessage['bias'] . " en " . $alertMessage['symbol'] . " a " . ($alertMessage['highPrice'] ?? $alertMessage['lowPrice']) . " con probabilidad de " . $alertMessage['probability'] . "%";
            $cliaction = $body;

            if ($token && $device) {
                $this->notificacionesAction->sendPushNotification($token, $device, $body, $cliaction);
            }

            $telegramToken = $this->notificacionesAction->getUsuariosResource()->getTelegramTokenBySymbol($alertMessage['symbol']);
            if ($telegramToken) {
                $data = [
                    'chat_id' => urlencode($telegramToken),
                    'text' => $body,
                    'disable_web_page_preview' => true,
                    'reply_to_message_id' => false,
                    'reply_markup' => '',
                    'parse_mode' => "HTML",
                ];
                $this->notificacionesAction->createTelegram(json_encode($data));
            }
        } elseif ($this->logger) {
            $this->logger->alert('Alerta de Ruptura', $alertMessage);
        } else {
            echo json_encode($alertMessage) . PHP_EOL;
        }
    }

    /**
     * Calcula y devuelve las estadísticas para el panel de estadísticas.
     *
     * @return array|string Array con las estadísticas o mensaje si el panel está desactivado.
     */
    public function getStatistics(): array|string
    {
        if (!$this->showStatisticPanel) {
            return "Panel de estadísticas desactivado.";
        }

        $firstLevelUp = 0;
        $firstLevelDown = 0;
        $attemptsUp = 0;
        $attemptsDown = 0;

        foreach ($this->priceHistory as $index => $record) {
            $previousRecord = $this->priceHistory[$index - 1] ?? null;
            $latestPrice = end($this->priceHistory)['close'];
            if (!empty($this->probabilities) && $previousRecord) {
                $upLevel = $latestPrice * (1 + $this->percentageStep / 100);
                $downLevel = $latestPrice * (1 - $this->percentageStep / 100);
                if ($record['high'] >= $upLevel) {
                    $firstLevelUp++;
                }
                if ($record['low'] <= $downLevel) {
                    $firstLevelDown++;
                }
                if ($previousRecord['close'] > $previousRecord['open']) {
                    $attemptsUp++;
                } else {
                    $attemptsDown++;
                }
            }
        }
        $winLossRatioUp = ($attemptsUp > 0) ? $firstLevelUp / $attemptsUp : 0;
        $winLossRatioDown = ($attemptsDown > 0) ? $firstLevelDown / $attemptsDown : 0;

        $statistics = [
            'firstLevelUp' => $firstLevelUp,
            'firstLevelDown' => $firstLevelDown,
            'winLossRatioUp' => $winLossRatioUp,
            'winLossRatioDown' => $winLossRatioDown,
        ];

        return $statistics;
    }

    /**
     * Devuelve los datos del indicador para visualización (JSON).
     *
     * @return string Datos del indicador en formato JSON.
     */
    public function getIndicatorData(): string
    {
        $levels = $this->calculateBreakoutLevels();
        $statistics = $this->getStatistics();

        $data = [
            'levels' => $levels,
            'probabilities' => $this->probabilities,
            'upTrendColor' => $this->upTrendColor,
            'downTrendColor' => $this->downTrendColor,
            'fillArea' => $this->fillArea,
            'fillColor' => $this->fillColor,
            'disableZeroProbability' => $this->disableZeroProbability,
            'showStatisticPanel' => $this->showStatisticPanel,
            'statistics' => $statistics,
        ];

        return json_encode($data);
    }
}