<?php
namespace App;

use App\ImpulseMACD;
use App\RSI;
use App\PivotPoints;
use App\PivotCalculator;
use App\SmartMoneyVolume;
use App\BreakoutProbabilitySR;
use App\Trader;

class Analysis {
    private $impulseMacd;
    private $rsi;
    private $pivotPoints;
    private $pivotCalculator;
    private $smartMoneyVolume;
    private $breakoutProbabilitysr;
    private $closes;
    private $highs;
    private $lows;
    private $volumes;
    private $trader;
    private $atr;
    private $atrPeriod = 14;
    private $atrMultiplier = 2;
    private $confidenceThresholdMultiplier = 0.5;

    /**
     * Constructor de la clase Analysis.
     *
     * @param array $closes Array de precios de cierre.
     * @param array $highs Array de precios máximos.
     * @param array $lows Array de precios mínimos.
     * @param array $volumes Array de volúmenes.
     */
    public function __construct(array $closes, array $highs, array $lows, array $volumes) {
        // Asegurarse de que los arrays tengan datos
        if (empty($closes) || empty($highs) || empty($lows) || empty($volumes)) {
            throw new \InvalidArgumentException("Todos los arrays de datos (precios, máximos, mínimos, volúmenes) deben contener datos.");
        }
        $this->prices = $closes;
        $this->highs = $highs;
        $this->lows = $lows;
        $this->closes = $closes;
        $this->volumes = $volumes;
        // Inicializar las clases de los indicadores
        $this->impulseMacd = new ImpulseMACD();
        $this->rsi = new RSI($closes);
        $this->pivotCalculator = new PivotCalculator($highs, $lows, $closes, $this->atrPeriod);
         $this->smartMoneyVolume = new SmartMoneyVolume($this->getDataForImpulseMacd());
        $this->breakoutProbabilitysr = new BreakoutProbabilitySR($highs, $lows, $closes, $volumes);
    }

    /**
     * Analiza los datos de mercado y genera una señal de trading.
     *
     * @return array Un array asociativo con la señal de trading ('LONG', 'SHORT', o 'WAIT'),
     * el ratio Take Profit/Stop Loss ('tpSlRatio'), y los niveles de Stop Loss y Take Profit.
     */
    public function analyze(): array {
        // Calcula atr
        $this->atr = Trader::ATR($this->highs, $this->lows, $this->closes, $this->atrPeriod);
        $atrValue = end($this->atr);
        $macdStrategy = $this->impulseMacd->generateStrategy($this->getDataForImpulseMacd());
        $rsiOversoldExit = $this->rsi->isOversoldExit();
        $rsiOverboughtExit = $this->rsi->isOverboughtExit();
        // Calcula Pivots Clasicos
        $pivotsData = $this->pivotCalculator->calculateClassicPivots();
        // Usar los pivotes clásicos como high y low
        $newHighPivot = $pivotsData['pivots'];
        $newLowPivot = $pivotsData['pivots'];
        $pivotRe = end($pivotsData['resistances']);
        $pivotSu = end($pivotsData['supports']);        
        // Determinar la tendencia para la opción pivotes clásicos
        $this->pivotPoints = new PivotPoints($newHighPivot, $newLowPivot, $atrValue);
        $bullishTrend = $this->pivotPoints->isTrendBullish(2);
        $bearishTrend = $this->pivotPoints->isTrendBearish(2);
        if($bullishTrend){
            $pivotTrend = 'alcista';
        }elseif($bearishTrend){
            $pivotTrend = 'bajista';
        }else{
            $pivotTrend = 'divergente';
        }

        // Obtener información de SmartMoneyVolume
        $lastSpikeDirection = $this->smartMoneyVolume->getLastSpikeDirection();
        $hasSpike = $this->smartMoneyVolume->hasSpike();
        $isAccumulation = $this->smartMoneyVolume->isAccumulation();
        $isDistribution = $this->smartMoneyVolume->isDistribution();

        $breakoutAnalysis = $this->breakoutProbabilitysr->getFullAnalysis();
        $breakoutRe = $breakoutAnalysis['key_levels']['resistances'][0];
        $breakoutSu = $breakoutAnalysis['key_levels']['supports'][0];
        // Verificar si hay datos suficientes para ATR
        if ($atrValue === false) {
            $atrValue = 0;
        }

        // Determinar la señal basada en la ponderación de indicadores
        $signal = $this->determineSignal(
            $macdStrategy,
            $rsiOversoldExit,
            $rsiOverboughtExit,
            $pivotTrend,
            $breakoutAnalysis,
            $atrValue,
            $lastSpikeDirection,
            $isAccumulation,
            $isDistribution,
            $hasSpike
        );

        // Ajustar el ratio TP/SL
        $tpSlRatio = $this->adjustTpSlRatio($signal['score']);

        if($tpSlRatio === 0.0){
            $signal['action'] = 'WAIT';
        }

        // Calcular SL y TP usando ATR
        $slTp = $this->calculateSlTp($signal['action'], $atrValue, $tpSlRatio, $macdStrategy);

        // Combinar la señal y los niveles de SL/TP
        $finalSignal = [
            'score' => $signal['score'],
            'action' => $signal['action'],
            'tpSlRatio' => $tpSlRatio,
            'actPrice' => end($this->closes),
            'stopLoss' => $slTp['stopLoss'],
            'takeProfit' => $slTp['takeProfit'],
            'pivotRe' => $pivotRe,
            'pivotSu' => $pivotSu,
            'pivotTrend' => $pivotTrend,
            'breakoutRe' => $breakoutRe,
            'breakoutSu' => $breakoutSu
        ];

        return $finalSignal;
    }

    /**
     * Prepara los datos en el formato esperado por ImpulseMACD.
     *
     * @return array Un array asociativo con las claves 'High', 'Low' y 'Close'.
     */
    private function getDataForImpulseMacd(): array {
        return [
            'high' => $this->highs,
            'low' => $this->lows,
            'close' => $this->closes,
            'volume' => $this->volumes
        ];
    }

    /**
     * Determina la señal de trading basada en una ponderación de los indicadores.
     *
     * @param array $macdStrategy La estrategia MACD.
     * @param bool $rsiOversoldExit El resultado del RSI sobreventa.
     * @param bool $rsiOverboughtExit El resultado del RSI sobrecompra.
     * @param string $pivotTrend La tendencia de los puntos pivot ('alcista' o 'bajista').
     * @param bool $volumeSpike Las anomalías de Smart Price Volume.
     * @param array $breakoutAnalysis El análisis de probabilidad de ruptura.
     * @param float $atrValue El valor actual del ATR.
     * @param ?string $lastSpikeDirection La dirección del último spike de volumen.
     * @param bool $isAccumulation Indica si hay acumulación.
     * @param bool $isDistribution Indica si hay distribución.
     * @param bool $hasSpike Indica si hay un spike de volumen detectado.
     * @return array Un array asociativo con la 'action' ('LONG', 'SHORT', o 'WAIT') y el 'score'.
     */
    private function determineSignal(
        array $macdStrategy,
        bool $rsiOversoldExit,
        bool $rsiOverboughtExit,
        string $pivotTrend,
        array $breakoutAnalysis,
        float $atrValue,
        ?string $lastSpikeDirection,
        bool $isAccumulation,
        bool $isDistribution,
        bool $hasSpike
    ): array {
        $score = 0;
        $longScore = 0;
        $shortScore = 0;
        $action = 'WAIT';
        $confidenceThreshold = $this->confidenceThresholdMultiplier * $atrValue;

        // Ponderación del MACD
        if ($macdStrategy && isset($macdStrategy[1]['signalType'])) {
            if ($macdStrategy[1]['signalType'] == 'BUY') {
                $longScore += 40;
            } elseif ($macdStrategy[1]['signalType'] == 'SELL') {
                $shortScore += 40;
            }
        }

        // Ponderación del RSI
        if ($rsiOversoldExit) {
            $longScore += 15;
        } elseif ($rsiOverboughtExit) {
            $shortScore += 15;
        }

        // Ponderación de los Puntos Pivot
        if ($pivotTrend == 'alcista') {
            $longScore += 10;
        } elseif ($pivotTrend == 'bajista') {
            $shortScore += 10;
        }

        // Ponderación de la Probabilidad de Ruptura
        if ($breakoutAnalysis) {
            if ($breakoutAnalysis['probability_matrix']['breakout_probabilities']['bullish'] >=70) {
                $longScore += 5;
            } elseif ($breakoutAnalysis['probability_matrix']['breakout_probabilities']['bearish'] >=70) {
                $shortScore += 5;
            }
        }

        // Ponderación de SmartMoneyVolume (spikes)
        if ($lastSpikeDirection === 'up') {
            $longScore += 20;
        } elseif ($lastSpikeDirection === 'down') {
            $shortScore += 20;
        } else {
            // No hay spike → mercado tranquilo
            $longScore += 5;
            $shortScore += 5;
        }

        // Ponderación de SmartMoneyVolume (acumulación/distribución)
        if ($isAccumulation) {
            $longScore += 6;
        } elseif ($isDistribution) {
            $shortScore += 6;
        }

        // Ponderación de SmartMoneyVolume (hasSpike)
        if ($hasSpike) {
            // Si hay spike pero no se detecta dirección, ponderar con menor peso.
            if ($lastSpikeDirection === null) {
                $longScore += 3;
                $shortScore += 3;
            }
        }

        // Umbral de confianza dinámico
        $confidenceThreshold = $this->confidenceThresholdMultiplier * $atrValue;

        if ($rsiOversoldExit || $rsiOverboughtExit) {
            $confidenceThreshold *= 1.2;
        }

        if ($pivotTrend == 'alcista' || $pivotTrend == 'bajista') {
            $confidenceThreshold *= 1.1;
        }

        // Determinar la acción final y la puntuación
        if (abs($longScore - $shortScore) >= $confidenceThreshold) {
            if ($longScore > $shortScore) {
                $action = 'LONG';
                $score = $longScore;
            } elseif ($shortScore > $longScore) {
                $action = 'SHORT';
                $score = $shortScore;
            }
        } else {
            $action = 'WAIT';
            $score = 0;
        }

        return ['action' => $action, 'score' => $score];
    }

    /**
     * Ajusta el ratio Take Profit/Stop Loss basado en la puntuación de la señal.
     *
     * @param int $score La puntuación de la señal.
     * @return float El ratio Take Profit/Stop Loss.
     */
    private function adjustTpSlRatio(int $score): float {
        if ($score >= 85) {
            return 3.0;
        } elseif ($score >= 75) {
            return 2.0;
        } elseif ($score >= 70) {
            return 1.5;
        } else {
            return 0.0; // Indica que no hay señal
        }
    }

    /**
     * Calcula los niveles de Stop Loss y Take Profit basados en la acción y el ATR,
     * y ajusta los valores de ImpulseMACD.
     *
     * @param string $action La acción de la señal ('LONG' o 'SHORT').
     * @param float $atrValue El valor del Average True Range.
     * @param float $tpSlRatio El ratio Take Profit/Stop Loss.
     * @param array $macdStrategy Los valores de Stop Loss y Take Profit de ImpulseMACD.
     * @return array Un array asociativo con los niveles de 'stopLoss' y 'takeProfit'.
     */
    private function calculateSlTp(string $action, float $atrValue, float $tpSlRatio, array $macdStrategy): array {
        $currentPrice = end($this->closes);
        $stopLoss = 0;
        $takeProfit = 0;

        if ($action == 'LONG') {
            $stopLoss = $currentPrice - ($this->atrMultiplier * $atrValue);
            $takeProfit = $currentPrice + ($tpSlRatio * $this->atrMultiplier * $atrValue);
        } elseif ($action == 'SHORT') {
            $stopLoss = $currentPrice + ($this->atrMultiplier * $atrValue);
            $takeProfit = $currentPrice - ($tpSlRatio * $this->atrMultiplier * $atrValue);
        }

        // Si ImpulseMACD proporciona SL/TP, compararlos y elegir el más conservador.
        if (isset($macdStrategy[1]['stopLoss']) && $macdStrategy[1]['stopLoss'] != 0) {
            $stopLoss = ($action == 'LONG') ? max($stopLoss, $macdStrategy[1]['stopLoss']) : min($stopLoss, $macdStrategy[1]['stopLoss']);
        }
        if (isset($macdStrategy[1]['takeProfit']) && $macdStrategy[1]['takeProfit'] != 0) {
            $takeProfit = ($action == 'LONG') ? min($takeProfit, $macdStrategy[1]['takeProfit']) : max($takeProfit, $macdStrategy[1]['takeProfit']);
        }

        return [
            'stopLoss' => $stopLoss,
            'takeProfit' => $takeProfit,
        ];
    }

    /**
     * Obtiene los datos de precios, altos, bajos y volúmenes.
     *
     * @return array
     */
    public function getPrices(): array {
        return $this->prices;
    }

    public function getHighs(): array {
        return $this->highs;
    }

    public function getLows(): array {
        return $this->lows;
    }

    public function getCloses(): array {
        return $this->closes;
    }

    public function getVolumes(): array {
        return $this->volumes;
    }
}
/*
Mejorar la Precisión de las Señales en la Clase AnalysisAjustar los umbrales de puntuación: 
Podrías modificar el método adjustTpSlRatio para que solo se generen señales cuando el score sea igual o superior a un cierto valor. 
Por ejemplo:private function adjustTpSlRatio(int $score): float {
    if ($score >= 80) {
        return 3.0;
    } elseif ($score >= 70) {
        return 2.0;
    } elseif ($score >= 65) { // Aumentamos el umbral mínimo
        return 1.5;
    } else {
        return 0.0; // Indica que no hay señal
    }
}
En este caso, si el score es menor de 65, el método devuelve 0.0, lo que podría interpretarse como "no hay señal". 
Luego, en el método analyze, podrías verificar este valor y decidir si se realiza la operación o no.
Ponderar los indicadores de forma diferente: Si consideras que algunos indicadores son más fiables que otros, puedes ajustar las puntuaciones asignadas en el método determineSignal. 
Por ejemplo, si el MACD y el volumen son tus indicadores principales, podrías darles una mayor ponderación que al RSI o a los puntos pivote.
Añadir más indicadores o condiciones: Podrías incorporar otros indicadores técnicos o condiciones fundamentales al análisis para aumentar la robustez de las señales. 
Cuantos más factores respalden una señal, mayor será la probabilidad de que sea exitosa.
Utilizar un umbral de confianza dinámico: En lugar de utilizar un umbral de confianza fijo, podrías hacerlo variar en función de las condiciones del mercado. 
Por ejemplo, podrías utilizar un umbral más alto en mercados volátiles y un umbral más bajo en mercados tranquilos.
Evaluar el rendimiento de la estrategia: Es fundamental realizar un backtesting exhaustivo de la estrategia con diferentes umbrales de puntuación para determinar cuáles ofrecen el mejor equilibrio entre la cantidad de señales generadas y su precisión.

Para mejorar el uso de un umbral de confianza dinámico, en lugar de un valor fijo, puedes adaptarlo a las condiciones del mercado. Aquí te presento algunas ideas sobre cómo implementar esto:Volatilidad del mercado:Utiliza un indicador de volatilidad, como el Average True Range (ATR), para ajustar el umbral. En mercados más volátiles, aumenta el umbral para evitar señales falsas. En mercados menos volátiles, reduce el umbral para capturar más oportunidades.Ejemplo:private function determineSignal(array $macdStrategy, bool $rsiOversoldExit, bool $rsiOverboughtExit, string $pivotTrend, bool $volumeSpike, array $breakoutAnalysis, float $atrValue): array {
    $score = 0;
    $longScore = 0;
    $shortScore = 0;
    $action = 'WAIT';
    $confidenceThreshold = $this->confidenceThresholdMultiplier * $atrValue; // Umbral dinámico
    // ... (resto del código)
}
Índice de fuerza relativa (RSI):Ajusta el umbral en función de la posición del RSI. Si el RSI se encuentra en niveles extremos (sobrecompra o sobreventa), puedes aumentar el umbral, ya que estas condiciones pueden generar señales falsas.Ejemplo:private function determineSignal(array $macdStrategy, bool $rsiOversoldExit, bool $rsiOverboughtExit, string $pivotTrend, bool $volumeSpike, array $breakoutAnalysis, float $atrValue): array
{
        $score = 0;
        $longScore = 0;
        $shortScore = 0;
        $action = 'WAIT';
        $confidenceThreshold = $this->confidenceThresholdMultiplier;

        if ($rsiOversoldExit) {
            $confidenceThreshold *= 1.2; // Aumenta el umbral
        } elseif ($rsiOverboughtExit) {
            $confidenceThreshold *= 1.2; // Aumenta el umbral
        }
// ...
}
Tendencia del mercado:Utiliza un indicador de tendencia, como una media móvil, para ajustar el umbral. Si el mercado se encuentra en una tendencia fuerte, puedes aumentar el umbral para filtrar señales que van en contra de la tendencia.Ejemplo: private function determineSignal(array $macdStrategy, bool $rsiOversoldExit, bool $rsiOverboughtExit, string $pivotTrend, bool $volumeSpike, array $breakoutAnalysis, float $atrValue): array
{
        $score = 0;
        $longScore = 0;
        $shortScore = 0;
        $action = 'WAIT';
        $confidenceThreshold = $this->confidenceThresholdMultiplier;
        if ($pivotTrend == 'alcista') {
            $confidenceThreshold *= 1.1;
        } elseif ($pivotTrend == 'bajista') {
            $confidenceThreshold *= 1.1;
        }
// ...
}
Combinación de factores:Puedes combinar varios de los factores anteriores para crear un umbral de confianza aún más dinámico y preciso.Ejemplo:private function determineSignal(array $macdStrategy, bool $rsiOversoldExit, bool $rsiOverboughtExit, string $pivotTrend, bool $volumeSpike, array $breakoutAnalysis, float $atrValue): array {
    $score = 0;
    $longScore = 0;
    $shortScore = 0;
    $action = 'WAIT';
    $confidenceThreshold = $this->confidenceThresholdMultiplier * $atrValue;

     if ($rsiOversoldExit || $rsiOverboughtExit) {
            $confidenceThreshold *= 1.2;
     }

     if ($pivotTrend == 'alcista' ||  $pivotTrend == 'bajista') {
         $confidenceThreshold *= 1.1;
     }
    // ...
}
    */