<?php

declare(strict_types=1); // Habilitar modo estricto para mejor control de tipos

namespace App;

// Usar declaraciones 'use' para todas las dependencias
use App\ResponseApi;
use App\TrendCalculator;
use App\Analysis; // Asumiendo que existe y funciona como se espera
use App\ImpulseMACDWS;
use App\ImpulseMacdStrategy;
use App\SmartMoneyVolume;
use App\SmartMoneyFusionStrategy;
use App\Exceptions\ResponseApiException; // Ejemplo de excepción personalizada
use App\Exceptions\CalculationException; // Ejemplo de excepción personalizada
use DateTime; // Usar la clase DateTime global

class AlgoritApi
{
    // --- Constantes para Parámetros (Centralizadas) ---
    private const KLINE_LIMIT = 300;
    // Parámetros para ImpulseMACD + EMA100
    private const IMPULSE_MACD_MA_LENGTH = 34;
    private const IMPULSE_MACD_SIGNAL_LENGTH = 9;
    private const IMPULSE_MACD_THRESHOLD_PERIOD = 2;
    private const IMPULSE_MACD_CONFIRMATION_PERIOD = 2;
    private const STRATEGY_ATR_PERIOD = 14;
    private const STRATEGY_ATR_MULTIPLIER_SL = 2.0; // Usado por ImpulseMacdStrategy
    private const STRATEGY_RR_RATIO = 1.5; // bajado de 2.5;         // Usado por ImpulseMacdStrategy
    private const STRATEGY_CONFIRMATION_PERIODS = 2;
    private const MIN_POTP_THRESHOLD = 0.4; // Umbral mínimo para %TP
    // Parámetros para BreakOutProbability
    private const PERCENTAGE_STEP = 1;
    private const SHOW_ZERO_PROBABILITY = true; // Indica si se muestran niveles con probabilidad cero
    private const NUMBER_OF_LEVELS = 4; 
    // Parámetros para BreakOutProbability Strategy
    private const NO_LEVEL_ZERO = false;  // Indica si se excluye el nivel 0 para calcular probabilidad
    private const SHOW_ZERO_PROBABILITIES = true;
    private const SHOW_STATS = true;
    private const STOP_LOSS_THRESHOLD = 10; // Umbral de probabilidad para stop loss
    private const TP_SL_RATIO = 1.75; // Ratio Take Profit / Stop Loss
    private const TREND_THRESHOLD = 50; // Umbral para determinar la tendencia

    // --- Propiedades ---
    private array $algResp = []; // Inicializar vacío, se llenará en los métodos
    private int $CV = 0;
    private int $ACT = 0; // Asumiendo que ACT también es un entero (0, 1, 2)

    // Propiedades de configuración (inyectadas)
    private readonly array $config;
    private readonly float $configFtp; // Factor TP para ATR (ej: desde config['atrtp'])
    private readonly float $configFsl; // Factor SL para ATR (ej: desde config['atrsl'])
    private readonly int $configRsiThreshold; // Umbral RSI (ej: desde config['rsi'])
    private readonly string $interval;
    private readonly bool $proMode; // $PRO renombrado a $proMode para claridad

    // Dependencias (Inyectadas)
    private readonly ResponseApi $apiResponse;
    private readonly TrendCalculator $trendCalculator;
    //private readonly DailyRotatingLogger $logger;
    // Nota: RsiCalculator y AtrCalculator no se usan en el código proporcionado, se omiten aquí.
    // Si se necesitan, deben inyectarse también.

    /**
     * Constructor con Inyección de Dependencias.
     *
     * @param array $config La configuración de la aplicación/algoritmo.
     * @param ResponseApi $apiResponse Cliente para interactuar con la API del exchange.
     * @param TrendCalculator $trendCalculator Calculador de tendencias.
     */
    public function __construct(
        //array $config,
        //ResponseApi $apiResponse,
        //TrendCalculator $trendCalculator
        //DailyRotatingLogger $dailyRotatingLogger
    ) {
        require_once __DIR__ . '/Settings.php';
        $params = new Settings();
        $config = $params->settings['config']; 
        $this->config = $config;
        // si se reciben como parametro $this->apiResponse = $apiResponse;
        // si se reciben como parametro $this->logger = $dailyRotatingLogger;
        // si se reciben como parametro $this->trendCalculator = $trendCalculator;

        $this->apiResponse = new ResponseApi();
        $this->trendCalculator = new TrendCalculator();
        //$this->logger =  new DailyRotatingLogger();
        // Validar y asignar configuración específica requerida por la clase
        $this->configFtp = (float) ($config['atrtp'] ?? 2.0); // Valor por defecto si no existe
        $this->configFsl = (float) ($config['atrsl'] ?? 1.5); // Valor por defecto si no existe
        $this->configRsiThreshold = (int) ($config['rsi'] ?? 70); // Asumiendo que es un umbral, ajustar según uso real
        $this->interval = (string) ($config['interval'] ?? '15m'); // Valor por defecto si no existe
        $this->proMode = (bool) ($config['proge'] ?? false);
    }

    /**
     * Algoritmo basado en la clase Analysis.
     *
     * @param string $symbol Símbolo del par (ej: "DOTUSDT").
     * @return void Establece la propiedad $this->algResp.
     */
    public function runAlgorithmAnalysis(string $symbol = "DOTUSDT"): void
    {
        $this->resetState(); // Reiniciar estado interno

        try {
            // 1. Obtener Datos K-line
            $klines = $this->fetchKlines($symbol);

            // 2. Extraer Datos OHLCV
            $data = $this->extractOhlcv($klines);
            $actprice = (float) end($data['close']);
            $lastKlineData = end($klines); // Para setTokenTel

            // 3. Ejecutar Clase Analysis
            // Considerar inyectar Analysis si tiene dependencias o estado complejo
            $analysis = new Analysis($data['close'], $data['high'], $data['low'], $data['volum']);
            $analysisResult = $analysis->analyze(); // Asume que analyze() devuelve un array como en el original

            // Validar resultado de Analysis
            if (!isset($analysisResult['action'], $analysisResult['stopLoss'], $analysisResult['takeProfit'], $analysisResult['actPrice'])) {
                throw new CalculationException("El resultado de Analysis::analyze() es incompleto.");
            }

            $signalAction = $analysisResult['action']; // 'LONG', 'SHORT', 'HOLD' (asumiendo)
            $tp = (float) $analysisResult['stopLoss']; // Parece que los nombres estaban invertidos en el original? Verificar clase Analysis
            $sl = (float) $analysisResult['takeProfit']; // Parece que los nombres estaban invertidos en el original? Verificar clase Analysis
            // $actprice = (float) $analysisResult['actPrice']; // Usar el calculado desde $data['close'] o este? Usaremos el de $data.

            $bands = [
                'supo' => $analysisResult['pivotSu'] ?? 0,
                'resi' => $analysisResult['pivotRe'] ?? 0,
                'trai' => '', // Valor por defecto o calcular si es necesario
                'tren' => $analysisResult['pivotTrend'] ?? ''
            ];

            // 4. Procesar Señal y Calcular TP/SL Porcentual (¡Lógica corregida!)
            $accion = "Mantener";
            $emitir = 0;
            $potp = 0.0;
            $posl = 0.0;

            if ($signalAction === 'LONG' && $tp > 0 && $sl > 0 && $sl < $actprice && $tp > $actprice) {
                $accion = "Comprar";
                $emitir = 1;
                $this->CV = 1;
                $this->ACT = 1;
                if ($actprice > 0) {
                    $potp = round((($tp / $actprice) - 1) * 100, 2); // % Ganancia TP
                    $posl = round((1 - ($sl / $actprice)) * 100, 2); // % Pérdida SL
                }
            } elseif ($signalAction === 'SHORT' && $tp > 0 && $sl > 0 && $sl > $actprice && $tp < $actprice) {
                $accion = "Vender";
                $emitir = 1;
                $this->CV = 2;
                $this->ACT = 2;
                if ($actprice > 0) {
                    $potp = round((1 - ($tp / $actprice)) * 100, 2); // % Ganancia TP
                    $posl = round((($sl / $actprice) - 1) * 100, 2); // % Pérdida SL
                }
            }

            // 5. Aplicar Lógica de Negocio Adicional (Umbral TP)
            if ($potp > 0 && $posl > 0) {
                 if ($potp < self::MIN_POTP_THRESHOLD) {
                     $this->printLog("Analysis - TP ({$potp}%) por debajo del umbral mínimo (" . self::MIN_POTP_THRESHOLD . "%). Ajustando a Mantener.", $symbol);
                     // Resetear a Mantener
                     $accion = "Mantener";
                     $emitir = 0;
                     $this->CV = 0;
                     $this->ACT = 0;
                     $potp = 0.0;
                     $posl = 0.0;
                     $tp = 0.0;
                     $sl = 0.0;
                 } else {
                      $this->printLog("Analysis - Señal: {$accion} | Precio: {$actprice} | TP: {$tp} ({$potp}%) | SL: {$sl} ({$posl}%)", $symbol);
                 }
            } else {
                 // Si potp o posl no son positivos, asegurar estado Mantener
                 if ($accion !== "Mantener") {
                     $this->printLog("Analysis - TP/SL inválidos o cero para señal {$accion}. Forzando a Mantener.", $symbol);
                     $accion = "Mantener";
                     $emitir = 0;
                     $this->CV = 0;
                     $this->ACT = 0;
                     $potp = 0.0;
                     $posl = 0.0;
                 }
            }


            // 6. Preparar Datos para Notificación (Telegram)
            $intervalData = ['interval' => $this->interval]; // Simplificado
            $tokentel = $this->setTokenTel($lastKlineData, $bands, $intervalData, $this->configRsiThreshold, $accion, $this->ACT);
            $tokentel["potp"] = $potp;
            $tokentel["posl"] = $posl;

            // 7. Establecer Resultado Final
            $this->algResp = [
                "CV" => $this->CV,
                "emitir" => $emitir,
                "tokentel" => $tokentel,
                "rsi" => 0, // El RSI real no se calcula aquí, se usa el umbral de config
                "act" => $this->ACT,
                "action" => $accion,
                "details" => $analysisResult // Incluir detalles de la clase Analysis si es útil
            ];

        } catch (ResponseApiException | CalculationException $e) {
            $this->printLog("Error en runAlgorithmAnalysis ({$symbol}): " . $e->getMessage(), $symbol);
            $this->algResp = $this->getDefaultErrorResponse("Error Analysis: " . $e->getMessage());
        } catch (\Exception $e) {
            $this->printLog("Error inesperado en runAlgorithmAnalysis ({$symbol}): " . $e->getMessage(), $symbol);
            $this->algResp = $this->getDefaultErrorResponse("Error inesperado Analysis: " . $e->getMessage());
        }
    }


    /**
     * Algoritmo basado en ImpulseMACD + EMA100.
     *
     * @param string $symbol Símbolo del par (ej: "DOTUSDT").
     * @return void Establece la propiedad $this->algResp.
     */
    public function runAlgorithmImpulseMacdEMA100(string $symbol = "DOTUSDT"): void
    {
        $this->resetState(); // Reiniciar estado interno

        try {
            // 1. Obtener Datos K-line
            $klines = $this->fetchKlines($symbol);

            // 2. Extraer Datos OHLCV
            $data = $this->extractOhlcv($klines);
            $actprice = (float) end($data['close']);
            $lastKlineData = end($klines); // Para setTokenTel

            // 3. Calcular Indicador ImpulseMACDWS
            // Considerar inyectar ImpulseMACDWS si tiene dependencias o estado complejo
            $impulseMacd = new ImpulseMACDWS(
                lengthMA: self::IMPULSE_MACD_MA_LENGTH,
                lengthSignal: self::IMPULSE_MACD_SIGNAL_LENGTH,
                thresholdPeriod: self::IMPULSE_MACD_THRESHOLD_PERIOD
            );
            $impulseMacd->calculate($data); // Calcula indicadores internos

            // 4. Ejecutar Estrategia y Generar Señales
            // Considerar inyectar ImpulseMacdStrategy
            $strategy = new ImpulseMacdStrategy(
                impulseMacd: $impulseMacd,
                ohlcData: $data,
                atrPeriod: self::STRATEGY_ATR_PERIOD,
                atrMultiplierSL: self::STRATEGY_ATR_MULTIPLIER_SL, // Usar constante de clase o config
                rrRatio: self::STRATEGY_RR_RATIO                 // Usar constante de clase o config
            );

            $signals = $strategy->generateSignals(confirmationPeriods: self::STRATEGY_CONFIRMATION_PERIODS);
            $lastSignal = end($signals); // Obtener la señal más reciente
            if (!$lastSignal || !isset($lastSignal['signal'], $lastSignal['takeProfit'], $lastSignal['stopLoss'])) {
                 //throw new CalculationException("No se pudo generar la última señal válida de la estrategia para {$symbol}.");
                 return;
            }

            // 5. Procesar Señal y Calcular TP/SL
            $signalValue = (int) $lastSignal['signal'];
            $tp = (float) $lastSignal['takeProfit'];
            $sl = (float) $lastSignal['stopLoss'];
            $currentEma = (float) $lastSignal['currentEma'];
            $currentAtr = (float) $lastSignal['currentAtr'];
            $currentMD = (float) $lastSignal['currentMD'];
            $currentSB = (float) $lastSignal['currentSB'];
            $currentSH = (float) $lastSignal['currentSH'];
            $currentTH = (float) $lastSignal['currentTH'];
            $accion = "Mantener";
            $emitir = 0;
            $potp = 0.0;
            $posl = 0.0;

            if ($signalValue === 1 && $tp > 0 && $sl > 0 && $sl < $actprice && $tp > $actprice) {
                $accion = "Comprar";
                $emitir = 1;
                $this->CV = 1;
                $this->ACT = 1;
                 if ($actprice > 0) {
                    $potp = round((($tp / $actprice) - 1) * 100, 2);
                    $posl = round((1 - ($sl / $actprice)) * 100, 2);
                 }
            } elseif ($signalValue === -1 && $tp > 0 && $sl > 0 && $sl > $actprice && $tp < $actprice) {
                $accion = "Vender";
                $emitir = 1;
                $this->CV = 2;
                $this->ACT = 2;
                 if ($actprice > 0) {
                    $potp = round((1 - ($tp / $actprice)) * 100, 2);
                    $posl = round((($sl / $actprice) - 1) * 100, 2);
                 }
            }

            // 6. Aplicar Lógica de Negocio Adicional (Umbral TP)
             if ($potp > 0 && $posl > 0) {
                 if ($potp < self::MIN_POTP_THRESHOLD) {
                     $this->printLog("ImpulseMACD - TP ({$potp}%) por debajo del umbral mínimo (" . self::MIN_POTP_THRESHOLD . "%). Ajustando a Mantener.", $symbol);
                     // Resetear a Mantener
                     $accion = "Mantener";
                     $emitir = 0;
                     $this->CV = 0;
                     $this->ACT = 0;
                     $potp = 0.0;
                     $posl = 0.0;
                     $tp = 0.0;
                     $sl = 0.0;
                 } else {
                    $this->printLog("IMACD - EMA:{$currentEma} ATR:{$currentAtr} MD:{$currentMD} SB:{$currentSB} SH:{$currentSH} TH:{$currentTH} Act: {$accion} | Pre: {$actprice} | TP: {$tp} ({$potp}%) | SL: {$sl} ({$posl}%)", $symbol);
                 }
            } else {
                 // Si potp o posl no son positivos, asegurar estado Mantener
                 if ($accion !== "Mantener") {
                     $this->printLog("ImpulseMACD - TP/SL inválidos o cero para señal {$accion}. Forzando a Mantener.", $symbol);
                     $accion = "Mantener";
                     $emitir = 0;
                     $this->CV = 0;
                     $this->ACT = 0;
                     $potp = 0.0;
                     $posl = 0.0;
                 }
            }

            // 7. Preparar Datos para Notificación (Telegram)
            // Los valores de $bands no se calculan aquí para esta estrategia, se pasan vacíos/default.
            $bands = ['supo' => 0, 'resi' => 0, 'trai' => '', 'tren' => ''];
            $intervalData = ['interval' => $this->interval];
            $tokentel = $this->setTokenTel($lastKlineData, $bands, $intervalData, $this->configRsiThreshold, $accion, $this->ACT);
            $tokentel["potp"] = $potp; // Añadir porcentajes calculados
            $tokentel["posl"] = $posl;

            // 8. Establecer Resultado Final
            $this->algResp = [
                "CV" => $this->CV,
                "emitir" => $emitir,
                "tokentel" => $tokentel,
                "rsi" => 0, // RSI no calculado explícitamente aquí
                "act" => $this->ACT,
                "action" => $accion,
                "details" => $lastSignal // Incluir detalles de la señal final
            ];

        } catch (ResponseApiException | CalculationException $e) {
            $this->printLog("Error en runAlgorithmImpulseMacdEMA100 ({$symbol}): " . $e->getMessage(), $symbol);
            $this->algResp = $this->getDefaultErrorResponse("Error ImpulseMACD: " . $e->getMessage());
        } catch (\Exception $e) {
            $this->printLog("Error inesperado en runAlgorithmImpulseMacdEMA100 ({$symbol}): " . $e->getMessage(), $symbol);
            $this->algResp = $this->getDefaultErrorResponse("Error inesperado ImpulseMACD: " . $e->getMessage());
        }
    }

    /**
     * Algoritmo basado en ImpulseMACD + BreakoutProbability.
     *
     * @param string $symbol Símbolo del par (ej: "DOTUSDT").
     * @return void Establece la propiedad $this->algResp.
     */
    public function runAlgorithmImpulseMacdBreakPro(string $symbol = "DOTUSDT"): void
    {
        $this->resetState(); // Reiniciar estado interno

        try {
            // 1. Obtener Datos K-line
            $klines = $this->fetchKlines($symbol);

            // 2. Extraer Datos OHLCV
            $data = $this->extractOhlcv($klines);
            $actprice = (float) end($data['close']);
            $lastKlineData = end($klines); // Para setTokenTel

            // 3. Calcular Indicador ImpulseMACDWS
            // Considerar inyectar ImpulseMACDWS si tiene dependencias o estado complejo
            $lastSignal = $this->getImpulseMACDWS($data);
            if ($lastSignal === null) {
                return;
            }           
            $filters = ['usarATR' => false, 'ponderacionTemporal' => false, 'filtrarVolumen' => false];
            $breakout = new BreakoutProbabilityKlinesCal($data, $filters);
            $breakout->setPercentageStep(self::PERCENTAGE_STEP);
            $breakout->setNumberOfLevels(self::NUMBER_OF_LEVELS);
            $breakout->setShowZeroProbabilities(self::SHOW_ZERO_PROBABILITY);
            $breakout->setAtrPeriod(self::STRATEGY_ATR_PERIOD);
            // Obtener el resultado en formato JSON
            $jsonResult = $breakout->calculate();  // Calcula indicadores internos upper y lower
            $probability = json_decode($jsonResult, true);
            // 4. Ejecutar Estrategia y Generar Señales

            $breakoutStr = new BreakoutProbabilityKLinesStrategy(
                //impulseMacd: $impulseMacd, 
                ohlcData: $data,
                probability: $probability,
                filters: $filters
            );
            $breakoutStr->setNoLevelZero(self::NO_LEVEL_ZERO);
            $breakoutStr->setShowZeroProbabilities(self::SHOW_ZERO_PROBABILITIES);
            $breakoutStr->setShowStats(self::SHOW_STATS);
            $breakoutStr->setStopLossThreshold(self::STOP_LOSS_THRESHOLD); // Umbral de probabilidad para stop loss
            $breakoutStr->setTpSlRatio(self::TP_SL_RATIO); // Ratio Take Profit / Stop Loss
	        $breakoutStr->setTrendThreshold(self::TREND_THRESHOL); // Umbral para determinar la tendencia
            $breakoutValue = $breakoutStr->calculate();
            // 5. Procesar Señal y Calcular TP/SL

            $entrySignal = $breakoutValue['entry_signal'];
            $signalValue = $entrySignal['action'];
            $tp = (float) $entrySignal['take_profit'];
            $sl = (float) $entrySignal['stop_loss'];
            $accion = "Mantener";
            $emitir = 0;
            $potp = 0.0;
            $posl = 0.0;

            if ($signalValue === 1 && $tp > 0 && $sl > 0 && $sl < $actprice && $tp > $actprice) {
                $accion = "Comprar";
                $emitir = 1;
                $this->CV = 1;
                $this->ACT = 1;
                 if ($actprice > 0) {
                    $potp = round((($tp / $actprice) - 1) * 100, 2);
                    $posl = round((1 - ($sl / $actprice)) * 100, 2);
                 }
            } elseif ($signalValue === -1 && $tp > 0 && $sl > 0 && $sl > $actprice && $tp < $actprice) {
                $accion = "Vender";
                $emitir = 1;
                $this->CV = 2;
                $this->ACT = 2;
                 if ($actprice > 0) {
                    $potp = round((1 - ($tp / $actprice)) * 100, 2);
                    $posl = round((($sl / $actprice) - 1) * 100, 2);
                 }
            }

            // 6. Aplicar Lógica de Negocio Adicional (Umbral TP)
             if ($potp > 0 && $posl > 0) {
                 if ($potp < self::MIN_POTP_THRESHOLD) {
                     $this->printLog("ImpulseMACD_BP - TP ({$potp}%) por debajo del umbral mínimo (" . self::MIN_POTP_THRESHOLD . "%). Ajustando a Mantener.", $symbol);
                     // Resetear a Mantener
                     $accion = "Mantener";
                     $emitir = 0;
                     $this->CV = 0;
                     $this->ACT = 0;
                     $potp = 0.0;
                     $posl = 0.0;
                     $tp = 0.0;
                     $sl = 0.0;
                 } else {
                    $this->printLog("IMACD_BP - EMA:{$currentEma} ATR:{$currentAtr} MD:{$currentMD} SB:{$currentSB} SH:{$currentSH} TH:{$currentTH} Act: {$accion} | Pre: {$actprice} | TP: {$tp} ({$potp}%) | SL: {$sl} ({$posl}%)", $symbol);
                 }
            } else {
                 // Si potp o posl no son positivos, asegurar estado Mantener
                 if ($accion !== "Mantener") {
                     $this->printLog("ImpulseMACD_BP - TP/SL inválidos o cero para señal {$accion}. Forzando a Mantener.", $symbol);
                     $accion = "Mantener";
                     $emitir = 0;
                     $this->CV = 0;
                     $this->ACT = 0;
                     $potp = 0.0;
                     $posl = 0.0;
                 }
            }

            // 7. Preparar Datos para Notificación (Telegram)
            // Los valores de $bands no se calculan aquí para esta estrategia, se pasan vacíos/default.
            $bands = ['supo' => 0, 'resi' => 0, 'trai' => '', 'tren' => ''];
            $intervalData = ['interval' => $this->interval];
            $tokentel = $this->setTokenTel($lastKlineData, $bands, $intervalData, $this->configRsiThreshold, $accion, $this->ACT);
            $tokentel["potp"] = $potp; // Añadir porcentajes calculados
            $tokentel["posl"] = $posl;

            // 8. Establecer Resultado Final
            $this->algResp = [
                "CV" => $this->CV,
                "emitir" => $emitir,
                "tokentel" => $tokentel,
                "rsi" => 0, // RSI no calculado explícitamente aquí
                "act" => $this->ACT,
                "action" => $accion,
                "details" => $lastSignal // Incluir detalles de la señal final
            ];

        } catch (ResponseApiException | CalculationException $e) {
            $this->printLog("Error en runAlgorithmImpulseMacdEMA100 ({$symbol}): " . $e->getMessage(), $symbol);
            $this->algResp = $this->getDefaultErrorResponse("Error ImpulseMACD: " . $e->getMessage());
        } catch (\Exception $e) {
            $this->printLog("Error inesperado en runAlgorithmImpulseMacdEMA100 ({$symbol}): " . $e->getMessage(), $symbol);
            $this->algResp = $this->getDefaultErrorResponse("Error inesperado ImpulseMACD: " . $e->getMessage());
        }
    }

    /**
     * Algoritmo basado en SmartMoney o  ImpulseMACD + BreakoutProbability.
     *
     * @param string $symbol Símbolo del par (ej: "DOTUSDT").
     * @return void Establece la propiedad $this->algResp.
     */
    public function runAlgorithmImpulseMacdSmartMoney(string $symbol = "DOTUSDT"): void
    {

        $this->resetState(); // Reiniciar estado interno

        try {
            // 1. Obtener Datos K-line
            $klines = $this->fetchKlines($symbol);

            // 2. Extraer Datos OHLCV
            $data = $this->extractOhlcv($klines);
            $actprice = (float) end($data['close']);
            $lastKlineData = end($klines); // Para setTokenTel
            // 3. Calcular Indicador SmartMoneyVolume
            $smartMoneyVolume = new SmartMoneyVolume($data);
            // 4. Ejecutar Estrategia y Generar Señales
            $smartMoneyFusionStrategy = new SmartMoneyFusionStrategy($smartMoneyVolume);
            // --- Obtener la señal potencial de SmartMoney ---
            $smartMoneySignal = $smartMoneyFusionStrategy->getPotentialSignal(); 
            if ($smartMoneySignal === null) {
                $this->runAlgorithmImpulseMacdEMA100();
            }
            $finalDecisionToExecute = false;
            //Calcular Indicador ImpulseMACDWS
            $macdSignal = $this->getImpulseMACDWS($data);
            if ($macdSignal === null) {
                return;
            }
            if ($smartMoneySignal['confidence'] >= 50 && $smartMoneySignal['signal'] === $macdSignal) {
                 $finalDecisionToExecute = true;
            }
            if (!$finalDecisionToExecute) {
                $this->printLog("ImpulseMACD_SM - no reporta señal potencial básica.\n");
                $this->algResp = [
                    "CV" => 0,
                    "emitir" => 0,
                    "tokentel" => $tokentel,
                    "rsi" => 0, // RSI no calculado explícitamente aquí
                    "act" => 0,
                    "action" => 'mantener',
                    "details" => '' // Incluir detalles de la señal final
                ];
                return;
            }
            // 5. Procesar Señal y Calcular TP/SL

            $signalValue = $smartMoneySignal['signal'];
            $tp = (float) $smartMoneySignal['take_profit'];
            $sl = (float) $smartMoneySignal['stop_loss'];
            $accion = "Mantener";
            $emitir = 0;
            $potp = 0.0;
            $posl = 0.0;     
            if ($signalValue === 1 && $tp > 0 && $sl > 0 && $sl < $actprice && $tp > $actprice) {
                $accion = "Comprar";
                $emitir = 1;
                $this->CV = 1;
                $this->ACT = 1;
                 if ($actprice > 0) {
                    $potp = round((($tp / $actprice) - 1) * 100, 2);
                    $posl = round((1 - ($sl / $actprice)) * 100, 2);
                 }
            } elseif ($signalValue === -1 && $tp > 0 && $sl > 0 && $sl > $actprice && $tp < $actprice) {
                $accion = "Vender";
                $emitir = 1;
                $this->CV = 2;
                $this->ACT = 2;
                 if ($actprice > 0) {
                    $potp = round((1 - ($tp / $actprice)) * 100, 2);
                    $posl = round((($sl / $actprice) - 1) * 100, 2);
                 }
            }

            // 6. Aplicar Lógica de Negocio Adicional (Umbral TP)
             if ($potp > 0 && $posl > 0) {
                 if ($potp < self::MIN_POTP_THRESHOLD) {
                     $this->printLog("ImpulseMACD_SM - TP ({$potp}%) por debajo del umbral mínimo (" . self::MIN_POTP_THRESHOLD . "%). Ajustando a Mantener.", $symbol);
                     // Resetear a Mantener
                     $accion = "Mantener";
                     $emitir = 0;
                     $this->CV = 0;
                     $this->ACT = 0;
                     $potp = 0.0;
                     $posl = 0.0;
                     $tp = 0.0;
                     $sl = 0.0;
                 } else {
                    $this->printLog("IMACD_SM - Act: {$accion} | Pre: {$actprice} | TP: {$tp} ({$potp}%) | SL: {$sl} ({$posl}%)", $symbol);
                 }
            } else {
                 // Si potp o posl no son positivos, asegurar estado Mantener
                 if ($accion !== "Mantener") {
                     $this->printLog("ImpulseMACD_SM - TP/SL inválidos o cero para señal {$accion}. Forzando a Mantener.", $symbol);
                     $accion = "Mantener";
                     $emitir = 0;
                     $this->CV = 0;
                     $this->ACT = 0;
                     $potp = 0.0;
                     $posl = 0.0;
                 }
            }                   
            // 7. Preparar Datos para Notificación (Telegram)
            // Los valores de $bands no se calculan aquí para esta estrategia, se pasan vacíos/default.
            $bands = ['supo' => 0, 'resi' => 0, 'trai' => '', 'tren' => ''];
            $intervalData = ['interval' => $this->interval];
            $tokentel = $this->setTokenTel($lastKlineData, $bands, $intervalData, $this->configRsiThreshold, $accion, $this->ACT);
            $tokentel["potp"] = $potp; // Añadir porcentajes calculados
            $tokentel["posl"] = $posl;

            // 8. Establecer Resultado Final
            $this->algResp = [
                "CV" => $this->CV,
                "emitir" => $emitir,
                "tokentel" => $tokentel,
                "rsi" => 0, // RSI no calculado explícitamente aquí
                "act" => $this->ACT,
                "action" => $accion,
                "details" => $lastSignal // Incluir detalles de la señal final
            ];

        } catch (ResponseApiException | CalculationException $e) {
            $this->printLog("Error en runAlgorithmImpulseMacdSmartMoney ({$symbol}): " . $e->getMessage(), $symbol);
            $this->algResp = $this->getDefaultErrorResponse("Error SmartMoney: " . $e->getMessage());
        } catch (\Exception $e) {
            $this->printLog("Error inesperado en runAlgorithmImpulseMacdSmartMoney ({$symbol}): " . $e->getMessage(), $symbol);
            $this->algResp = $this->getDefaultErrorResponse("Error inesperado SmartMoney: " . $e->getMessage());
        }
    }    

    /**
     * Obtiene la última respuesta generada por un algoritmo.
     *
     * @return array La respuesta del algoritmo o una respuesta de error por defecto.
     */
    public function getAlgResp(): array
    {
        // Devolver la respuesta almacenada o una estructura vacía/error si no se ha ejecutado nada
        return $this->algResp ?: $this->getDefaultErrorResponse("Ningún algoritmo ha sido ejecutado aún.");
    }

    // --- Métodos Privados/Protegidos Auxiliares ---

    /**
     * Reinicia el estado interno antes de ejecutar un algoritmo.
     */
    private function resetState(): void
    {
        $this->CV = 0;
        $this->ACT = 0;
        $this->algResp = []; // Limpiar respuesta anterior
    }

    /**
     * Obtiene los datos K-line de la API.
     *
     * @param string $symbol
     * @return array Array de Klines.
     * @throws ResponseApiException Si la API falla o devuelve datos vacíos.
     */
    private function fetchKlines(string $symbol): array
    {
        $klines = $this->apiResponse->getKlines($this->interval, self::KLINE_LIMIT, $symbol);
        if (empty($klines)) {
            throw new ResponseApiException("No se recibieron datos K-line para {$symbol} en el intervalo {$this->interval}.");
        }
        return $klines;
    }

    /**
     * Extrae datos OHLCV de los datos K-line.
     *
     * @param array $klines Array de Klines obtenido de fetchKlines.
     * @return array Array asociativo con claves 'high', 'low', 'close', 'volum'.
     * @throws CalculationException Si faltan datos después de la extracción.
     */
    private function extractOhlcv(array $klines): array
    {
        // Asume que las claves son 'high', 'low', 'close', 'volact' (o el nombre correcto del volumen)
        $data = [
            'open'  => array_column($klines, 'open'),
            'high'  => array_column($klines, 'high'),
            'low'   => array_column($klines, 'low'),
            'close' => array_column($klines, 'close'),
            'volum' => array_column($klines, 'volact'), // ¡Verifica esta clave!
        ];

        // Validar que todos los arrays extraídos tienen datos y la misma longitud
        if (empty($data['close']) || count($data['close']) !== count($klines)) {
             throw new CalculationException("Faltan datos 'close' o longitud incorrecta tras extracción para el símbolo.");
        }
         if (empty($data['high']) || count($data['high']) !== count($klines)) {
             throw new CalculationException("Faltan datos 'high' o longitud incorrecta tras extracción para el símbolo.");
         }
         if (empty($data['low']) || count($data['low']) !== count($klines)) {
             throw new CalculationException("Faltan datos 'low' o longitud incorrecta tras extracción para el símbolo.");
         }
         if (empty($data['volum']) || count($data['volum']) !== count($klines)) {
            // Podrías permitir volumen vacío si tu estrategia no lo usa estrictamente
            // throw new CalculationException("Faltan datos 'volum' o longitud incorrecta tras extracción para el símbolo.");
            $this->printLog("Advertencia: Faltan datos de volumen o longitud incorrecta.", "OHLCV Extraction");
         }

        // Convertir a float explícitamente si la API devuelve strings
        $data['high'] = array_map('floatval', $data['high']);
        $data['low'] = array_map('floatval', $data['low']);
        $data['close'] = array_map('floatval', $data['close']);
        $data['volum'] = array_map('floatval', $data['volum']);


        return $data;
    }

    /**
     * Calcula ImpulseMACDWS
     *
     * @param array data Datos OHLC.
     * @return array ultima señal
     */

    private function getImpulseMACDWS($data): array
    {
        $impulseMacd = new ImpulseMACDWS(
            lengthMA: self::IMPULSE_MACD_MA_LENGTH,
            lengthSignal: self::IMPULSE_MACD_SIGNAL_LENGTH,
            thresholdPeriod: self::IMPULSE_MACD_THRESHOLD_PERIOD
        );
        $impulseMacd->calculate($data); // Calcula indicadores internos
        $signals = $impulseMacd->generateSignals(self::IMPULSE_MACD_CONFIRMATION_PERIOD);
        $lastSignal = end($signals); // Obtener la señal más reciente
        if (!$lastSignal || $lastSignal == 0) {
            //throw new CalculationException("No se pudo generar la última señal válida de la estrategia para {$symbol}.");
            $this->algResp = [
                "CV" => 0,
                "emitir" => 0,
                "tokentel" => $tokentel,
                "rsi" => 0, // RSI no calculado explícitamente aquí
                "act" => 0,
                "action" => 'mantener',
                "details" => $lastSignal // Incluir detalles de la señal final
            ];
            return null;
        } 
        return $lastSignal;
    }

    /**
     * Prepara el payload para la notificación de Telegram.
     *
     * @param array $lastKlineData Datos de la última vela.
     * @param array $bands Datos de soportes/resistencias/tendencia.
     * @param array $intervalData Información del intervalo.
     * @param int $rsiThreshold Umbral RSI configurado (no el valor calculado).
     * @param string $accion Acción decidida ("Comprar", "Vender", "Mantener").
     * @param int $act Estado ACT.
     * @return array Payload para Telegram.
     */
    private function setTokenTel(array $lastKlineData, array $bands, array $intervalData, int $rsiThreshold, string $accion, int $act): array
    {
        // Usar operador de fusión null para seguridad
        $openTime = ($lastKlineData["openTime"] ?? null);
        $closeTime = ($lastKlineData["closeTime"] ?? null);

        $fechaope = $openTime ? (new DateTime('@' . floor($openTime / 1000)))->format('Y-m-d H:i:s') : null;
        $fechaclo = $closeTime ? (new DateTime('@' . floor($closeTime / 1000)))->format('Y-m-d H:i:s') : null;

        $interarry = "[" . ($intervalData["interval"] ?? '?') . "]";

        return [
            "fechaope" => $fechaope,
            "fechaclo" => $fechaclo,
            "support" => $bands['supo'] ?? 0,
            "resistance" => $bands['resi'] ?? 0,
            "potp" => 0, // Se establecerá después
            "posl" => 0, // Se establecerá después
            "rsi_threshold" => $rsiThreshold, // Indicar que es el umbral
            'rsi' => 0,
            "act" => $act,
            "close" => $lastKlineData['close'] ?? null,
            "open" => $lastKlineData['open'] ?? null,
            "interval" => $interarry,
            "interlow" => $intervalData["interval"] ?? null,
            "trail" => $bands['trai'] ?? '',
            "trend" => $bands['tren'] ?? '',
            "accion" => $accion,
            "may" => '', // ¿Qué representa 'may'?
            "tim" => $closeTime ? floor($closeTime / 1000) : null,
            "init" => 0, // ¿Qué representa 'init'?
            'timezone' => $this->config['timezone'] ?? 'UTC',
            'locale' => $this->config['locale'] ?? 'en_US',
            'telusers' => $this->config['telusers'] ?? [],
            'algorit' => $this->config['algorit'] ?? 'unknown',
            "resCV" => [] // ¿Qué representa 'resCV'?
        ];
    }

    /**
     * Imprime un mensaje de log.
     * Reemplazar con una implementación de Logger (PSR-3) para producción.
     *
     * @param string $message Mensaje a loguear.
     * @param string|null $context Contexto adicional (ej: símbolo).
     */
    private function printLog(string $message, ?string $context = null): void
    {
        $date = (new DateTime('now'))->format('Y-m-d H:i:s');
        $logLine = "Now: " . $date;
        if ($context) {
            $logLine .= " [{$context}]";
        }
        $logLine .= "| " . $message . chr(10);
        // Cambiar 'print' por un sistema de logging real (ej: Monolog)
        print($logLine);
        // error_log($logLine); // Alternativa simple
    }

    /**
     * Devuelve una estructura de respuesta por defecto en caso de error.
     *
     * @param string $errorMessage Mensaje de error.
     * @return array
     */
     private function getDefaultErrorResponse(string $errorMessage): array
     {
         return [
             "CV" => 0,
             "emitir" => 0,
             "tokentel" => [ // Token mínimo para evitar errores posteriores
                 'error' => $errorMessage,
                 'accion' => 'Error',
             ],
             "rsi" => 0,
             "act" => 0,
             "action" => "Error",
             "error" => $errorMessage
         ];
     }

    /**
     * Método de prueba para cálculos de TP/SL (Corregido para aceptar parámetros).
     *
     * @param float $price Precio actual.
     * @param float $atrlast Último valor ATR.
     * @param array|null $bands Datos de soportes/resistencias (opcional).
     */
    public function testTpSlCalculations(float $price, float $atrlast, ?array $bands = null): void
    {
        $this->printLog("--- Iniciando Test TP/SL ---", "TEST");
        $this->printLog("Precio: {$price}, ATR: {$atrlast}, FTP: {$this->configFtp}, FSL: {$this->configFsl}", "TEST");

        if ($price <= 0) {
            $this->printLog("Precio inválido para test TP/SL.", "TEST");
            return;
        }

        // --- Cálculos por ATR ---
        $tp_atr_long = ($price + ($this->configFtp * $atrlast));
        $sl_atr_long = ($price - ($this->configFsl * $atrlast));
        $potp_atr_long = round((($tp_atr_long / $price) - 1) * 100, 2);
        $posl_atr_long = round((1 - ($sl_atr_long / $price)) * 100, 2);
        $this->printLog("ATR Compra -> TP: {$tp_atr_long} ({$potp_atr_long}%) SL: {$sl_atr_long} ({$posl_atr_long}%)", "TEST");

        $tp_atr_short = ($price - ($this->configFtp * $atrlast));
        $sl_atr_short = ($price + ($this->configFsl * $atrlast));
        $potp_atr_short = round((1 - ($tp_atr_short / $price)) * 100, 2);
        $posl_atr_short = round((($sl_atr_short / $price) - 1) * 100, 2);
        $this->printLog("ATR Venta  -> TP: {$tp_atr_short} ({$potp_atr_short}%) SL: {$sl_atr_short} ({$posl_atr_short}%)", "TEST");

        // --- Cálculos por S/R (si se proporcionan) ---
        if ($bands && isset($bands['supo'], $bands['resi']) && $bands['supo'] > 0 && $bands['resi'] > 0 && $bands['resi'] > $bands['supo']) {
            $this->printLog("Soporte: {$bands['supo']}, Resistencia: {$bands['resi']}", "TEST");
            $margin = 0.005; // Margen, podría ser configurable

            // SR Compra (TP en Resistencia, SL en Soporte)
            $tp_sr_long = $bands['resi'] - $margin;
            $sl_sr_long = $bands['supo'] - $margin;
            if ($tp_sr_long > $price && $sl_sr_long < $price && $sl_sr_long > 0) { // Validaciones básicas
                 $potp_sr_long = round((($tp_sr_long / $price) - 1) * 100, 2);
                 $posl_sr_long = round((1 - ($sl_sr_long / $price)) * 100, 2);
                 $this->printLog("SR Compra  -> TP: {$tp_sr_long} ({$potp_sr_long}%) SL: {$sl_sr_long} ({$posl_sr_long}%)", "TEST");
            } else {
                 $this->printLog("SR Compra -> Valores TP/SL inválidos basados en S/R y precio actual.", "TEST");
            }


            // SR Venta (TP en Soporte, SL en Resistencia)
            $tp_sr_short = $bands['supo'] + $margin;
            $sl_sr_short = $bands['resi'] + $margin;
             if ($tp_sr_short < $price && $sl_sr_short > $price) { // Validaciones básicas
                 $potp_sr_short = round((1 - ($tp_sr_short / $price)) * 100, 2);
                 $posl_sr_short = round((($sl_sr_short / $price) - 1) * 100, 2);
                 $this->printLog("SR Venta   -> TP: {$tp_sr_short} ({$potp_sr_short}%) SL: {$sl_sr_short} ({$posl_sr_short}%)", "TEST");
             } else {
                  $this->printLog("SR Venta  -> Valores TP/SL inválidos basados en S/R y precio actual.", "TEST");
             }

        } else {
            $this->printLog("No se proporcionaron datos de S/R válidos para el test.", "TEST");
        }
         $this->printLog("--- Fin Test TP/SL ---", "TEST");
    }

    // --- Métodos obsoletos (reemplazados por funciones nativas de PHP >= 7.3) ---
    /*
    private function array_key_last(array $arr) {
        // Usar array_key_last($arr) en su lugar
    }
    private function array_key_first(array $arr) {
        // Usar array_key_first($arr) en su lugar
    }
    */
}