<?php

require __DIR__ . '/vendor/autoload.php';
use App\SendTelegram;
use App\TelegramTemplate; 
use App\DailyRotatingLogger;
use React\EventLoop\LoopInterface;
use React\EventLoop\Loop;
use React\Http\Browser;
use React\Promise\PromiseInterface;
use React\ChildProcess\Process;

/**
 * Clase principal del demonio de monitoreo
 */
class BotMonitorDaemonActual
{
    // --- Configuración por defecto y Constantes ---
    private const DEFAULT_MONITOR_CONFIG_PATH = __DIR__ . '/config/monitor_config.json';
    private const DEFAULT_CHECK_INTERVAL = 60 * 15; // segundos (frecuencia general de chequeo de estado de bots)
    private const DEFAULT_RESTART_DELAY = 5; // segundos
    private const DEFAULT_STARTUP_DELAY = 2; // segundos
    private const DEFAULT_LOG_PATH = __DIR__ . '/logs';
    private const DEFAULT_LOG_RETENTION_DAYS = 7;
    
    // --- Propiedades de la clase ---
    private LoopInterface $loop;
    private DailyRotatingLogger $logger;
    private array $monitorConfig;
    private array $botStates = [];
    private Browser $httpClient;
    private SendTelegram $telegramSender;
    
    // --- Configuración de aplicaciones a monitorear ---
    private array $applications = [
        'bitradenes' => [
            'service_name' => 'bitradenes-bot.service',
            'bot_name' => 'Bitradenes (Neskk)',
            'description' => 'Bot de trading Bitradenes',
            'telsock' => 'https://api.telegram.org/bot6790494013:AAGVCyUCLOc1fJNPofONFZjo4UBYgckjlTo/' //AliNesTrade
        ],
        'bitrademacdrsi' => [
            'service_name' => 'bitradeali-bot.service',
            'bot_name' => 'Bitrademacdrsi (MACD-RSI)',
            'description' => 'Bot de trading MACD-RSI',
            'telsock' => 'https://api.telegram.org/bot7820191435:AAHpAbA724C5sx4EsMCKOtwXIV0BzJ7oqO0/', //IlNostroCommercio
        ]
    ];


    /**
     * Constructor
     */
    public function __construct(
        ?string $configPath = null,
        ?string $logPath = null,
        ?SendTelegram $telegramSender = null
    ) {
        $this->loop = Loop::get();
        $this->httpClient = new Browser($this->loop);
        
        // Inicializar Telegram sender
        $this->telegramSender = $telegramSender ?? new SendTelegram();
        
        // Inicializar logger
        $logPath = $logPath ?? self::DEFAULT_LOG_PATH . "/monitor_bot.log";
        $this->initializeLogger($logPath);
        
        // Cargar configuración
        $configPath = $configPath ?? self::DEFAULT_MONITOR_CONFIG_PATH;
        $this->loadConfig($configPath);
        
        $this->logger->info("BotMonitorDaemonActual inicializado");
    }

    /**
     * Inicializar el logger
     */
    private function initializeLogger(string $logPath): void
    {
        $logDir = dirname($logPath);
        if (!is_dir($logDir)) {
            mkdir($logDir, 0755, true);
        }
        
        $this->logger = new DailyRotatingLogger(
            $logPath,
            self::DEFAULT_LOG_RETENTION_DAYS
        );
    }

    /**
     * Cargar configuración desde archivo JSON
     */
    private function loadConfig(string $configPath): void
    {
        if (file_exists($configPath)) {
            $config = json_decode(file_get_contents($configPath), true);
            if (json_last_error() === JSON_ERROR_NONE) {
                $this->monitorConfig = $config;
                $this->logger->info("Configuración cargada desde: $configPath");
                return;
            }
        }
        
        // Configuración por defecto
        $this->monitorConfig = [
            'check_interval' => self::DEFAULT_CHECK_INTERVAL,
            'restart_delay' => self::DEFAULT_RESTART_DELAY,
            'startup_delay' => self::DEFAULT_STARTUP_DELAY
        ];
        
        $this->logger->warning("Usando configuración por defecto");
    }

    /**
     * Verificar si un servicio systemd está activo
     */
    private function isServiceActive(string $serviceName): bool
    {
        $command = "systemctl is-active " . escapeshellarg($serviceName);
        $output = trim(shell_exec($command) ?? '');
        return $output === 'active';
    }

    /**
     * Verificar si un servicio systemd está habilitado
     */
    private function isServiceEnabled(string $serviceName): bool
    {
        $command = "systemctl is-enabled " . escapeshellarg($serviceName);
        $output = trim(shell_exec($command) ?? '');
        return $output === 'enabled';
    }

    /**
     * Obtener el estado detallado de un servicio
     */
    private function getServiceStatus(string $serviceName): array
    {
        $commands = [
            'active' => "systemctl is-active " . escapeshellarg($serviceName),
            'enabled' => "systemctl is-enabled " . escapeshellarg($serviceName),
            'failed' => "systemctl is-failed " . escapeshellarg($serviceName)
        ];

        $status = [];
        foreach ($commands as $key => $command) {
            $status[$key] = trim(shell_exec($command) ?? '');
        }

        return $status;
    }

    /**
     * Obtener fecha y hora actual formateada
     */
    private function getCurrentDateTime(): array
    {
        return [
            'date' => date('d/m/Y'),
            'time' => date('H:i')
        ];
    }

    /**
     * Enviar notificación de alerta vía Telegram
     */
    private function sendTelegramAlert(string $message, string $appName = ''): void
    {
        try {
            $telusers="";
            $this->logger->info("Enviando alerta de Telegram para: $appName");
            
            // Enviar alerta usando el método de la clase SendTelegram
            $result = $this->telegramSender->sendAlertTelegram($message,$telusers);
            
            if ($result) {
                $this->logger->info("Alerta de Telegram enviada correctamente para: $appName");
            } else {
                $this->logger->error("Error al enviar alerta de Telegram para: $appName");
            }
            
        } catch (Exception $e) {
            $this->logger->error("Excepción al enviar alerta de Telegram para $appName: " . $e->getMessage());
        }
    }

    /**
     * Enviar notificación usando el script PHP (método legacy de respaldo)
     */
    private function sendLegacyNotification(string $notificationScript, string $message): void
    {
        if (!file_exists($notificationScript)) {
            $this->logger->error("Script de notificación no encontrado: $notificationScript");
            return;
        }

        $command = "php " . escapeshellarg($notificationScript) . " " . escapeshellarg($message);
        $process = new Process($command);
        
        $process->start($this->loop);
        $process->on('exit', function($exitCode) use ($message) {
            if ($exitCode === 0) {
                $this->logger->info("Notificación legacy enviada correctamente");
            } else {
                $this->logger->error("Error al enviar notificación legacy. Código de salida: $exitCode");
            }
        });
    }

    /**
     * Reiniciar un servicio systemd
     */
    private function restartService(string $serviceName): bool
    {
        $this->logger->info("Reiniciando servicio: $serviceName");
        
        $command = "systemctl restart " . escapeshellarg($serviceName);
        $exitCode = 0;
        $output = [];
        
        exec($command . " 2>&1", $output, $exitCode);
        
        if ($exitCode === 0) {
            $this->logger->info("Servicio $serviceName reiniciado correctamente");
            return true;
        } else {
            $this->logger->error("Error al reiniciar servicio $serviceName. Output: " . implode("\n", $output));
            return false;
        }
    }

    /**
     * Habilitar un servicio systemd si no está habilitado
     */
    private function enableServiceIfNeeded(string $serviceName): void
    {
        if (!$this->isServiceEnabled($serviceName)) {
            $this->logger->warning("Servicio $serviceName no está habilitado. Habilitando...");
            
            $command = "systemctl enable " . escapeshellarg($serviceName);
            $exitCode = 0;
            $output = [];
            
            exec($command . " 2>&1", $output, $exitCode);
            
            if ($exitCode === 0) {
                $this->logger->info("Servicio $serviceName habilitado correctamente");
            } else {
                $this->logger->error("Error al habilitar servicio $serviceName. Output: " . implode("\n", $output));
            }
        }
    }

    /**
     * Verificar el estado de una aplicación específica (servicio systemd)
     */
    public function checkApplicationStatus(string $appName): void
    {
        if (!isset($this->applications[$appName])) {
            $this->logger->error("Aplicación no configurada: $appName");
            return;
        }

        $appConfig = $this->applications[$appName];
        $serviceName = $appConfig['service_name'];
        $botName = $appConfig['bot_name'];
        $telusers = $appConfig['telsock'];
        $description = $appConfig['description'];
        $datetime = $this->getCurrentDateTime();

        // Obtener estado detallado del servicio
        $serviceStatus = $this->getServiceStatus($serviceName);
        $isActive = $this->isServiceActive($serviceName);

        if ($isActive) {
            $message = "{$datetime['date']} - {$datetime['time']}: ✅ $description ($botName) está ejecutándose correctamente.";
            $this->logger->info($message);
            echo $message . PHP_EOL;
        } else {
            // Verificar si el servicio falló
            $statusInfo = "Estado: {$serviceStatus['active']}, Habilitado: {$serviceStatus['enabled']}";
            if ($serviceStatus['failed'] === 'failed') {
                $statusInfo .= ", Error: {$serviceStatus['failed']}";
            }

            $message = "{$datetime['date']} - {$datetime['time']}: 🚨 ALERTA - $description ($botName) NO está activo. $statusInfo. Reiniciando servicio...";
            $this->logger->warning($message);
            echo $message . PHP_EOL;

            // Habilitar servicio si es necesario
            $this->enableServiceIfNeeded($serviceName);
            
            // Reiniciar el servicio
            $restartSuccess = $this->restartService($serviceName);
            
            // Enviar alerta principal vía Telegram
            $this->sendTelegramAlert($message, $botName, $telusers); 
            
            // Programar verificación de reinicio exitoso
            $this->scheduleServiceRestartVerification($appName, $appConfig, $restartSuccess);
        }
    }

    /**
     * Programar verificación de que el reinicio del servicio fue exitoso
     */
    private function scheduleServiceRestartVerification(string $appName, array $appConfig, bool $restartSuccess): void
    {
        $restartDelay = $this->monitorConfig['restart_delay'] ?? self::DEFAULT_RESTART_DELAY;
        
        $this->loop->addTimer($restartDelay + 10, function() use ($appName, $appConfig, $restartSuccess) {
            $serviceName = $appConfig['service_name'];
            $botName = $appConfig['bot_name'];
            $telusers = $appConfig['telsock'];
            $description = $appConfig['description'];
            $datetime = $this->getCurrentDateTime();
            
            if (!$restartSuccess) {
                $message = "{$datetime['date']} - {$datetime['time']}: ❌ ERROR CRÍTICO - Falló el comando de reinicio para $description ($botName). Requiere intervención manual.";
                $this->logger->error($message);
                $this->sendTelegramAlert($message, $botName, $telusers); 
                return;
            }
            
            if ($this->isServiceActive($serviceName)) {
                $message = "{$datetime['date']} - {$datetime['time']}: ✅ ÉXITO - $description ($botName) se ha reiniciado correctamente.";
                $this->logger->info($message);
                $this->sendTelegramAlert($message, $botName, $telusers); 
            } else {
                $serviceStatus = $this->getServiceStatus($serviceName);
                $statusInfo = "Estado: {$serviceStatus['active']}, Habilitado: {$serviceStatus['enabled']}";
                if ($serviceStatus['failed'] === 'failed') {
                    $statusInfo .= ", Error: {$serviceStatus['failed']}";
                }
                
                $message = "{$datetime['date']} - {$datetime['time']}: ❌ ERROR CRÍTICO - $description ($botName) NO pudo reiniciarse. $statusInfo. Requiere intervención manual.";
                $this->logger->error($message);
                $this->sendTelegramAlert($message, $botName, $telusers); 
            }
        });
    }

    /**
     * Verificar todas las aplicaciones configuradas
     */
    public function checkAllApplications(): void
    {
        foreach (array_keys($this->applications) as $appName) {
            $this->checkApplicationStatus($appName);
        }
    }

    /**
     * Iniciar el monitoreo periódico
     */
    public function startMonitoring(): void
    {
        $interval = $this->monitorConfig['check_interval'] ?? self::DEFAULT_CHECK_INTERVAL;
        
        $this->logger->info("Iniciando monitoreo periódico cada $interval segundos");
        
        // Ejecutar primera verificación después del startup delay
        $this->loop->addTimer(
            $this->monitorConfig['startup_delay'] ?? self::DEFAULT_STARTUP_DELAY,
            function() {
                $this->checkAllApplications();
            }
        );
        
        // Configurar timer periódico
        $this->loop->addPeriodicTimer($interval, function() {
            $this->checkAllApplications();
        });
    }

    /**
     * Ejecutar una sola verificación (para uso con systemd oneshot)
     */
    public function runSingleCheck(string $appName = null): void
    {
        if ($appName) {
            $this->checkApplicationStatus($appName);
        } else {
            $this->checkAllApplications();
        }
    }

    /**
     * Ejecutar el daemon
     */
    public function run(): void
    {
        // Enviar notificación de inicio
        $this->sendStartupNotification();
        
        $this->startMonitoring();
        
        // Manejar señales de terminación
        if (function_exists('pcntl_signal')) {
            pcntl_signal(SIGTERM, [$this, 'shutdown']);
            pcntl_signal(SIGINT, [$this, 'shutdown']);
        }
        
        $this->logger->info("Daemon iniciado y ejecutándose");
        $this->loop->run();
    }

    /**
     * Shutdown graceful del daemon
     */
    public function shutdown(): void
    {
        $this->logger->info("Deteniendo daemon...");
        $this->sendShutdownNotification();
        $this->loop->stop();
    }

    /**
     * Método estático para uso directo desde línea de comandos
     */
    public static function runForApplication(string $appName, ?SendTelegram $telegramSender = null): void
    {
        $daemon = new self(null, null, $telegramSender);
        $daemon->runSingleCheck($appName);
    }

    /**
     * Enviar notificación de inicio del daemon
     */
    public function sendStartupNotification(): void
    {
        $datetime = $this->getCurrentDateTime();
        $message = "{$datetime['date']} - {$datetime['time']}: 🔄 Monitor de bots de Telegram iniciado correctamente";
        $this->logger->info($message);
    }

    /**
     * Enviar notificación de cierre del daemon
     */
    public function sendShutdownNotification(): void
    {
        $datetime = $this->getCurrentDateTime();
        $message = "{$datetime['date']} - {$datetime['time']}: ⏹️ Monitor de bots de Telegram detenido";
        $this->logger->info($message);
    }
}

// --- Uso desde línea de comandos ---
if (php_sapi_name() === 'cli') {
    $appName = $argv[1] ?? null;
    
    // Opcionalmente, puedes pasar una instancia personalizada de SendTelegram
    // $customTelegramSender = new SendTelegram();
    
    if ($appName && in_array($appName, ['bitradenes', 'bitrademacdrsi'])) {
        // Ejecutar verificación para una aplicación específica
        BotMonitorDaemonActual::runForApplication($appName);
    } elseif ($appName === 'daemon') {
        // Ejecutar como daemon continuo
        $daemon = new BotMonitorDaemonActual();
        $daemon->run();
    } else {
        // Ejecutar verificación de todas las aplicaciones
        $daemon = new BotMonitorDaemonActual();
        $daemon->runSingleCheck();
    }
}